unit EQFilters;

// This is the equalizer unit.
// This source code is based on theory presented at the Motorola technical
// documents apr2-d.pdf and apr7.pdf. In order to make the formulas
// of the coefficients applicable to all frequencies (20hz to 22000 hz) i had
// to make some additinal calculations that are not presented in the documents.
// I'm not sure about the way that i combine all the 10 filters. If i don't
// scale them with the ActiveFilters variable the output result gets too high
// and nothing is heard. If you figure out a better method of combining all the
// 10 filters i would be very happy if you notify me.
// The code is a little bit optimized in order to reduce the CPU Usage:
// I sum all the coefficients of the 10 filters and then i filter the input
// with the Total Equalizer filter.
// The filters that are used are Second Order IIR BandPass.
// Note this please:
// This software is provided "as-is," without any express or implied warranty.
// In no event shall the author be held liable for any damages arising from the
// use of this software.
// Copyright (c) 2002
// George Boudouris
// E-mail: georgebou@hotmail.com
// HomePage: http://www.hobbysoft.gr
// This is also the homepage of my audio player, Passion Audio Player
// Note: Motorola is a registered trademark of Motorola Inc.

interface

Uses
    Windows,Math;
Type
    StereoBufferPointer=^Cardinal;
    MonoBufferPointer=^Word;
    TwoChannelsBuffer=Record
             Left:Double;
             Right:Double;
    End;
    StereoSamplesBufferArrayDouble=Array[0..63] Of TwoChannelsBuffer;

Var
   InputBufferForEffect,  // Samples that come from the original source of samples
   OutputBufferForEffect:StereoSamplesBufferArrayDouble; // Samples that come from the Filtered source of samples
   // Then number of samples needed to buffer for filter is the number
   // of the filter of the coefficients for the input samples +1
   NumberOfSamplesNeededToBufferForFilter:Integer;

Const
     SampleRate=44100;

Type
    FilterTypes=(ftLowPassFilter,ftHiPassFilter,ftBandPassFilter,ftNotchFilter,ftPeakingEQFilter,
                 ftLowSelfFilter,ftHiSelfFilter);

Type
    EQBandsFrequenciesArray=Array[0..9] Of Double;

Const
     WinAmpEQBandsFrequencies:EQBandsFrequenciesArray=(60,170,310,600,1000,3000,6000,12000,14000,16000);
     EQBandsFrequencies:EQBandsFrequenciesArray=(31,62,125,250,500,1000,2000,4000,8000,16000);
Var
   CurrentEQBandsFrequencies:EQBandsFrequenciesArray;

Type
    FilterCoefficients=Record
         FilterFrequency:Double;   // Frequency To Filter
         QualityFactor:Double;     // Obvious
         FilterGain:Double;        // Obvious
         Alpha,Beta,Gama:Double; // For filter coefficients
         AmplifySignalValue:Double;
    End;
Var
   FilterCoeff:Array[0..9] OF FilterCoefficients; // 10 Bands Coefficients for 10-Band Equalizer
   TotalFilterCoeff:FilterCoefficients;
   CurrentEQFilter:FilterTypes;

Procedure SetBandPassFilter(InputFilterType:FilterTypes;Var InputFilterCoeffs:FilterCoefficients;Frequency,QualityFactor,FilterGain:Double;SetStereoSamplesBufferedForEffect:Integer);
Function BandPassFilter(Var InSn,InSn1,InSn2,OutSn1,OutSn2:TwoChannelsBuffer):TwoChannelsBuffer;
Function BandPassFilterOptimized(Var InSn,InSn1,InSn2,OutSn1,OutSn2:TwoChannelsBuffer):TwoChannelsBuffer;
Procedure CalcTotalCoefficientsForAllBands(TotalGain:Double);
Procedure InitializeCoefficientsForEQBands(FilterType:FilterTypes;EQBandFrequencies:EQBandsFrequenciesArray);
Procedure SetCoefficientsForEQBand(EQBandIndex:Integer;FilterType:FilterTypes;QualityFactor,FilterGain:Double);
Procedure ResetInputOutputBuffers;

implementation

function Clip(a: Double): Double;
begin
  if a <= -32768 then a := -32768
  else if a >= 32767 then a := 32767;
  Result := a;
end;

///////////////////////////////////////////
///////////// Graphic Equalizer ///////////
///////////////////////////////////////////
// The Filter Gain Of Each Band Should Be -0.2< Gain <1; The Center Value is 0 //
// For 10 Bands the result with Filter Gain=1 is 5 = (+14dB)             //
// For 10 Bands the result with Filter Gain=-0.2 is 0.2 = (-14dB)        //
// The volume scale factor should be < 1                                 //
// For Fi= IIR Filter for the i Band and gi= Fi Gain following the previous conditions
// yn=F0(x(n))*g0+F1(x(n))*g1+F2(x(n))*g2+F3(x(n))*g3+F4(x(n))*g4+F5(x(n))*g5+
//    F6(x(n))*g6+F7(x(n))*g7+F8(x(n))*g8+F9(x(n))*g9
// TotalOutput=yn * volume Gain
Procedure SetBandPassFilter(InputFilterType:FilterTypes;Var InputFilterCoeffs:FilterCoefficients;Frequency,QualityFactor,FilterGain:Double;SetStereoSamplesBufferedForEffect:Integer);
Var
   FreqC,     //Center Frequency
   Theta0,
   Q,
   Alpha,Beta,Gama:Double;
   Lamda,Kapa:Double; // Temporary variables to calculate the beta
Begin
     NumberOfSamplesNeededToBufferForFilter:=SetStereoSamplesBufferedForEffect;
     InputFilterCoeffs.FilterFrequency:=Frequency;
     InputFilterCoeffs.QualityFactor:=QualityFactor;
     // -14< Gain < 14
     If FilterGain>0 Then
        InputFilterCoeffs.FilterGain:=0.999/14*FilterGain
     Else
         InputFilterCoeffs.FilterGain:=0.2/14*FilterGain;

     Q:=QualityFactor; // This example selects Q=1.4
     FreqC:=Frequency;
     Theta0:=2*pi*FreqC/SampleRate;

     // The Following Functions are Simplified for unity gain at Center Frequency
     // and they are accurate for f<fs/8
     // Beta:=(Q-(Theta0/2))/(2*Q+Theta0);
     // Beta:=(1/2)*(1-tan(Theta0/(2*Q)))/(1+tan(Theta0/(2*Q)));
     // The complete calculations of beta for al frequencies are below
     Kapa:=(1+sqrt(1+4*Q*Q))/(2*Q);
     Lamda:=sin(Theta0/kapa)/(cos(Theta0/kapa)-cos(Theta0));
     beta:=(lamda-1)/(2*(Lamda+1));
     Gama:=(1/2+Beta)*Cos(Theta0);
     Alpha:=(1/2-Beta)/2;

     // The 2 multiplier is moved here from the filter equation
     InputFilterCoeffs.Alpha:=Alpha*2;
     InputFilterCoeffs.Beta:=Beta*2;
     InputFilterCoeffs.Gama:=Gama*2;
End;

Function BandPassFilter(Var InSn,InSn1,InSn2,OutSn1,OutSn2:TwoChannelsBuffer):TwoChannelsBuffer;
Var
   InSnMinusInSn2:TwoChannelsBuffer;
   lcDouble,rcDouble:Double;
   Counter:Integer;
   FilteredSample:TwoChannelsBuffer;
Begin
     InSnMinusInSn2.Left:=InSn.Left-InSn2.Left;
     InSnMinusInSn2.Right:=InSn.Right-InSn2.Right;

     lcDouble:=InSn.Left;
     rcDouble:=InSn.Right;
     for Counter:=0 To 9 Do Begin
         // The 2 multiplier of the equation is directly calculated to the coefficients
         // y[n]:=2*(A*(x[n]-x[n-2])+C*y[n-1]-B*y[n-2])
         lcDouble:=lcDouble+(FilterCoeff[Counter].Alpha*InSnMinusInSn2.Left+FilterCoeff[Counter].Gama*OutSn1.Left-FilterCoeff[Counter].Beta*OutSn2.Left)*FilterCoeff[Counter].FilterGain;
         rcDouble:=rcDouble+(FilterCoeff[Counter].Alpha*InSnMinusInSn2.Right+FilterCoeff[Counter].Gama*OutSn1.Right-FilterCoeff[Counter].Beta*OutSn2.Right)*FilterCoeff[Counter].FilterGain;
     End;

     // Amplify OutPutSignal
     lcDouble:=lcDouble*TotalFilterCoeff.AmplifySignalValue;
     rcDouble:=rcDouble*TotalFilterCoeff.AmplifySignalValue;

     lcDouble:=Clip(lcDouble);    // Limit Left  Sample at 16 bit
     rcDouble:=Clip(rcDouble);    // Limit Right Sample at 16 bit

     FilteredSample.Left:=lcDouble;
     FilteredSample.Right:=rcDouble;

     BandPassFilter:=FilteredSample;
End;

// InSn   =Input Sample n
// InSn1  =Input Sample n-1
// InSn2  =Input Sample n-2
// OutSn1 =Output Sample n-1
// OutSn2 =Output Sample n-2
Function BandPassFilterOptimized(Var InSn,InSn1,InSn2,OutSn1,OutSn2:TwoChannelsBuffer):TwoChannelsBuffer;
Var
   InSnMinusInSn2:TwoChannelsBuffer;
   lcDouble,rcDouble:Double;
   Counter:Integer;
   FilteredSample:TwoChannelsBuffer;
Begin
     InSnMinusInSn2.Left:=InSn.Left-InSn2.Left;
     InSnMinusInSn2.Right:=InSn.Right-InSn2.Right;

     lcDouble:=InSn.Left;
     rcDouble:=InSn.Right;
     // The 2 multiplier of the equation is directly calculated at the filters coefficients Calculations
     // In Order to Reduce the CPU Usage!!!
     // y[n]:=2*(A*(x[n]-x[n-2])+C*y[n-1]-B*y[n-2])
     lcDouble:=lcDouble+(TotalFilterCoeff.Alpha*InSnMinusInSn2.Left+TotalFilterCoeff.Gama*OutSn1.Left-TotalFilterCoeff.Beta*OutSn2.Left);
     rcDouble:=rcDouble+(TotalFilterCoeff.Alpha*InSnMinusInSn2.Right+TotalFilterCoeff.Gama*OutSn1.Right-TotalFilterCoeff.Beta*OutSn2.Right);

     // Amplify OutPutSignal
     lcDouble:=lcDouble*TotalFilterCoeff.AmplifySignalValue;
     rcDouble:=rcDouble*TotalFilterCoeff.AmplifySignalValue;

     lcDouble:=Clip(lcDouble);    // Limit Left  Sample at 16 bit
     rcDouble:=Clip(rcDouble);   // Limit Right Sample at 16 bit

     FilteredSample.Left:=lcDouble;
     FilteredSample.Right:=rcDouble;

     BandPassFilterOptimized:=FilteredSample;
End;

Procedure CalcTotalCoefficientsForAllBands(TotalGain:Double);
Var
   Counter:Integer;
   ActiveFilters:Integer;
Begin
     TotalFilterCoeff.Alpha:=0;
     TotalFilterCoeff.Beta:=0;
     TotalFilterCoeff.Gama:=0;
     ActiveFilters:=1; // The 1st filter is the passthrough filter
     For Counter:=0 To 9 Do Begin
         TotalFilterCoeff.Alpha:=TotalFilterCoeff.Alpha+FilterCoeff[Counter].Alpha*FilterCoeff[Counter].FilterGain;
         TotalFilterCoeff.Beta:=TotalFilterCoeff.Beta+FilterCoeff[Counter].Beta*FilterCoeff[Counter].FilterGain;
         TotalFilterCoeff.Gama:=TotalFilterCoeff.Gama+FilterCoeff[Counter].Gama*FilterCoeff[Counter].FilterGain;
         If FilterCoeff[Counter].FilterGain<>0 Then Inc(ActiveFilters);
     End;
     // My method for adjusting the 10 filters coefficients scaling is simple:
     // If the band filter gain is <>0 then this means that the filter is active, else
     // the filter is not active. So I count how much are the active filters and then
     // i divide the coefficients with the number of active filters.
     // All filters enabled give a factor of 11 (10 EQ Band filters+Passthrough filter=11) 
     TotalFilterCoeff.Alpha:=TotalFilterCoeff.Alpha/ActiveFilters;
     TotalFilterCoeff.Beta:=TotalFilterCoeff.Beta/ActiveFilters;
     TotalFilterCoeff.Gama:=TotalFilterCoeff.Gama/ActiveFilters;
     TotalFilterCoeff.AmplifySignalValue:=(TotalGain*0.9999);
End;

Procedure InitializeCoefficientsForEQBands(FilterType:FilterTypes;EQBandFrequencies:EQBandsFrequenciesArray);
Var
   Counter:Integer;

Begin
     CurrentEQBandsFrequencies:=EQBandFrequencies;
     CurrentEQFilter:=FilterType;
     For Counter:=0 To 9 Do Begin
         SetBandPassFilter(FilterType,FilterCoeff[Counter],CurrentEQBandsFrequencies[Counter],1.4,14,4);
     End;
End;

Procedure SetCoefficientsForEQBand(EQBandIndex:Integer;FilterType:FilterTypes;QualityFactor,FilterGain:Double);
Begin
     SetBandPassFilter(FilterType,FilterCoeff[EQBandIndex],CurrentEQBandsFrequencies[EQBandIndex],QualityFactor,FilterGain,4);
End;

Procedure ResetInputOutputBuffers;
Var
   Counter:Integer;
Begin
     For Counter:=0 To 63 Do Begin
         InputBufferForEffect[Counter].Left:=0;
         InputBufferForEffect[Counter].Right:=0;
         OutputBufferForEffect[Counter].Left:=0;
         OutputBufferForEffect[Counter].Right:=0;
     End;
End;

Begin
     ResetInputOutputBuffers;
     InitializeCoefficientsForEQBands(ftBandPassFilter,EQBandsFrequencies);
end.
