#!/usr/bin/perl
#use strict;
use IO::Socket;
use IO::Select;

$progname = $0;
$progname =~ s,.*[/\\],,;  # use basename only
$progname =~ s/\.\w*$//; # strip extension, if any

$VERSION = sprintf("%d.%02d", q$Revision: 1.32 $ =~ /(\d+)\.(\d+)/);

use Getopt::Long;

my @getopt_args = (
	'f=s',  # read in a configuration file
    'h',	# print usage
    'v',	# print version
    'l',    # the positional parameter is a list file, not an address
    'w',	# on completion, wait for a GUI user to close the window
);

Getopt::Long::config("noignorecase", "bundling");
unless (GetOptions(\%options, @getopt_args)) {
    usage();
	};

usage() if $options{'h'} || !(@ARGV or $options{'f'} or $options{'v'});

if ($options{'v'}) 	{
	my $DISTNAME = 'statProxy ' . $VERSION;
	die <<"EOT";
This is statProxy $VERSION ($DISTNAME)

Author: wayne\@nym.alias.net

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.
EOT
};

getConfigFileParams;	#from config file

#make this an xml file eventually - then i can distribute updates easily
#or part of the config files?
%tests=	(
		 0, {type=>'HEAD', host=>'www.panix.com', port=>'80', check=>'', 
		 	text=>'normal http'},
		 1, {type=>'CONNECT', host=>'panix.com', port=>22, check=>'', 
		 	text=>'ssh'}, 
		 2, {type=>'CONNECT', host=>'panix.com', port=>23, check=>'', 
		 	text=>'telnet'}, 
		 3, {type=>'CONNECT', host=>'smtp.panix.com', port=>25, 
		 	check=>'', text=>'smtp'}, 
		 4, {type=>'CONNECT', host=>'www.panix.com', port=>80, 
		 	check=>'', text=>'http (proxy or website) via port 80'}, 
		 5, {type=>'CONNECT', host=>'www.panix.com', port=>443, 
		 	check=>'', text=>'secure http'}, 
		 6, {type=>'CONNECT', host=>'news.panix.com', port=>119, 
		 	check=>'', text=>'news'}, 
		 7, {type=>'CONNECT', host=>'216.72.24.211', port=>1080, 
		 	check=>'', text=>'socks'}, 
		 8, {type=>'CONNECT', host=>'137.204.148.11', port=>3128, 
		 	check=>'', text=>'http proxy via port 3128'}, 
		 9, {type=>'CONNECT', host=>'irc.dalnet.com', port=>6667, 
		 	check=>'', text=>'irc'}, 
		 10, {type=>'CONNECT', host=>'139.13.25.160', port=>8080, 
		 	check=>'', text=>'http proxy via port 8080'}, 
		); #check is a regex for response contents (one day)

$proxyStr=(shift or $proxyStr); #command line positional param wins.

if ($options{'l'})	{ #need to get our proxyStr from a file in $ARGV[0]
	open(LIST, "<$proxyStr") or die "can't open file $proxyStr: $!";
	chomp(@proxyStr=<LIST>) or die "can't read file $proxyStr: $!";
	}
else {@proxyStr=($proxyStr)};

unless ($pingInstalled = eval('require Net::Ping'))	{
	print "Not using ping to determine if hosts are alive because the \n";
	print "Net::Ping module is not installed";
	print "If you use ActiveState Perl, try \n";
	print "ppm install Net-Ping\n";
	print "in a command window\n"};

########################################################################

$safeMode=(defined($safeMode) ?$safeMode :1); # 0 guarantees to test the whole list
							# 1 skips dialups, dsl etc., .mil, .gov, etc.
							#
%tested=();
for $i (0..$#proxyStr) {	#randomize the list, efficiency not important
	$swapIndex=int(rand($i+1));
	$temp=$proxyStr[$swapIndex];
	$proxyStr[$swapIndex]=$proxyStr[$i];
	$proxyStr[$i]=$temp;
	$tested{$temp}=0;
	};

print STDOUT "\nstatProxy v$VERSION report:\n";
for $proxyStr (@proxyStr) {		# the big loop
	next unless ($proxyStr); 	#maybe blank lines in file
	next if ($tested{$proxyStr});
	if ($safeMode) {
		#use some of Craig's purifying rituals:
		next if ($proxyStr=~/dial|modem|connect|ppp/i);
		next if ($proxyStr=~/[^a-z]*(ip|adsl|cisco|user|usr|ascend|max)[^a-z]+/i);
		next if ($proxyStr=~/cert\.|sprint\.|firewal/i);
		next if ($proxyStr=~/\.(mil|navy|fed.us|gov|govt|gouv|defen|state\.)/i);
		next if ($proxyStr=~/(chem|phys)[-_\.](lab|bess)[^a-z]/i);
		next if ($proxyStr=~/bess/i);
		next if ($proxyStr=~/telia.co.|idirect.com|rr.com|home.com/i);
		# Discard all hostnames containing a digit except those you wish to keep and except for many "k12" hostnames.
		next if ($proxyStr=~/\.(ae|sa|sg)/i);
		
		## and some extra ones
		next if ($proxyStr=~/[^a-z]+pub[^a-z]*/i);
		next if ($proxyStr=~/[^a-z]dialin]\./);
		next if ($proxyStr=~/\.swbell\.net/); #got a OS fingerprint from them
		};
	#print STDOUT "ps: $proxyStr\n";
	($proxyAddr, $proxyPort)=split(/:/, $proxyStr);
	$proxyPort=($proxyPort or '?');
		next if ($safeMode and $tested{$proxyAddr}); #even if we've only tested one port 
								#there before, it's best not to return
	$printHTML=0;	# for me (to get the string for my proxy capabilities pages)
	print STDOUT "$proxyAddr:$proxyPort| ";
	if ($proxyPort=~/\?/) { #we want a port scan to be done
		#default to a (rather useless) list of ports to scan if 
		#they haven't spec'd it in the config file
		if ($safeMode) { #in safe mode, default to one port per host
						#hopefully, any config file value is safe too
#			$portsToScan=($portsToScan or '8000')}
			$portsToScan='8000'}
		else {
			$portsToScan=($portsToScan or '8000#9000#8010#9001#880')};
		print STDOUT "scanning .. ";
		#print STDOUT "$portsToScan\n";
		};
#	unless ($options{'l'}) {
#		print STDOUT "let me check how this sucker works\n";
#		print STDOUT "please wait one or two minutes...\n";
#		};

	$protocol = getprotobyname("tcp");
	$proxyAgent=''; $via='';
	foreach $test (sort {$a <=> $b} keys(%tests)) {	#ref to test object
		$pass=0;
		#print STDOUT "$test ";
		if ($pingInstalled and !$> and ($test==0)) { #needs root
			$p = Net::Ping->new("icmp"); # we lose the hosts which are not 
									# icmp pingable - must change this 
									# to tcp pings, with select to timeout
									# the blocked ports and dead hosts quickly.
									# icmp ping also requires root access
									# 'coz it makes it's own packets (!), 
									# so it won't work on shell accounts.
			if ($p->ping($proxyAddr))	{
				&makeConnection() ;	#connect to the proxy only 
				}					#if it's icmp reachable (for now)
			else {
				print STDOUT "error 1: ignoring\n";
#				"\nhost $proxyAddr is unresolvable, dead or not icmp reachable";
				$connected=0;
				}
			$p->close()}
		else {&makeConnection()} ;	#connect to the proxy
		last unless ($connected);
		($type, $host, $port)=($tests{$test}->{type}, $tests{$test}->{host}, 
			$tests{$test}->{port}); #there's a more elegant way ... map?

		# Send command to proxy:
		if ($test == 0) { #format this HEAD test properly
			print PROXY "$type http://$host:$port/ HTTP/1.0\r\n"}
		else { #format CONNECT properly
			print PROXY "$type $host:$port HTTP/1.0\r\n"};	#change \r\n to 
															#$CRLF for Macs
		print PROXY "Host: $host:$port\r\n";#some HTTP/1.1 proxies 
											#insist on this now
		# send a non-agressive User-Agent header here?
		print PROXY 
			'User-Agent: Mozilla/4.0 (compatible; MSIE 5.5; ';
		print PROXY 'Windows NT 5.0)', "\r\n";
#		if ($user and $pass) {print PROXY 
#				"Proxy-Authorization: Basic ".
#				MIME::Base64::encode("$user:$pass", ""). 
#				"\r\n"};
		print PROXY "\r\n";

		($status) = (split(/\s+/,<PROXY>))[1];
	#	print STDOUT "$test, $status\n";
		if ($status)	{
			$pass=1 if ( int($status/100) == 2 ); #good, got 2nn status code
			#add stuff to say access controlled if code =500?
			# Skip through remaining part of HTTP header (until blank line)
			# catch 'Proxy-Agent: ' on the way
			while (<PROXY>)	{
				last if ( /^[\r\n]+$/ );
				unless ($proxyAgent) {($proxyAgent)=
					($_ =~ /Proxy-Agent: (.*?)[\n\r]*$/i)}; # only get this on a 
													#successful connect
				unless ($via) {($via) = ($_ =~ /Via: (.*?)[\n\r]*$/i)};
				};
			}
		else {$pass=0}; #should I ignore the rest of the tests if the proxy  
						#fails the HEAD test (test 0)? Dunno, guess so.
		$tests{$test}->{result}=$pass;
		close PROXY;
		};
	next unless ($connected); # failed to even connect, we've printed an 
					# error to that effect, so just go on to the next proxy
	#print STDOUT "\n";
	unless ($proxyAgent) {$proxyAgent='unknown'}; 
	unless ($via) {$via='unknown'};
	print STDOUT "$proxyAddr:$proxyPort is ", ($tests{'0'}->{result} ?"\7" :'not '), "a working http proxy\n";
	$str=''; $goodStr=''; $badStr=''; $lastStat=-1; @tests=();
	foreach $test (sort {$a <=> $b} keys(%tests)) {	#ref to test object
		next unless $test; #test 0 is special
		($type, $host, $port, $result)=($tests{$test}->{type}, 
			$tests{$test}->{host}, $tests{$test}->{port}, 
			$tests{$test}->{result}); #there's a more elegant way ... map?
		if ($result) {
			$color="00FF00";
			$goodStr.="$port, ";
			push(@tests, $test);
			}
		else {
			$color="FF0000";
			$badStr.="$port, ";};
		# do something different here, to get a simple list of good 
		# and bad ports when $printHTML=0

		$str.=(($result == $lastStat) 
			?"" :"<font color=\"\#$color\">").$port.", ";
		$lastStat=$result;
		};
	for ($str, $goodStr, $badStr) {$_=~s/, $//}; 
	#print STDOUT "CONNECT <host>:<port> tests (by port):\n";
	print STDOUT "This proxy allows the following outside services to be used:\n";
	print STDOUT "passed: $goodStr\n";
	print STDOUT "failed: $badStr\n";
	if ($printHTML) {print STDOUT "$str\n"};
	print STDOUT "proxy type: agent: $proxyAgent, via: $via\n";

	if (!@tests) {print STDOUT "Can't use $proxyAddr:$proxyPort for anything special\n"; next};
	
	unless ($options{'l'}) {
		print "\n";
		print STDOUT "You could run a local proxy/server (with HTTPort, or ";
		print STDOUT "localProxy.pl using CONNECT) for: \n"; 
		for (@tests) {
			#print STDOUT "$_\n";
			next unless ($tests{$_}->{type} eq 'CONNECT'); #skip non-CONNECT tests 
														#for this local proxy list
			print STDOUT $tests{$_}->{text}, " (eg ", 
			$tests{$_}->{host}, ":", $tests{$_}->{port}, ")\n"
			};
		print STDOUT "and maybe on other ports I haven't tested\n";
		};
	};
1;

sub makeConnection{
	# connect to proxy server ... try a port list if no port specified
	# In this case, we find the first listening port
	#print STDOUT "\n" if ($test==0);
	$connected=0;
	for $pPort ($proxyPort=~/\?/ ?split(/#/, $portsToScan) :$proxyPort) {
		print STDOUT "$pPort, " if ($test==0);
		socket (PROXY, PF_INET, SOCK_STREAM, $protocol) or
			die("Failed to create socket:$!");
		#if there's no returned reset (e.g. blocked ports), this is slow
		#try, wait 3 secs, try, wait 6 secs, try, wait 12 secs, timeout.
		#total of 21 secs for each failed connection attempt
		if (connect (PROXY, sockaddr_in($pPort,inet_aton($proxyAddr)))) {
			$connected=1;
			$proxyPort=$pPort;
			last;		#should fix this to ensure an address gets all it's 
						#ports tested when we have some speed. 
			}
		else {
			#print STDOUT "failed\n";
			$connected=0;
			next};
		};
	unless ($connected) {
		print STDOUT "no connection\n";
		return;
		};
	print STDOUT "connected .. testing\n" if ($test==0);
	select(\*PROXY); $|=1;		#make it hot
	};

sub setConfigFileParams {
	if ($options{'f'}) { #user specified a configuration file
		open(CONFIG, "<$options{'f'}") or die "can't open the config file";
		my @config=<CONFIG>;
		close(CONFIG);
		foreach (@config) {
			chomp;
			next if (/^#/);
			my ($var,$val) = split(/=>/);
			next if (eval("defined(".$var.")")); #make sure the command line options 
								#override any config file settings
			eval("$var=".$val); #set the variables to config file values
			};
		};
	};
	
sub usage	{	# command line options
    die <<"EOT";
Usage: $progname [-f file] ((-l <list file>)|<proxy addr>[:<proxy port>])

    -f <file>       Use a configuration file (command line options still 
                    override)
    -v              Show program version
    -h              Print this message
    -l              Read list of proxies from file given on command line 
                    Note: this is l ('ell'), not 1 (one). 
                    
Examples:
statProxy 194.170.1.66:8080 (will check that proxy)
statProxy 194.170.1.66      (will port scan that proxy and then check it) 
                             The list of ports to scan comes from the 
                             configuration file, or is defaulted to some 
                             values you probably don't want in the code)
statProxy -l file.txt       (will read  a list of proxy:port or just addresses 
                             from file.txt and check them)
EOT
}

__END__

=head1 NAME

statProxy

=head1 SYNOPSIS

statProxy [-f file] [-h] [-v] ((-l <list file>)|<proxy addr>[:<proxy port>])

=head1 OPTIONS

=over 4

=item -f <file>       Use a configuration file to set some of the options 
                      (and some of the internal variables)
                      Note that all can be set, and they will not override 
                      any command line options specified. 

=item -h help         Prints out a brief help message.

=item -v version      Prints program version number

=item -l list file    The file specified is read and each line contains 
                      a proxy to check (in the format addr:port, or 
                      addr:?, or just addr by itself on the line)

=back

=head1

=head1 DESCRIPTION

This program will run anywhere and check web proxies for you. 
If the proxy port is specified as '?', it will do a port scan 
(from a list specified in the config file, or defaulted to 
in the code) to see which port the proxy is listening on.

It performs various specified tests on the proxy:port given:

=over 4

=item * basic http proxy function (GET)

=item * CONNECT tests to various ports

=back

It then lists the results and suggests ways to use them.

=head1 REQUIREMENTS

=over 4

=item * a system with perl (http://www.activestate.com/ for win32 systems, 
http://www.perl.org/ for linux)

=item * Perl

=back

=head1 VERSION RELEASE HISTORY

=over 4

=item * 00/??/?? v1.0 released

=item * 00/12/29 v1.20 : Now does a 'port scan' of specified ports 
if the port parameter was given as '?'. Many cleanups. Added basic 
test that proxy is working as a http proxy. Config file option and 
other options added. Some documentation added. Changed this file 
format to Unix, since both Dos and Unix seem to work in Windows, 
but Unix #! line doesn't work if the file format is Dos because 
of the extra CR. This must be the case with the whole *Proxy 
series :-(

=item * 00/12/29 v1.30 beta release: added ability to read file of proxies

=item * 00/12/31 v1.31 : fixed safeMode not configurable bug.

=item * 00/1/3 v1.32: rearranged command line and config file options

=back

=head1 TO DO

=over 4

=item * make it save the current state when interrupted. At the 
moment you lose the state (and since the list was randomized,  
it's difficult  to start again). Need a signal 
handler, data freeze and thaw, option to start with a saved file etc.

=item * use parallel socket connections to speed things up (a lot)

=item * don't require root access for the ping (tcp or udp are ok), 
so the thing will run on shells.

=back

=head1 AUTHOR

wayne@nym.alias.net (http://www.angelfire.com/wy/1waynes/)

=cut

