#!/usr/bin/perl -w
# localProxy - server that multiplexes without forking and allocates 
# connections via CONNECT proxies.
use strict;

use POSIX;
use IO::Socket;
use IO::Select;
use Socket;
use Fcntl;
use Tie::RefHash;
use Getopt::Long;

my ($progname, $VERSION, $lPort, $proxyStr, $remoteStr, $proxyAddr, 
$proxyPort, $host, $port, %options, $sparePSocksNeeded);

$SIG{BREAK} = \&signal_handler;	#toggle debug mode

$progname = $0;
$progname =~ s,.*[/\\],,;  # use basename only
$progname =~ s/\.\w*$//; # strip extension, if any

$VERSION = sprintf("%d.%02d", q$Revision: 1.20 $ =~ /(\d+)\.(\d+)/);

my @getopt_args = (
	'f=s',  # read in a configuration file
    'h',	# print usage
    'v',	# print version
	);

Getopt::Long::config("noignorecase", "bundling");
unless (GetOptions(\%options, @getopt_args)) {
    usage();
	};

if ($options{'v'}) {
#    my $DISTNAME = 'findProxy ' . $VERSION;
    my $DISTNAME = 'findProxy.zip';
    die <<"EOT";
This is localProxy $VERSION ($DISTNAME)

Author: wayne\@nym.alias.net

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.
EOT
}

usage() if ($options{'h'} or !(@ARGV or $options{'f'}));

unless ($options{'f'}) {
	if ($#ARGV == 2) {($lPort, $proxyStr, $remoteStr) = @ARGV}
	elsif ($#ARGV == 1) {($proxyStr, $remoteStr) = @ARGV};
	
	};
	
#set parameters. command line settings always override, config 
#file params next, and coded defaults only if nothing else defines it
setConfigFileParams();	#options from config file
setCodeDefaultParams();	#resolve the remaining params by coded defaults

usage() unless ($proxyStr and $remoteStr);

($proxyAddr, $proxyPort) = split(/:/, $proxyStr);
($host, $port) = split(/:/, $remoteStr);

$port=($port or 80); #default to port 80 if not specified
unless ($lPort) {$lPort=10000 + $port}; #default the daemon port

$sparePSocksNeeded = ($sparePSocksNeeded or 2);
#my ($proxyAddr, $proxyPort) = ('130.131.134.26', '80'); #a proxy which allows CONNECT through
my $type = 'CONNECT';
#my ($host, $port) = ('12.16.32.11', '80'); #a proxy which may not allow CONNECT through

# Listen to port.
my $server = IO::Socket::INET->new(LocalPort => $lPort, #don't need SO_REUSEADDR?
                                Listen    => 10 )
  or die "Can't make server socket: $@\n";

# begin with empty buffers
my (%inbuffer, %outbuffer, %ready) = ((), (), ());
my (%assoc, %pSocks, @socketsToShutdown) = ((), (), ());
my $debug=0;

tie %ready, 'Tie::RefHash';

nonblock($server); #this doesn't work in windows, yet.
my $select = IO::Select->new($server);

# Main loop: check reads/accepts, check writes, check ready to process
while (1) {
    my $client;
    my $rv;
    my $data;

    # check for new information on the connections we have

    # anything to read or accept?
    foreach $client ($select->can_read(1)) {

        if ($client == $server) {
            # accept a new connection

            $client = $server->accept();
            if ($debug) {print "accepted: $client\n"};
            $select->add($client);
            nonblock($client);
            allocatePSock($client); #associate this 
            						#client handle with 
            						#a proxy socket (and make a new 
            						#proxy socket if there isn't one 
            						#available)
            }
        else {
					# read data
					$data = '';
					$rv   = $client->recv($data, POSIX::BUFSIZ, 0);

					unless (defined($rv) && length $data) {
							# This would be the end of file, so close the socket gracefully
							if ($debug) {print "queue for close - client: $client\n"};
							queueShutdownSocket($client);
							next;
						}

					$inbuffer{$client} .= $data;

					# test whether the data in the buffer or the data we
					# just read means there is a complete request waiting
					# to be fulfilled.  If there is, set $ready{$client}
					# to the requests waiting to be fulfilled.
					#if ($pSocks{$client}) { #these guys don't terminate images (for example)
																	#no FIN, no indication of end except internally 
																	#specified content length header ???
						#push( @{$ready{$client}}, $inbuffer{$client} );
						#$inbuffer{$client} = '';
						#}
					#else {
						#while ($inbuffer{$client} =~ s/(.*\n)//) { #post image data?
								#push( @{$ready{$client}}, $1 );
							#}
						#};
					#try to send everything
					push( @{$ready{$client}}, $inbuffer{$client} );
					$inbuffer{$client} = '';
					}
	    }

    # Any complete requests to process?
    foreach $client (keys %ready) {
        handle($client);
	    }

    # Buffers to flush?
    foreach $client ($select->can_write(1)) {
        # Skip this client if we have nothing to say
        next unless exists $outbuffer{$client};

        $rv = $client->send($outbuffer{$client}, 0);
        unless (defined $rv) {
            # Whine, but move on.
            warn "I was told I could write, but I can't.\n";
            next;
	        }
        if ($rv == length $outbuffer{$client} ||
            $! == POSIX::EWOULDBLOCK) {
            substr($outbuffer{$client}, 0, $rv) = '';
            delete $outbuffer{$client} unless length $outbuffer{$client};
            if (!defined($assoc{$client}) and !$outbuffer{$client}) {
            	queueShutdownSocket($client)};
			}
        else {
            # Couldn't write all the data, and it wasn't because
            # it would have blocked.  Shutdown and move on.
            if ($debug) {print "removing client: $client\n"};
            queueShutdownSocket($client);
            next;
	        }
	    }

    # Out of band data?
    foreach $client ($select->has_exception(0)) {  # arg is timeout
        # Deal with out-of-band data here, if you want to.
        if ($debug) {print "oob data\n"};
	    }
	    
	# enough spare proxy sockets ready?
	my $spare=0;
	for my $sock (keys(%assoc)) { #note: this is a scalar only
		if ($assoc{$sock})	{
			#unless ((scalar($assoc{$sock}) =~ /^-\d$/) #or 
			#(scalar($assoc{$assoc{$sock}}) eq $sock)) 
				#{
				#if ($debug) { #this check is no good with the quick switch from 
				#one proxy to another for each full request by an http client 
				#implemented below (2 proxy sockets may both be sending to one 
				#client), temporarily
					#print "mismatch: $sock, $assoc{$sock}, $assoc{$assoc{$sock}}\n"
					#};
				#}
			}
		else {$spare++ if ($pSocks{$sock})};
		};
	if ($spare < $sparePSocksNeeded) {
		if ($debug) {print "creating new pSock: $spare/$sparePSocksNeeded\n"};
		allocatePSock(0); #make a new one
		}; #just one each loop
	
	shutdownSockets(); #shutdown any queued sockets
	
	sub queueShutdownSocket {
		my ($client) = @_;
		push(@socketsToShutdown, $client);
		};
		
	sub shutdownSockets {
			while ($client = shift(@socketsToShutdown)) {
				if ($debug and ($inbuffer{$client} or $outbuffer{$client} or 
					$ready{$client})) {
						print "shutdown of socket with data pending: $client\n"};
					
				delete $inbuffer{$client};
				delete $outbuffer{$client};
				delete $ready{$client};
				if (defined($assoc{$client})) { #the other end
					if ($pSocks{$client}) { #the socket closing is a proxy socket
									#so we need to point any associated client 
									#to a new proxy socket. 
									#remake his requests for him???
									#damn - just shut him down too, for now.
						unless (scalar($assoc{$client}) =~ /^(0|-1|-2)$/) {
							#$assoc{$assoc{$client}} = allocatePSock($assoc{$client});
							queueShutdownSocket($assoc{$client});
							};
						}
					else {
						delete $assoc{$assoc{$client}}; #the socket 
										#closing is not a proxy 
										#socket so we remove the other end's link 
										#back here and he will shut down when his 
										#outbuffer is flushed.
						};
					delete $assoc{$client}; #the link from here to the other end
					};
				delete $pSocks{$client}; # if it exists
				$select->remove($client);
				close($client);
				};
			};
	}

# handle($socket) deals with all pending requests for $client 
sub handle {
    # requests are in $ready{$client}
    # send output to $outbuffer{$anotherClient}
    my $client = shift;
    my $request;
		if (defined($assoc{$client})) { #could be no assoc defined, if  
														#the other end has shutdown prematurely (maybe 
														#user changed pages, clicked 'stop' 
														#in the browser etc.). In this case, 
														#we ignore this incoming data. The  
														#socket will shut down by itself.

			$request = join('', @{$ready{$client}});
			#if ($debug) {print "request from $client: $request\n"};
			# $request is the text of the request from $client
			# put text of reply into $outbuffer{$assoc{$client}}
			#handle CONNECT response here
			unless (ref($assoc{$client})) {
				#print "ref: ", ref($assoc{$client}), "\n";
				#print "test2: ", $assoc{$client}, "\n";
				if (scalar($assoc{$client}) eq '-2') {
					#my ($status) = (split(/\s+/,$request))[1];
					my $status='';
					if ($request =~ s|HTTP/1\.\d\s(\d+)\sConnection Established[\r]?\n||i) {
						$status=$1;
						#if ($debug) {print "status: $status\n"};
						};
					warn "failed to CONNECT to $host:$port" unless ($status);
					warn "bad response code CONNECTING to $host:$port" unless 
						( int($status/100) == 2 );
					$assoc{$client}='-1'; #mark it as 'connected'
										#(assoc with me)
					#print "pSock marked as connected\n";
					};
				if (scalar($assoc{$client}) eq '-1') { #connected, but not yet ready
					$assoc{$client}='0' if ($request =~ s|\n[\r]?\n||); #empty line 
															#received, ignore anything 
															#else. pSock is now ready
					#if ($debug) {print "pSock ready: $client\n" if ($assoc{$client}==0)};
					};
				#if (scalar($assoc{$client}) eq '0') { #CONNECTED already, ignore everything
					#if ($debug) {print "pSock data ignored: $request\n"};
					#}
				#else {
					#if ($debug) {
						#print "shit\! - what am I doing here\? ",$client, $assoc{$client}, 
							#"\= $request\n";}}
				};
			$outbuffer{$assoc{$client}} .= $request; #send it to the right handle
			#if ($debug) {print "request sent from $client to: $assoc{$client}\n"};
			#if the socket is a web browser, need to change the proxy socket it's 
			#using after each request, because otherwise IE5.5 tries to use 
			#the same socket again (assumes it's keep alive header will be 
			#honored, i think, but it isn't!).
			if (!$pSocks{$client} and ($request =~ /[\r]?[\n][\r]?[\n]$/)) {
				#$assoc{$client} = allocatePSock($client);
				#delete $assoc{$client}; # should make it shutdown after next output
				};
			} 
		else {#if ($debug) {print "ignoring request(s) from $client (no assoc defined)\n"}
			};
    delete $ready{$client};
	}

# nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;

#use IO::Handle;    
#print $socket->blocking(), "\n";
#$socket->blocking(0) or warn "can't use nonblocking mode: $!"; # needs IO::Handle

#$flags = $socket->sockopt(F_GETFL); #sockopt sets options in the 'SOL_SOCKET level'
#$socket->sockopt(F_SETFL, $flags|O_NONBLOCK);
#$flags = $socket->sockopt(3); #sockopt sets options in the 'SOL_SOCKET level'
#$socket->sockopt(4, $flags|2048);

#    $flags = fcntl($socket, F_GETFL, 0)	#fcntl not implemented in windows systems
#    $flags = fcntl($socket, 3, 0)	#it's 3 in linux, not def in win32
#            or die "Can't get flags for socket: $!\n";
#    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
#    fcntl($socket, 4, $flags | O_NONBLOCK)	#it's 4 in linux, not def in win32. O_NONBLOCK is 2048 in linux.
#            or die "Can't make socket nonblocking: $!\n";

	}

sub connectToProxy {
	my $pSock;
	while (!$pSock) {
		$pSock = IO::Socket::INET->new(PeerAddr => $proxyAddr,
	                                 PeerPort => $proxyPort,
	                                 Proto    => 'tcp', 
									 Timeout => 10, 
									 ) or warn "socket:$@";	#is there any point in 
									 												#continuing if the time 
									 												#to connect is more than 
									 												#10 seconds?
		};
	$pSocks{$pSock}=$pSock; # need to keep scalar -> handle capability
	$assoc{$pSock} = '-2';	# mark it as 'connecting'
													# (assoc with me)
	$select->add($pSock); #do this here so the prog can handle 
						  #anything coming in
#	print "pSock connecting: $pSock\n";
	return $pSock;
	};
	
sub connectThruProxy {
	my ($pSock) = @_;
	$outbuffer{$pSock} = 
	"$type $host:$port HTTP/1.1\r\nHost: $host\r\nContent-Length: 0\r\n\r\n";
								#change \r\n to $CRLF for Macs. HTTP/1.1 ok.
#	print "pSock CONNECTING: $pSock\n";
	};

sub allocatePSock { #allocate (or make) a proxy socket
	my ($client) = @_; # $client = 0 if we are just making a new spare socket
	my $pSock;
	if ($client) {
		for $pSock (keys(%assoc)) { #find a *scalar* key for the next free proxy socket
			next if ($assoc{$pSock});
			$assoc{$pSock}=$client; #ok to put a real handle as a value
			$assoc{$client}=$pSocks{$pSock};	# but need the real handle here, 
																				#not just the scalar repr.
			if ($debug) {print "allocated $pSock for use by $client\n"};
			return $pSocks{$pSock};
			};
		#no spares, lets try to make sure it doesn't happen again
		$sparePSocksNeeded++;
		};
	#there are no spares (the client will need to try again), 
	#or we are making a new spare for the pool
	#need to make a new pSock
	$pSock = connectToProxy(); #ok to wait for this here? probably not.
	connectThruProxy($pSock);
	};
	
sub signal_handler {
	my $signame = shift;
  $debug = 1 - $debug;	#signal to toggle debug mode
  $SIG{$signame} = \&signal_handler;
	};

sub usage	{	# command line options
    die <<"EOT";
Usage: $progname [-options] [<localPort>] <proxy addr>:<proxy port> <remote addr>:<remote port>
    -f <file>       Use a configuration file (command line options still 
                    override)
    -v              Show program version
    -h              Print this message
EOT
}

sub setConfigFileParams {
	if ($options{'f'}) { #user specified a configuration file
		open(CONFIG, "<$options{'f'}") or die "can't open the config file";
		my @config=<CONFIG>;
		close(CONFIG);
		foreach (@config) {
			chomp;
			next if (/^#/);
			my ($var,$val) = split(/=>/);
			next if (eval("defined(".$var.")")); #make sure the command line options 
								#override any config file settings
			eval("$var=".$val); #set the variables to config file values
			};
		};
	};
	
sub setCodeDefaultParams {};

__END__

=head1 NAME

localProxy

=head1 SYNOPSIS

localProxy [-f <config file>] [-h] [-v] [<localPort>] <proxy addr>:<proxy port> <remote addr>:<remote port>

=head1 OPTIONS

=over 4

=item -f <file>       Use a configuration file to set some of the options 
                      (and some of the internal variables)
                      Note that all can be set, and they will not override 
                      any command line options specified. 

=item -h help         Prints out a brief help message.

=item -v version      Prints program version number

=back

=head1

=head1 DESCRIPTION

This program will run anywhere (where the Perl is installed). It 
performs similar functions to HTTPort. It allows you to specify 
(probably local) CONNECT-capable proxies, remote 
uncensored normal proxies or services, and will forward connections 
to those remote services through the CONNECT proxies. This allows 
you to access services which are otherwise blocked, such as web 
access, newsgroup access, ssh, telnet, irc etc. All parameters are 
configurable and can be read in from a configuration file.
A nice touch is that you can use ctrl-break to toggle debug mode on 
and off. This was invaluable in testing, so I left it there. 

The program is one of a series of tools (localProxy, 
statProxy, sortProxy, findProxy) which will soon be supplemented by an 
overall diagnostic tool to analyse your environment and write 
configuration files for these to use to enable your access automatically.

=head1 REQUIREMENTS:

=over 4

=item * a system with Perl (http://www.activestate.com/ for win32 systems, 
http://www.perl.org/ for linux)

=back

=head1

=head1 VERSION RELEASE HISTORY

=over 4

=item * 01/1/4 v1.20 the old localProxy was completely rewritten to 
enable use of parallel sockets and proxies. Web browsing in the old 
version sucked. This one has all the capabilities required.

=back

=head1

=head1 TO RUN IT

=over 4

=item 2. start it from the command shell by typing:

 perl localProxy.pl [options] [<localPort>] <proxy addr>:<proxy port> <remote addr>:<remote port>
 
=item 3. print this documentation in a pretty format by typing:

 perldoc localProxy.pl
 
 and maybe convert it to html, but this doesn't work yet! :-)
 
 perldoc localProxy.pl | pod2html > findProxy.html

=back

=head1 TO DO

=over 4

=item * works well without it, but i'd like to fix the F_GETFL thing. 
Is non-blocking socket i/o not available in windows perl at all?

=item * more tuning (reduce the number of proxy sockets in the pool 
when we see one closing in a free state, down to some minimum (say 2)).

=item * Some ftp notes:

=over 4

=item * must use passive mode to have any chance of success. 

=item * client command is 'passive', but MS command line ftp client doesn't 
have this (!). 

=item * 'ls' in passive mode results in the server sending back addr:port it's 
listening on and the client attempts a direct connection to that addr (!), 
rather than the host it's connected to (probably localhost).

=back

=head1

=head1 KNOWN PROBLEMS

=over 4

=item * none yet.

=back

=head1

=head1 VARIABLES INSIDE THE CODE

=item * none, so far

=head1 AUTHOR

wayne@nym.alias.net (http://www.angelfire.com/wy/1waynes/)

=cut



