#include <stdio.h>
#include <string.h>
#include <stdlib.h>

typedef struct _THuffmanTree
{
    short isLeaf;
    unsigned long total;
    unsigned short character;/* to keep compatibility, because "char" may be negative */
    struct _THuffmanTree *left;
    struct _THuffmanTree *right;
    struct _THuffmanTree *parent;
}THuffmanTree;

int IsBiggerHT(THuffmanTree **tree1, THuffmanTree **tree2);

FILE *fp0, *fp1;
unsigned long total[256];
THuffmanTree *tree[256], *backtree[256];
short treeFloater, backTreeFloater;
char *huffmanTable;
short tableLength;

short main()
{

    if((fp0 = fopen("in.dat", "rb")) == NULL)
    {
        printf("Error opening file!\n");
        return 1;
    }
    memset(total, 0, sizeof(total));
    {
     unsigned short ch;/* too keep compatibility, because "char" may be negative */
     while((ch = getc(fp0)) != 0xFFFF)
      {
         total[ch]++;
     }
    }

    printf("File reading finished!\n");

    /* initialize */
    {
     short i, j;
     j = 0;
     for(i = 0; i < 256; i++)
     {
         if(total[i] != 0)
         {
             tree[j] = (THuffmanTree *)malloc(sizeof(THuffmanTree));
             if(tree[j] == NULL)
             {
                 printf("Not enough memory to locate!\n");
                 return 2;
             }
             tree[j]->isLeaf = 1;
             tree[j]->character = i;
             tree[j]->total = total[i];
             tree[j]->parent = NULL;
             j++;
         }
     }
     treeFloater = j;
    }

    if(treeFloater == 0)
        return 3; /* file length should not be zero */
    /* backup the tree */
    memcpy(backtree, tree, treeFloater * sizeof(THuffmanTree *));
    backTreeFloater = treeFloater;
    /* test output */
    /*{
     short i;
     printf("treeFloater = %d\n", treeFloater);
     for(i = 0; i < treeFloater; i++)
     {
         printf("tree[%d]->isLeaf = %d; tree[%d]->total = %d; tree[%d]->character = %d\n",
                i, tree[i]->isLeaf, i, tree[i]->total, i, tree[i]->character);
     }
    }*/

    /* treatment */
    while(treeFloater != 1)
    {
        /* things to synchronize:
         * 1. tree;
         * 2. treeFloater;
         */
        {
         short i;
         THuffmanTree *newtree;
         qsort(tree, treeFloater, sizeof(THuffmanTree *), IsBiggerHT);
         /* assume: treeFloater >= 2 */
         newtree = (THuffmanTree *)malloc(sizeof(THuffmanTree));
         if(newtree == NULL)
         {
             printf("Not enough memory to locate!\n");
             return 2;
         }
    /* test */
    /*printf("tree[0]->total = %d\n", tree[0]->total);
    {
     short i;
     THuffmanTree *tree;
     for(i = 0; i < backTreeFloater; i++)
     {
         printf("backtree[%d]->character = %d; ", i, backtree[i]->character);
         for(tree = backtree[i]; tree->parent != NULL; tree = tree->parent)
         {
             printf("%c", (tree->parent->right == tree) ? '1' : '0' );
         }
         printf("\n");
     }
    }*/
    /* ~test */
         newtree->left = tree[0];
         tree[0]->parent = newtree;
         newtree->right = tree[1];
         tree[1]->parent = newtree;
         newtree->isLeaf = 0;/* so character is not needed */
         newtree->total = tree[0]->total + tree[1]->total;
         newtree->parent = NULL;
         for(i = 1; i <= treeFloater - 2; i++)
         {
             tree[i] = tree[i+1];
         }
         tree[0] = newtree;
         /* newtree should be reserved, so don't free it! */
         treeFloater--;
        }
    }
    /* test result 1 */
    /*{
     short i;
     THuffmanTree *tree;
     for(i = 0; i < backTreeFloater; i++)
     {
         printf("backtree[%d]->character = %d; ", i, backtree[i]->character);
         for(tree = backtree[i]; tree->parent != NULL; tree = tree->parent)
         {
             printf("%c", (tree->parent->right == tree) ? '1' : '0' );
         }
         printf("\n");
     }
    }*/

    /* Creating Huffman table */

    huffmanTable = (char *)malloc(8960);
    /* maximum size = 256 * (3 + 256 / 8) = 8960 */
    if(huffmanTable == NULL)
    {
        printf("Not enough memory to locate!\n");
        return 2;
    }
    memset(huffmanTable, 0, 8960);/* fill zeroes because this will be written to file */

    {
     short i, huffmanPtr, markLength, innerPtr;
     /* for more compatibility, substitute "short" for "int" */

     THuffmanTree *tree;
     huffmanPtr = 0;
     for(i = 0; i < backTreeFloater; i++)
     {
         /* Add the character */
         *(huffmanTable + huffmanPtr) = backtree[i]->character;
         huffmanPtr++; /* remember to do so */
         /* add then length of the mark and the mark */
         /* reserve the space for storing length information */
         huffmanPtr += 2;
         innerPtr = 0;
         for(tree = backtree[i]; tree->parent != NULL; tree = tree->parent)
         {
             /* because all the allocated memory area has been filled with zeroes, so it is not necessary to make zeroes here */
             *(huffmanTable + huffmanPtr + innerPtr / 8) ^= ((tree->parent->right == tree) ? 1 : 0) << (innerPtr % 8);
             innerPtr++;
         }
         markLength = innerPtr;/* save current mark length */
         /* insert length information */
         huffmanPtr -= 2;
         *((short *)(huffmanTable + huffmanPtr)) = markLength;
         huffmanPtr += 2;
         huffmanPtr += (markLength - 1) / 8 + 1;/* calculate the mininum necessary byte length, and move on */
         /* ~add the length of the mark and the mark */
     }
     tableLength = huffmanPtr;
    }

    /* ~Creating Huffman table */
    printf("Huffman Table finished!\n");

    /* Write to a file */
    if((fp1 = fopen("OUT.DAT", "wb")) == NULL)
    {
        printf("Error writing to file!\n");
        return 1;
    }
    {
     short i, outChar;
     unsigned short ch;/* to keep compatibility, because "char" may be negative */
     char *markBuffer;
     THuffmanTree *tree;
     short innerPtr, markLength, inBytePtr;
     /* Firstly write the length of the huffman table */
     putc(tableLength & 0x00FF, fp1);
     putc(tableLength >> 8, fp1);/* see tableLength is "short" */
     /* Secondly write the table */
     for(i = 0; i < tableLength; i++)
     {
         putc(*(huffmanTable + i), fp1);
     }
     free(huffmanTable);/* finished writing huffman table */

     /* Thirdly write the information */
     markBuffer = (char *)malloc(16);/* 16 * 8 = 256, the max length of a huffman mark here */
     if(markBuffer == NULL)
     {
         printf("Not enough memory to allocate!\n");
         return 2;
     }
     rewind(fp0);
     inBytePtr = 0; outChar = 0;
     while((ch = getc(fp0)) != 0xFFFF)
     {
         /* assume: the character is in one of the leaves */
         for(i = 0; backtree[i]->character != ch; i++)
             ;
         innerPtr = 0;
         memset(markBuffer, 0, 16);/* clear so that clearing bit not needed, speed can be raised */
         for(tree = backtree[i]; tree->parent != NULL; tree = tree->parent)
         {
             *(markBuffer + innerPtr / 8) ^= ((tree->parent->right == tree) ? 1 : 0) << (innerPtr % 8);
             innerPtr++;/* remember */
         }
         markLength = innerPtr;
         for(innerPtr = markLength - 1; innerPtr >= 0; innerPtr--)
         {
             if(inBytePtr == 8)
             {
                 putc(outChar, fp1);
                 inBytePtr = 0;
                 outChar = 0;
             }
             /* because the character is originally clear, so no need to clear the bit */
             outChar ^= (((*(markBuffer + innerPtr / 8)) & (1 << innerPtr % 8)) ? 1 : 0) << inBytePtr;
             inBytePtr++;
         }
     }
     /* remember to put the rest character */
     putc(outChar, fp1);
     /* write the ending byte valid length information at the last byte */
     putc(inBytePtr, fp1);
     free(markBuffer);
    }

    /* all things done, let the files be closed automatically, and let the memory be freed automatically */
    printf("File writing finished, program finished.\n");
    return 0;
}

int IsBiggerHT(THuffmanTree **tree1, THuffmanTree **tree2)
{
    return (*tree1)->total - (*tree2)->total;
}
