// Sphere.cpp: implementation of the CSphere class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GraphicsConclusion.h"
#include "Sphere.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

const CSphere::lineRes = 12;

bool CSphere::CalculateIntersection(const CRay &ray, C3DVector &intersection, bool negative) const
{
    C3DDirection lineDir; // direction of the line
    C3DVector sttPnt; // start point
    C3DVector ballCtr; // ball center
    double theta;
    double phi;
    double y;
    double z;
    double dist; // distance between the ray and the center of the ball
    double xbase; // x-value of the base (center of the ball)
    double xbias; // bias in the x direction
    double itsnX; // x-value of the intersection

    lineDir = ray.GetDirection();
    theta = lineDir.GetTheta();
    phi = lineDir.GetPhi();
    sttPnt = ray.GetStartPoint();
    ballCtr = center;
    // turn according the direction to make the ray parallel to the x-axis
    sttPnt *= C3DMatrix::RotateMatrix(-theta, 2);
    sttPnt *= C3DMatrix::RotateMatrix(PI / 2.0 - phi, 1);
    ballCtr *= C3DMatrix::RotateMatrix(-theta, 2);
    ballCtr *= C3DMatrix::RotateMatrix(PI / 2.0 - phi, 1);
	// move the ball center to X-axis
    y = sttPnt.GetY() - ballCtr.GetY();
    z = sttPnt.GetZ() - ballCtr.GetZ();
    dist = sqrt(y * y + z * z); // the distance between the ball center and the line of the ray
    if (dist >= radius) { return false; }
    xbase = ballCtr.GetX();
    xbias = sqrt(radius * radius - dist * dist);
	if (!negative)
	{ // the shape isn't a negative shape
		itsnX = xbase - xbias; // the rear point
		// check whether the point is on the ray
		if (itsnX <= sttPnt.GetX() + C3DVector::EPSILON)
		{
			itsnX = xbase + xbias; // the front point
			if (itsnX <= sttPnt.GetX() + C3DVector::EPSILON)
			{
				return false;
			}
		}
	}
	else
	{ // the shape is a negative shape so find the nearest out intersection
		// to a sphere, the nearest out intersection may only 
		// be the front intersection
		itsnX = xbase + xbias;
		if (itsnX <= sttPnt.GetX() + C3DVector::EPSILON)
		{
			return false;
		}
	}
    intersection.Set(itsnX, sttPnt.GetY(), sttPnt.GetZ());
    // turn the intersection point back
    intersection *= C3DMatrix::RotateMatrix(phi - PI / 2.0, 1);
    intersection *= C3DMatrix::RotateMatrix(theta, 2);
    return true;
}

CSphere::CSphere(const C3DVector &vctCenter, double dblRadius)
{
    center = vctCenter;
    radius = dblRadius;
}

CSphere::CSphere()
{
    SetToDefault();
}

CSphere::~CSphere()
{
}

void CSphere::Draw(CDC *pDC, const CScreenViewer &screenViewer1, POINT ptBasePosition) const
{
	C3DVector vctPoints[4]; // four ends of planes that are used to represent the sphere
	double dblPhi;  // the phi angle; the angle between the direction and the x-z platform
	double dblTheta;    // the theta angle, the angle of projection of the direction on the x-z platform
	int i;  // loop variable, which leads the longtitude lines
    int j;  // loop variable, which leads the parallel lines
	double dblRadius1;  // radius of smaller horizontal circles
 	CPlane plane1;  // buffer for planes
	CPlane plnClippedPlane; // clipped plane

	for(i = 0; i < lineRes; i++)
	{
		dblPhi = (double)i / lineRes * PI;
		dblRadius1 = radius * sin(dblPhi);
		for(j = 0; j < 2 * lineRes; j++)
		{
			dblTheta = (double)j / (double)lineRes * PI;
			vctPoints[0].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblPhi = (double)(i + 1) / (double)lineRes * PI;
			dblRadius1 = radius * sin(dblPhi);  // dblPhi has been modified, so update dblRadius1
			vctPoints[1].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblTheta = (double)(j + 1) / (double)lineRes * PI;
			vctPoints[2].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblPhi = (double)i / (double)lineRes * PI;
			dblRadius1 = radius * sin(dblPhi);
			vctPoints[3].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[0]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[1]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[2]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[3]));
			screenViewer1.Clip(plnClippedPlane,plane1);
			plane1.Clear(); // release resources for the next round
			plnClippedPlane.Rewind();
			if(plnClippedPlane.IsCPAvailable())
			{
				pDC->MoveTo(
                    Draw_PntInCSToMnt(
                        pDC,
                        plnClippedPlane.GetCurrentPoint(),
                        screenViewer1,
                        ptBasePosition
                    )
                );
				plnClippedPlane.GoToNextPoint();
			    while(plnClippedPlane.IsCPAvailable())
			    {
				    pDC->LineTo(Draw_PntInCSToMnt(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition));
				    plnClippedPlane.GoToNextPoint();
			    }
			    plnClippedPlane.Rewind();
			    if(plnClippedPlane.IsCPAvailable())
                {
				    pDC->LineTo(Draw_PntInCSToMnt(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition));
                }
			}
		}
	}
}

void CSphere::Draw2(CDC *pDC, const CScreenViewer &screenViewer1, POINT ptBasePosition, int width) const
{
	C3DVector vctPoints[4]; // four ends of planes that are used to represent the sphere
	double dblPhi;  // the phi angle; the angle between the direction and the x-z platform
	double dblTheta;    // the theta angle, the angle of projection of the direction on the x-z platform
	int i;  // loop variable, which leads the longtitude lines
    int j;  // loop variable, which leads the parallel lines
	double dblRadius1;  // radius of smaller horizontal circles
 	CPlane plane1;  // buffer for planes
	CPlane plnClippedPlane; // clipped plane

	for(i = 0; i < lineRes; i++)
	{
		dblPhi = (double)i / lineRes * PI;
		dblRadius1 = radius * sin(dblPhi);
		for(j = 0; j < 2 * lineRes; j++)
		{
			dblTheta = (double)j / lineRes * PI;
			vctPoints[0].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblPhi = (double)(i + 1) / (double)lineRes * PI;
			dblRadius1 = radius * sin(dblPhi);  // dblPhi has been modified, so update dblRadius1
			vctPoints[1].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblTheta = (double)(j + 1) / (double)lineRes * PI;
			vctPoints[2].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			dblPhi = (double)i / (double)lineRes * PI;
			dblRadius1 = radius * sin(dblPhi);
			vctPoints[3].Set(center.GetX() + dblRadius1 * cos(dblTheta),
				center.GetY() + dblRadius1 * sin(dblTheta),
				center.GetZ() + radius * cos(dblPhi));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[0]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[1]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[2]));
			plane1.AddPoint(screenViewer1.ConvertPointWToC(vctPoints[3]));
			screenViewer1.Clip(plnClippedPlane,plane1);
			plane1.Clear(); // release resources for the next round
			plnClippedPlane.Rewind();
			if(plnClippedPlane.IsCPAvailable())
			{
				pDC->MoveTo(
                    Draw_PntInCSToMnt2(
                        pDC,
                        plnClippedPlane.GetCurrentPoint(),
                        screenViewer1,
                        ptBasePosition,
                        width
                    )
                );
				plnClippedPlane.GoToNextPoint();
			    while(plnClippedPlane.IsCPAvailable())
			    {
				    pDC->LineTo(Draw_PntInCSToMnt2(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition, width));
				    plnClippedPlane.GoToNextPoint();
			    }
			    plnClippedPlane.Rewind();
			    if(plnClippedPlane.IsCPAvailable())
                {
				    pDC->LineTo(Draw_PntInCSToMnt2(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition, width));
                }
			}
		}
	}
}

C3DDirection CSphere::GetNormal(const C3DVector &pntOnSurf) const
{
    ASSERT(fabs(pntOnSurf.DistanceTo(center) - radius) <= C3DVector::EPSILON);
    return C3DDirection(pntOnSurf - center);
}

void CSphere::GetParameters(
    double &center_x, double &center_y, double &center_z, 
    double &o_radius
) const
{
    center.Get(center_x, center_y, center_z);
    o_radius = radius;
}

CString CSphere::GetShapeType(void) const
{
    return CString("Sphere");
}

bool CSphere::IsInObject(const C3DVector &pnt) const
{
    if(pnt.DistanceTo(center) < radius - C3DVector::EPSILON){
        return true;
    }else{
        return false;
    }
}

bool CSphere::IsInOrOnObject(const C3DVector &pnt) const
{
    if(pnt.DistanceTo(center) <= radius + C3DVector::EPSILON){
        return true;
    }else{
        return false;
    }
}

bool CSphere::IsOnObject(const C3DVector &pnt) const
{
    if(fabs(pnt.DistanceTo(center) - radius) <= C3DVector::EPSILON){
        return true;
    }else{
        return false;
    }
}

void CSphere::Set(const C3DVector &vctCenter, double dblRadius)
{
    center = vctCenter;
    radius = dblRadius;
}

void CSphere::SetParameters(
    double center_x, double center_y, double center_z, 
    double i_radius
)
{
    center.Set(center_x, center_y, center_z);
    radius = i_radius;
}

void CSphere::SetToDefault(void)
{
    center.Set(-0.15, 0.0, 0.0);
    radius = 0.05;
}
