// ScreenViewer.cpp: implementation of the CScreenViewer class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GraphicsConclusion.h"
#include "ScreenViewer.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

// CScreenViewer - screen viewer class.
// The following are coordinate systems that may appear during the world-to-screen conversion.
// world coordinate system
// viewer coordinate system
// screen coordinate system (3D)
// clipping coordinate system
// screen coordinate system (2D)
//
// Default values:
// vrp(0.0, 0.0, 0.0), cop(0.5, 0.0, 0.0), vpn(-1.0, 0.0, 0.0), vup(0.0, 0.0, 1.0),
// uMax(0.140208), uMin(-0.140208), vMax(0.105156), vMin(-0.105156),
// frClip(0.0), bkClip(1.0)

C3DMatrix CScreenViewer::ChMatrix(void) const
{
    C3DMatrix mtxResult;    // matrix to return

	// Reverses z coordinates of points in viewer coordinate system.
    mtxResult.SetToUnit(false);
    mtxResult.SetItem(2 * 4 + 2, -1.0);
    return mtxResult;
}

void CScreenViewer::Clip(CPlane &plnClippedPlane, CPlane &plnSource) const
{
    // Points in plnSource should be in clipping mode coordinate space.
    C3DVector vctPoint1;    // leading point
    C3DVector vctPoint2;    // following point

    plnClippedPlane.Clear();    // clear the output plane
    plnSource.Rewind(); // rewind to the first point; operation is required on plnSource
    if(!plnSource.IsCPAvailable())
    {
        // No points composed the plane.
        return;
    }
    vctPoint1 = plnSource.GetCurrentPoint();
    plnSource.GoToNextPoint();
    while(plnSource.IsCPAvailable())
    {
        vctPoint2 = vctPoint1;
        vctPoint1 = plnSource.GetCurrentPoint();
        if(IsInsideOrOnside(vctPoint2))
        {
            if(IsInsideOrOnside(vctPoint1))
            {
                plnClippedPlane.AddPoint(vctPoint1);
            }
            else
            {
                plnClippedPlane.AddPoint(ClpBndIntersection(vctPoint2, vctPoint1));
            }
        }
        else if(IsInsideOrOnside(vctPoint1))
        {
            plnClippedPlane.AddPoint(ClpBndIntersection(vctPoint1, vctPoint2));
            plnClippedPlane.AddPoint(vctPoint1);
        }
        plnSource.GoToNextPoint();
    }
    vctPoint2 = vctPoint1;
    plnSource.Rewind();
    vctPoint1 = plnSource.GetCurrentPoint();
    if(IsInsideOrOnside(vctPoint2))
    {
        if(IsInsideOrOnside(vctPoint1))
        {
            plnClippedPlane.AddPoint(vctPoint1);
        }
        else
        {
            plnClippedPlane.AddPoint(ClpBndIntersection(vctPoint2, vctPoint1));
        }
    }
    else if(IsInsideOrOnside(vctPoint1))
    {
        plnClippedPlane.AddPoint(ClpBndIntersection(vctPoint1, vctPoint2));
        plnClippedPlane.AddPoint(vctPoint1);
    }
}

C3DVector CScreenViewer::ClpBndIntersection(const C3DVector &vctInnerPoint, const C3DVector &vctOuterPoint) const
{
    // UpdateClpEnvironment should be called before calling this function.
    // vctInnerPoint and vctOuterPoint should be in clipping mode space.
	double dblT;    // "canshu" in Chinese
	double dblT2;   // "canshu" in Chinese; for storing a temporary value
	double a[6] = { 0.0, 0.0, 0.0, 0.0, 1.0, -1.0 }; // Six clipping planes in the form of a * x + b * y + c * z + d = 0.
	double b[6] = { 0.0, 0.0, -1.0, 1.0, 0.0, 0.0 }; // clipping planes are listed in the order "front, back, up, down, left, right".
	double c[6] = { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 };
	double d[6] = { -buffer.clipA, -1.0, 0.0, 0.0, 0.0, 0.0 };
	int i;  // loop variable
	double dblDenominator;  // the common denominator
	const C3DVector &vctPoint1 = vctInnerPoint;    // point buffer/reference 1
    const C3DVector &vctPoint2 = vctOuterPoint;    // point buffer/reference 2

	i = 0;
	dblDenominator = a[i] * (vctPoint2.GetX() - vctPoint1.GetX()) + b[i] * (vctPoint2.GetY() - vctPoint1.GetY()) + c[i] * (vctPoint2.GetZ() - vctPoint1.GetZ());
	if( dblDenominator == 0.0 )
    {
		dblT = -1; // set it to an invalid value
    }
	else
    {
		dblT = (-d[i] - a[i] * vctPoint1.GetX() - b[i] * vctPoint1.GetY() - c[i] * vctPoint1.GetZ()) / dblDenominator;
    }
	if( dblT < 0.0 || dblT > 1.0 ) // dblT is invalid
    {
		dblT = -1.0;
    }
	i++;
	for( ; i < 6; i++ )
	{
		dblDenominator = a[i] * (vctPoint2.GetX() - vctPoint1.GetX()) + b[i] * (vctPoint2.GetY() - vctPoint1.GetY()) + c[i] * (vctPoint2.GetZ() - vctPoint1.GetZ());
		if( dblDenominator == 0.0 )
        {
			dblT2 = -1.0; // be other than any possible value
        }
		else
        {
			dblT2 = (-d[i] - a[i] * vctPoint1.GetX() - b[i] * vctPoint1.GetY() - c[i] * vctPoint1.GetZ()) / dblDenominator;
        }
		if( dblT2 >= 0.0 && dblT2 <= 1.0 )
        {   // dblT2 is valid
			if( dblT < 0.0 || dblT2 < dblT )
            {   // dblT is invalid or dblT2 is smaller than dblT
				dblT = dblT2;
            }
		}
	}
	if( dblT >= 0.0 )
    {   // dblT is valid
		return C3DVector(dblT * (vctOuterPoint.GetX() - vctInnerPoint.GetX()) + vctInnerPoint.GetX(),
			dblT * (vctOuterPoint.GetY() - vctInnerPoint.GetY()) + vctInnerPoint.GetY(),
			dblT * (vctOuterPoint.GetZ() - vctInnerPoint.GetZ()) + vctInnerPoint.GetZ());
    }
	else
    {
		return vctInnerPoint;
    }
}

C3DVector CScreenViewer::ConvertPointCToS(const C3DVector &vctPointInClpSpace) const
{
    // UpdateAll should be called before calling this function.
    return vctPointInClpSpace * buffer.CToSMatrix;
}

C3DVector CScreenViewer::ConvertPointSToC(const C3DVector &vctPointInScreen) const
{
    // UpdateAll should be called before calling this function.
    return vctPointInScreen * buffer.SToCMatrix;
}

POINT CScreenViewer::ConvertPointCToMonitor(const C3DVector &vctPointInClpSpc) const
{
	POINT ptResult;     // result point
    C3DVector vctPointBuffer;   // point buffer
	double dblScale;    // scale

    vctPointBuffer = ProjectPointInClippingSpace(vctPointInClpSpc);
	dblScale = 800.0 / (uMax - uMin);
	ptResult.x = (long)floor((vctPointBuffer.GetX() * dblScale) + 0.5);
	ptResult.y = -((long)floor((vctPointBuffer.GetY() * dblScale) + 0.5));    // the y axis of the screen of the computer is reversed
	return ptResult;
}

POINT CScreenViewer::ConvertPointCToMonitor2(const C3DVector &pntInC, int width) const
{
    // int width: width used to display the picture on the screen
    POINT rslt_point; // result point
    C3DVector pntBuf; // point buffer
	double scale; // the scale

    pntBuf = ProjectPointInClippingSpace(pntInC);
	scale = width / (uMax - uMin);
	rslt_point.x = (long)floor((pntBuf.GetX() * scale) + 0.5);
	rslt_point.y = -((long)floor((pntBuf.GetY() * scale) + 0.5)); // the y axis of the screen of the computer is reversed
	return rslt_point;
}

POINT CScreenViewer::ConvertPointSToMonitor(const C3DVector &vctPointInScreen) const
{
	POINT ptResult;     // result point
    C3DVector vctPointBuffer;   // point buffer
	double dblScale;    //scale

    vctPointBuffer = ProjectPoint(vctPointInScreen);
	dblScale = 800.0 / (uMax - uMin);
	ptResult.x = (long)floor((vctPointBuffer.GetX() * dblScale) + 0.5);
	ptResult.y = -((long)floor((vctPointBuffer.GetY() * dblScale) + 0.5));    // the y axis of the screen of the computer is reversed
	return ptResult;
}

C3DVector CScreenViewer::ConvertPointWToC(const C3DVector &vctPointInWorld) const
{
    // UpdateAll should be called before calling this function.
    return vctPointInWorld * buffer.WToCMatrix;
}

C3DVector CScreenViewer::ConvertPointWToS(const C3DVector &vctPointInWorld) const
{
    // UpdateAll should be called before calling this function.
    return vctPointInWorld * buffer.WToSMatrix;
}

void CScreenViewer::CreateRayThroughScreen(CRay &result, int xpos, int ypos, int xmax, int ymax) const
{
    C3DMatrix vtow_mtx; // viewer to world matrix
    bool invnRslt; // inversion result

    vtow_mtx = buffer.WToVMatrix; // copy the world to screen matrix
    invnRslt = vtow_mtx.Inverse(); // inverse the matrix and record the result
    ASSERT(invnRslt);

    C3DVector sp; // start point of the ray
	C3DVector ep; // end point of the ray
    C3DVector dir; // direction of the ray in the form of a vector
    double scrx; // screen x; x coordinate on the screen
    double scry; // screen y; y coordinate on the screen

    scrx = (double)xpos * (uMax - uMin) / (double)xmax / 2.0;
    scry = (double)ypos * (vMax - vMin) / (double)ymax / 2.0;
    sp.Set(0.0, 0.0, 0.0);
    ep.Set(scrx, -scry, buffer.vrp2.GetZ()); // scry is reversed in the viewer coordinate system
    sp *= vtow_mtx;
    ep *= vtow_mtx;
    dir.Set(ep.GetX() - sp.GetX(), ep.GetY() - sp.GetY(), ep.GetZ() - sp.GetZ());
    result.SetDirection(C3DDirection(dir));
    result.SetStartPoint(sp);
}

CScreenViewer::CScreenViewer(const CScreenViewer &svrSource)
{
    ASSERT(false);  // no copy method is currently available
}

CScreenViewer::CScreenViewer():
vrp(0.0, 0.0, 0.0), cop(0.5, 0.0, 0.0), vpn(-1.0, 0.0, 0.0), vup(0.0, 0.0, 1.0),
uMax(0.140208), uMin(-0.140208), vMax(0.105156), vMin(-0.105156),
frClip(0.0), bkClip(1.0)
{
    UpdateAll();
}

CScreenViewer::~CScreenViewer()
{
}

bool CScreenViewer::IsInsideOrOnside(const C3DVector &vctPoint) const
{
    // UpdateClpEnvironment should be called before calling this function.
    // vctPoint should be in clipping mode space.
    bool blnOutside;    // outside flag

	if(vctPoint.GetZ() < buffer.clipA || vctPoint.GetZ() > 1.0 ||
        vctPoint.GetY() > vctPoint.GetZ() || vctPoint.GetY() < -vctPoint.GetZ() ||
        vctPoint.GetX() > vctPoint.GetZ() || vctPoint.GetX() < -vctPoint.GetZ())
    {
		blnOutside = true;
    }
    else
    {
        blnOutside = false;
    }
    return !blnOutside;
}

C3DMatrix CScreenViewer::ProjectionMatrix(void) const
{
	// buffer.vrp2 should have been updated before calling this function.
	C3DMatrix mtxResult;    // result matrix

	mtxResult.SetToUnit(false);
	mtxResult.SetItem(3 * 4 + 3, 0.0);
	mtxResult.SetItem(2 * 4 + 3, 1.0 / buffer.vrp2.GetZ());
	return mtxResult;
}

C3DVector CScreenViewer::ProjectPoint(const C3DVector &vctPointInScreen) const
{
	C3DVector vctResultPoint;   // result point

	vctResultPoint = vctPointInScreen * ProjectionMatrix();
	vctResultPoint.StandardizePoint();  // apply after a projection
	return vctResultPoint;
}

C3DVector CScreenViewer::ProjectPointInClippingSpace(const C3DVector &vctPointInClpSpc) const
{
    C3DVector vctResultPoint;   // result point

    vctResultPoint = vctPointInClpSpc * buffer.CToSPrjMatrix;
    vctResultPoint.StandardizePoint();  // apply after a projection
    return vctResultPoint;
}

C3DMatrix CScreenViewer::RMatrix(void) const
{
    C3DMatrix mtxResult;    // matrix to return
    int i;  // loop variable
    double dblArray[16];    // array for storing matrix data

    // Rotates points from world coordinates to viewer coordinates.
    // This function requires buffer.u and buffer.v to be updated.
    mtxResult.Get(dblArray);
    for(i = 0; i < 3; i++)
    {
        dblArray[i * 4 + 0] = buffer.u.GetArrayItem(i);
    }
    for(i = 0; i < 3; i++)
    {
        dblArray[i * 4 + 1] = buffer.v.GetArrayItem(i);
    }
    for(i = 0; i < 3; i++)
    {
        dblArray[i * 4 + 2] = -vpn.GetArrayItem(i);
    }
    dblArray[3 * 4 + 3] = 1.0;
    mtxResult.Set(dblArray);
    return mtxResult;
}

void CScreenViewer::SetBasic(const C3DVector &newvrp, const C3DVector &newcop, const C3DDirection &newvpn, const C3DDirection &newvup)
{
    vrp = newvrp;
    cop = newcop;
    vpn = newvpn;
    vup = newvup;
    UpdateAll();
}

C3DMatrix CScreenViewer::TMatrix(void) const
{
    C3DMatrix mtxResult;    // matrix to return
    double dblArray[16];    // array for storing matrix data

	// Moves points from world coordinates to viewer coordinates.
    mtxResult.SetToUnit(false);
    mtxResult.Get(dblArray);
    dblArray[3 * 4 + 0] = -(vrp.GetX() + cop.GetX());
    dblArray[3 * 4 + 1] = -(vrp.GetY() + cop.GetY());
    dblArray[3 * 4 + 2] = -(vrp.GetZ() + cop.GetZ());
    mtxResult.Set(dblArray);
    return mtxResult;
}

void CScreenViewer::UpdateAll(void)
{
    // basic initializations
    UpdateUV();
    UpdateWToVMatrix();
    UpdateWToSMatrix();
    UpdateClpEnvironment();
    // acceleration initializations
    buffer.WToCMatrix = buffer.WToSMatrix * buffer.SToCMatrix;
    buffer.projectionMatrix = ProjectionMatrix();
    buffer.CToSPrjMatrix = buffer.CToSMatrix * buffer.projectionMatrix;
}

void CScreenViewer::UpdateClpEnvironment(void)
{
	double dblScaleX;   // x scaler
    double dblScaleY;   // y scaler
    double dblScaleZ;   // z scaler

	buffer.clipA = (buffer.vrp2.GetZ() + frClip) / (buffer.vrp2.GetZ() + bkClip);
	dblScaleX = 2.0 * buffer.vrp2.GetZ() / ((buffer.vrp2.GetZ() + bkClip) * (uMax - uMin));
	dblScaleY = 2.0 * buffer.vrp2.GetZ() / ((buffer.vrp2.GetZ() + bkClip) * (vMax - vMin));
	dblScaleZ = 1.0 / (buffer.vrp2.GetZ() + bkClip);
	buffer.SToCMatrix.SetToUnit(true);
	buffer.SToCMatrix.SetItem(0 * 4 + 0, dblScaleX);
	buffer.SToCMatrix.SetItem(1 * 4 + 1, dblScaleY);
	buffer.SToCMatrix.SetItem(2 * 4 + 2, dblScaleZ);
	buffer.CToSMatrix.SetToUnit(true);
	buffer.CToSMatrix.SetItem(0 * 4 + 0, 1.0 / dblScaleX);
	buffer.CToSMatrix.SetItem(1 * 4 + 1, 1.0 / dblScaleY);
	buffer.CToSMatrix.SetItem(2 * 4 + 2, 1.0 / dblScaleZ);
}

void CScreenViewer::UpdateUV(void)
{
    buffer.v = vup - double(vpn * vup) * vpn;
    buffer.u = vpn.XMultiply(buffer.v);
}

void CScreenViewer::UpdateWToVMatrix(void)
{
    buffer.WToVMatrix = (TMatrix() * RMatrix()) * ChMatrix();
}

void CScreenViewer::UpdateWToSMatrix(void)
{
    double dblCenterU;      // u coordinate of the center of the screen in screen coordinate system
    double dblCenterV;      // v coordinate of the center of the screen in screen coordinate system
    double dblCenterX;      // x coordinate of the center of the screen in viewer coordinate system
    double dblCenterY;      // y coordinate of the center of the screen in viewer coordinate system
    C3DMatrix mtxShMatrix;  // auxiliary shifting matrix, which makes the screen center on axis Z

    // Call UpdateUV then call UpdateWToVMatrix before calling me.
    buffer.vrp2 = vrp * buffer.WToVMatrix;  // VRP in viewer coordinate system
    dblCenterU = 0.5 * (uMin + uMax);
    dblCenterV = 0.5 * (vMin + vMax);
    dblCenterX = buffer.vrp2.GetX() + dblCenterU;
    dblCenterY = buffer.vrp2.GetY() + dblCenterV;
    mtxShMatrix.SetToUnit(false);
    mtxShMatrix.SetItem(2 * 4 + 0, -dblCenterX / buffer.vrp2.GetZ());
    mtxShMatrix.SetItem(2 * 4 + 1, -dblCenterY / buffer.vrp2.GetZ());
    buffer.WToSMatrix = buffer.WToVMatrix * mtxShMatrix;
}
