// Cuboid.cpp: implementation of the CCuboid class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GraphicsConclusion.h"
#include "Cuboid.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

bool CCuboid::CalculateIntersection(const CRay &ray, C3DVector &vctIntersection, bool negative) const
{
    C3DVector startPoint;	// start point of the ray
    C3DDirection rayDir;	// direction of the ray
    C3DVector pointArray[6];	// array of points
    bool pntvld_arr[6];	// point validity array
    int i;	// loop variable
    double t[6];	// parameters ("time"s) of the line
    int nrstPntIdx;	// the index of the nearest point
    double most_t;	// the minimum/maximum one of the parameters ("t"s)
    double dblSwlDepth;	// swelling m_depth
    double dblSwlWidth;	// swelling m_width
    double dblSwlHeight;	// swelling m_height

    startPoint = ray.GetStartPoint();
    rayDir = ray.GetDirection();
    // move the ray to the cuboid-based coordinate system
    startPoint -= m_referenceEnd;
    // turn the ray from the world coordinate system to cuboid-based coordinate system
    startPoint *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    startPoint *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    startPoint *= C3DMatrix::RotateMatrix(-m_alpha, 0);
	// turn the direction like the point
    rayDir *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    rayDir *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    rayDir *= C3DMatrix::RotateMatrix(-m_alpha, 0);
    rayDir.FineTune();
    // find out all intersections between 
	// the line that contains the ray and the faces of the cuboid
    for (i = 0; i < 6; i++) {
        pntvld_arr[i] = true;
    }
	// first eliminate those faces that are parallel with the line
	// faces 0 and 3 are parallel to the Y-Z plane
    if (fabs(rayDir.GetX()) <= C3DVector::EPSILON)
	{
		pntvld_arr[0] = false;
		pntvld_arr[3] = false;
	}
	// faces 1 and 4 are parallel to the Z-X plane
    if (fabs(rayDir.GetY()) <= C3DVector::EPSILON)
	{
		pntvld_arr[1] = false;
		pntvld_arr[4] = false;
	}
	// faces 2 and 5 are parallel to the X-Y plane
    if (fabs(rayDir.GetZ()) <= C3DVector::EPSILON)
	{
		pntvld_arr[2] = false;
		pntvld_arr[5] = false;
	}
	// second calculate the T value of the intersections
    if (pntvld_arr[0])
	{
		// -startPoint.GetX() means 0 - startPoint.GetX() 
		// (the end point minuses the start point)
        t[0] = -startPoint.GetX() / rayDir.GetX(); // intersection with the y-z plane
        pointArray[0] = startPoint + t[0] * rayDir;
    }
    if (pntvld_arr[1])
	{
        t[1] = -startPoint.GetY() / rayDir.GetY(); // intersection with the z-x plane
        pointArray[1] = startPoint + t[1] * rayDir;
    }
    if (pntvld_arr[2])
	{
        t[2] = -startPoint.GetZ() / rayDir.GetZ(); // intersection with the x-y plane
        pointArray[2] = startPoint + t[2] * rayDir;
    }
    if (pntvld_arr[3])
	{
		// intersection with the plane of the opposite y-z face
        t[3] = (m_depth - startPoint.GetX()) / rayDir.GetX();
        pointArray[3] = startPoint + t[3] * rayDir;
    }
    if (pntvld_arr[4])
	{
		// intersection with the plane of the opposite z-x face
		// the plane intersects the y-axis at a negative position due to 
		// the definition of the cuboid
        t[4] = (-m_width - startPoint.GetY()) / rayDir.GetY();
        pointArray[4] = startPoint + t[4] * rayDir;
    }
    if (pntvld_arr[5])
	{
		// intersection with the plane of the opposite x-y face
        t[5] = (m_height - startPoint.GetZ()) / rayDir.GetZ();
        pointArray[5] = startPoint + t[5] * rayDir;
    }
    // eliminate points that are not on the ray
    for (i = 0; i < 6; i++) {
        if (t[i] <= C3DVector::EPSILON) {
            pntvld_arr[i] = false;
        }
    }
    // eliminate points that are not on the faces of the cuboid
	// (some points may be outside the faces, but are on the planes)
	// assume all points are on the planes so eliminate all that 
	// are outside the faces
    dblSwlDepth = m_depth + C3DVector::EPSILON;
    dblSwlWidth = -m_width - C3DVector::EPSILON;
    dblSwlHeight = m_height + C3DVector::EPSILON;
    for(i = 0; i < 6; i++)
    {
		if (pntvld_arr[i]) {
			if (pointArray[i].GetX() < -C3DVector::EPSILON || 
				pointArray[i].GetY() > C3DVector::EPSILON || 
				pointArray[i].GetZ() < -C3DVector::EPSILON || 
				pointArray[i].GetX() > dblSwlDepth || 
				pointArray[i].GetY() < dblSwlWidth || 
				pointArray[i].GetZ() > dblSwlHeight)
			{
				pntvld_arr[i] = false;
			}
		}
    }
    // find the nearest intersection
	nrstPntIdx = -1;
	if (!negative)
	{ // the shape isn't a negative shape
		// just find the nearest intersection
		for (i = 0; i < 6; i++)
		{
			if (pntvld_arr[i])
			{
				if (nrstPntIdx == -1)
				{
					most_t = t[i];
					nrstPntIdx = i;
				}
				else
				{
					if (t[i] < most_t)
					{
						most_t = t[i];
						nrstPntIdx = i;
					}
				}
			}
		}
	}
	else
	{ // the shape is a negative shape
		// because it is a cuboid, the nearest out intersection
		// may only be the farthest intersection
		for (i = 0; i < 6; i++)
		{
			if (pntvld_arr[i])
			{
				if (nrstPntIdx == -1)
				{
					most_t = t[i];
					nrstPntIdx = i;
				}
				else
				{
					if (t[i] > most_t)
					{
						most_t = t[i];
						nrstPntIdx = i;
					}
				}
			}
		}
	}
    if (nrstPntIdx == -1) {
        return false;
    } else {
        vctIntersection = pointArray[nrstPntIdx];
		// turn the point back into World Coordinate System
		// in the order "m_alpha, phi, theta, move"
		vctIntersection *= C3DMatrix::RotateMatrix(m_alpha, 0);
		vctIntersection *= C3DMatrix::RotateMatrix(m_buffer.phi - CShape::PI / 2.0, 1);
		vctIntersection *= C3DMatrix::RotateMatrix(m_buffer.theta, 2);
		vctIntersection += m_referenceEnd;
        return true;
    }
}

CCuboid::CCuboid(const C3DVector &vctRfrEnd, 
				 const C3DDirection &dirCbdDrc, 
				 double dblAlpha, double dblDepth, 
				 double dblHeight, double dblWidth)
	:m_referenceEnd(vctRfrEnd), 
	m_cuboidDirection(dirCbdDrc)
{
	Set(vctRfrEnd, dirCbdDrc, dblAlpha, dblDepth, dblHeight, dblWidth); // for simplicity and consistency
}

CCuboid::CCuboid()
{
    SetToDefault();
}

CCuboid::~CCuboid()
{
}

void CCuboid::Draw(CDC *pDC, const CScreenViewer &screenViewer1, POINT ptBasePosition) const
{
    C3DVector vctTmpPnts[8];	// temporary points
	int i;	// loop variable
	CPlane plane1;	// m_buffer for planes
	CPlane plnClippedPlane;	// clipped plane1
	int intVertexIndices[24] =
	{
		0, 1, 2, 3,
		4, 5, 6, 7,
		0, 1, 5, 4,
		1, 2, 6, 5,
		2, 3, 7, 6,
		3, 0, 4, 7
	};
    POINT ptTmpPoint;	// temporary point

    for(i = 0; i < 8; i++)
	{
        // convert points from world space to clipping space
		vctTmpPnts[i] = screenViewer1.ConvertPointWToC(m_buffer.ends[i]);
	}
	for(i = 0; i < 24; i += 4)
	{
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 1]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 2]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 3]]);
		screenViewer1.Clip(plnClippedPlane, plane1);
		plane1.Clear();
		plnClippedPlane.Rewind();
		if(plnClippedPlane.IsCPAvailable())
		{
            // get and convert the point from clipping space to the computer monitor
            ptTmpPoint = Draw_PntInCSToMnt(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition);
            // draw the first point
			pDC->MoveTo(ptTmpPoint);
			plnClippedPlane.GoToNextPoint();
		    while( plnClippedPlane.IsCPAvailable() )
		    {
                // get and convert the point from clipping space to the computer monitor
                ptTmpPoint = Draw_PntInCSToMnt(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition);
                // draw the point
                pDC->LineTo(ptTmpPoint);
			    plnClippedPlane.GoToNextPoint();
		    }
            // draw back to the first point
		    plnClippedPlane.Rewind();
            // get and convert the point from clipping space to the computer monitor
            ptTmpPoint = Draw_PntInCSToMnt(pDC, plnClippedPlane.GetCurrentPoint(), screenViewer1, ptBasePosition);
            pDC->LineTo(ptTmpPoint);
		}
	}
}

void CCuboid::Draw2(CDC *pDC, const CScreenViewer &scrnVwr, POINT basePos, int m_width) const
{
    C3DVector vctTmpPnts[8];	// temporary points
	int i;	// loop variable
	CPlane plane1;	// m_buffer for planes
	CPlane plnClippedPlane;	// clipped plane1
	int intVertexIndices[24] =
	{
		0, 1, 2, 3,
		4, 5, 6, 7,
		0, 1, 5, 4,
		1, 2, 6, 5,
		2, 3, 7, 6,
		3, 0, 4, 7
	};
    POINT ptTmpPoint;	// temporary point

    for(i = 0; i < 8; i++)
	{
        // convert points from world space to clipping space
		vctTmpPnts[i] = scrnVwr.ConvertPointWToC(m_buffer.ends[i]);
	}
	for(i = 0; i < 24; i += 4)
	{
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 1]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 2]]);
		plane1.AddPoint(vctTmpPnts[intVertexIndices[i + 3]]);
		scrnVwr.Clip(plnClippedPlane, plane1);
		plane1.Clear();
		plnClippedPlane.Rewind();
		if(plnClippedPlane.IsCPAvailable())
		{
            // get and convert the point from clipping space to the computer monitor
            ptTmpPoint = Draw_PntInCSToMnt2(pDC, plnClippedPlane.GetCurrentPoint(), scrnVwr, basePos, m_width);
            // draw the first point
			pDC->MoveTo(ptTmpPoint);
			plnClippedPlane.GoToNextPoint();
		    while( plnClippedPlane.IsCPAvailable() )
		    {
                // get and convert the point from clipping space to the computer monitor
                ptTmpPoint = Draw_PntInCSToMnt2(pDC, plnClippedPlane.GetCurrentPoint(), scrnVwr, basePos, m_width);
                // draw the point
                pDC->LineTo(ptTmpPoint);
			    plnClippedPlane.GoToNextPoint();
		    }
            // draw back to the first point
		    plnClippedPlane.Rewind();
            // get and convert the point from clipping space to the computer monitor
            ptTmpPoint = Draw_PntInCSToMnt2(pDC, plnClippedPlane.GetCurrentPoint(), scrnVwr, basePos, m_width);
            pDC->LineTo(ptTmpPoint);
		}
	}
}

C3DDirection CCuboid::GetNormal(const C3DVector &pntOnSurf) const
{
    C3DVector tmpPnt;	// temporary point
    int i;
    
    // turn pntOnSurf to the cuboid-based coordinate system
    tmpPnt = pntOnSurf - m_referenceEnd;
    tmpPnt *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    tmpPnt *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    tmpPnt *= C3DMatrix::RotateMatrix(-m_alpha, 0);
    if (fabs(tmpPnt.GetX() - m_depth) <= C3DVector::EPSILON) {
        i = 0;
    } else if (fabs(tmpPnt.GetX()) <= C3DVector::EPSILON) {
        i = 1;
    } else if (fabs(tmpPnt.GetY() + m_width) <= C3DVector::EPSILON) {
        i = 2;
    } else if (fabs(tmpPnt.GetY()) <= C3DVector::EPSILON) {
        i = 3;
    } else if(fabs(tmpPnt.GetZ() - m_height) <= C3DVector::EPSILON) {
        i = 4;
    } else if(fabs(tmpPnt.GetZ()) <= C3DVector::EPSILON) {
        i = 5;
    } else {
        i = 6;
    }
    ASSERT(i != 6); // if i equals to 6, the point is not on the surface.
    switch(i){
    case 0:
		// normals in the X direction don't need to turn the m_alpha angle
        return C3DVector(1.0, 0.0, 0.0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    case 1:
        return C3DVector(-1.0, 0.0, 0.0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    case 2:
        return C3DVector(0.0, -1.0, 0.0) * 
			C3DMatrix::RotateMatrix(m_alpha, 0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    case 3:
        return C3DVector(0.0, 1.0, 0.0) * 
			C3DMatrix::RotateMatrix(m_alpha, 0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    case 4:
        return C3DVector(0.0, 0.0, 1.0) * 
			C3DMatrix::RotateMatrix(m_alpha, 0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    case 5:
        return C3DVector(0.0, 0.0, -1.0) * 
			C3DMatrix::RotateMatrix(m_alpha, 0) * 
			C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * 
			C3DMatrix::RotateMatrix(m_buffer.theta, 2);
        break;
    default:
		ASSERT(false);
        return C3DVector(0.0, 0.0, 0.0); // impossible
    }
}

void CCuboid::GetParameters(
    double &refend_x, double &refend_y, double &refend_z, 
    double &o_depth, double &o_width, double &o_height, 
    double &o_theta, double &o_phi, double &o_alpha
) const
{
    m_referenceEnd.Get(refend_x, refend_y, refend_z);
    o_depth = m_depth;
    o_width = m_width;
    o_height = m_height;
    o_theta = m_buffer.theta;
    o_phi = m_buffer.phi;
    o_alpha = m_alpha;
}

CString CCuboid::GetShapeType(void) const
{
    return CString("Cuboid");
}

bool CCuboid::IsInObject(const C3DVector &pnt) const
{
    C3DVector tmpPnt; // temporary point
    int i;
    
    // turn pnt to the cuboid-based coordinate system
    tmpPnt = pnt - m_referenceEnd;
    tmpPnt *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    tmpPnt *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    tmpPnt *= C3DMatrix::RotateMatrix(-m_alpha, 0);
	// exclude all outer situations
    if (tmpPnt.GetX() >= m_depth - C3DVector::EPSILON) {
        i = 0;
    } else if (tmpPnt.GetX() <= C3DVector::EPSILON) {
        i = 1;
    } else if (tmpPnt.GetY() <= -m_width + C3DVector::EPSILON) {
        i = 2;
    } else if (tmpPnt.GetY() >= -C3DVector::EPSILON) {
        i = 3;
    } else if (tmpPnt.GetZ() >= m_height - C3DVector::EPSILON) {
        i = 4;
    } else if (tmpPnt.GetZ() <= C3DVector::EPSILON) {
        i = 5;
    } else {
        i = 6;
    }
    if(i == 6){
        return true;
    }else{
        return false;
    }
}

bool CCuboid::IsInOrOnObject(const C3DVector &pnt) const
{
    C3DVector tmpPnt; // temporary point
    int i;
    
    // turn pnt to the cuboid-based coordinate system
    tmpPnt = pnt - m_referenceEnd;
    tmpPnt *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    tmpPnt *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    tmpPnt *= C3DMatrix::RotateMatrix(-m_alpha, 0);
    if (tmpPnt.GetX() > m_depth + C3DVector::EPSILON) {
        i = 0;
    } else if (tmpPnt.GetX() < -C3DVector::EPSILON) {
        i = 1;
    } else if (tmpPnt.GetY() < -m_width - C3DVector::EPSILON) {
        i = 2;
    } else if (tmpPnt.GetY() > C3DVector::EPSILON) {
        i = 3;
    } else if (tmpPnt.GetZ() > m_height + C3DVector::EPSILON) {
        i = 4;
    } else if (tmpPnt.GetZ() < -C3DVector::EPSILON) {
        i = 5;
    } else {
        i = 6;
    }
    if(i == 6){
        return true;
    }else{
        return false;
    }
}

bool CCuboid::IsOnObject(const C3DVector &pnt) const
{
    C3DVector tmpPnt; // temporary point
    int i;
    
    // turn pnt to the cuboid-based coordinate system
    tmpPnt = pnt - m_referenceEnd;
    tmpPnt *= C3DMatrix::RotateMatrix(-m_buffer.theta, 2);
    tmpPnt *= C3DMatrix::RotateMatrix(PI / 2.0 - m_buffer.phi, 1);
    tmpPnt *= C3DMatrix::RotateMatrix(-m_alpha, 0);
    if(fabs(tmpPnt.GetX() - m_depth) <= C3DVector::EPSILON){
        i = 0;
    }else if(fabs(tmpPnt.GetX()) <= C3DVector::EPSILON){
        i = 1;
    }else if(fabs(tmpPnt.GetY() + m_width) <= C3DVector::EPSILON){
        i = 2;
    }else if(fabs(tmpPnt.GetY()) <= C3DVector::EPSILON){
        i = 3;
    }else if(fabs(tmpPnt.GetZ() - m_height) <= C3DVector::EPSILON){
        i = 4;
    }else if(fabs(tmpPnt.GetZ()) <= C3DVector::EPSILON){
        i = 5;
    }else{
        i = 6;
    }
    if(i != 6){
        return true;
    }else{
        return false;
    }
}

void CCuboid::Set(const C3DVector &vctRfrEnd, const C3DDirection &dirCbdDrc, double dblAlpha, double dblDepth, double dblHeight, double dblWidth)
{
    ASSERT(dblDepth > C3DVector::EPSILON && 
		dblHeight > C3DVector::EPSILON && 
		dblWidth > C3DVector::EPSILON);
    m_referenceEnd = vctRfrEnd;
    m_cuboidDirection = dirCbdDrc;
    m_alpha = dblAlpha;
    m_depth = dblDepth;
    m_height = dblHeight;
    m_width = dblWidth;
    UpdateBuffer();
}

void CCuboid::SetParameters(
    double refend_x, double refend_y, double refend_z, 
    double o_depth, double o_width, double o_height, 
    double o_theta, double o_phi, double o_alpha
)
{
    m_referenceEnd.Set(refend_x, refend_y, refend_z);
    m_depth = o_depth;
    m_width = o_width;
    m_height = o_height;
    m_cuboidDirection.SetThetaPhi(o_theta, o_phi);
    m_alpha = o_alpha;
    UpdateBuffer();
}

void CCuboid::SetToDefault(void)
{
	m_referenceEnd.Set(-0.10, -0.05, -0.05);
    m_cuboidDirection.Set(-1.0, 0.0, 0.0);
    m_alpha = 0.0;
    m_depth = 0.1;
    m_height = 0.1;
    m_width = 0.1;
	UpdateBuffer();
}

void CCuboid::UpdateBuffer(void)
{
    m_buffer.phi = m_cuboidDirection.GetPhi();
    m_buffer.theta = m_cuboidDirection.GetTheta();
    m_buffer.lines[0] = m_depth * m_cuboidDirection;  // complete
    m_buffer.lines[1] = C3DVector(0.0, 0.0, 1.0) * C3DMatrix::RotateMatrix(m_alpha, 0) * C3DMatrix::RotateMatrix(m_buffer.phi - PI / 2.0, 1) * C3DMatrix::RotateMatrix(m_buffer.theta, 2); // not complete: m_height not applied
    m_buffer.lines[2] = m_width * m_cuboidDirection.XMultiply(m_buffer.lines[1]);   // complete
    m_buffer.lines[1] = m_height * m_buffer.lines[1]; // complete
    m_buffer.ends[0] = m_referenceEnd;
    m_buffer.ends[1] = m_referenceEnd + m_buffer.lines[2];
    m_buffer.ends[2] = m_referenceEnd + m_buffer.lines[2] + m_buffer.lines[1];
    m_buffer.ends[3] = m_referenceEnd + m_buffer.lines[1];
    m_buffer.ends[4] = m_referenceEnd + m_buffer.lines[0];
    m_buffer.ends[5] = m_referenceEnd + m_buffer.lines[0] + m_buffer.lines[2];
    m_buffer.ends[6] = m_referenceEnd + m_buffer.lines[0] + m_buffer.lines[2] + m_buffer.lines[1];
    m_buffer.ends[7] = m_referenceEnd + m_buffer.lines[0] + m_buffer.lines[1];
}

static void unused01()
{
}
