// ComplexObject.cpp: implementation of the CComplexObject class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GraphicsConclusion.h"
#include "ComplexObject.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

void CComplexObject::AddCuboid(void)
{
    Clear();
    stRoot = new tagShapeTree;
    stRoot->pShape = new CCuboid();
    stRoot->parent = NULL;
    stRoot->lchild = NULL;
    stRoot->rchild = NULL;
    leafptrs = new tagShapeTree *[1];
    leafptrs[0] = stRoot;
    numofleaves = 1;
}

void CComplexObject::AddSphere(void)
{
    Clear();
    stRoot = new tagShapeTree;
    stRoot->pShape = new CSphere();
    stRoot->parent = NULL;
    stRoot->lchild = NULL;
    stRoot->rchild = NULL;
    leafptrs = new tagShapeTree *[1];
    leafptrs[0] = stRoot;
    numofleaves = 1;
}

bool CComplexObject::CalculateIntersection(const CRay &ray, C3DVector &intersection, int &o_leafIdx) const
{
    int i; // loop variable
    C3DVector nrstPnt; // nearest point
    C3DVector tmpPnt; // temporary point
    C3DVector stPnt; // start point of the ray
    int itsCnt; // intersection count
    int leafIdx; // tree leaf index

    stPnt = ray.GetStartPoint();
    itsCnt = 0;
    for (i = 0; i < numofleaves; i++) {
        if (GetIntersection(i, ray, tmpPnt)) {
            if (IsValidIntersection(tmpPnt)) { // whether the intersection is on the surface
                itsCnt++;
                if (itsCnt == 1) {
                    nrstPnt = tmpPnt;
                    leafIdx = i;
                } else if (tmpPnt.DistanceTo(stPnt) < nrstPnt.DistanceTo(stPnt)) {
                    nrstPnt = tmpPnt;
                    leafIdx = i;
                }
            }
        }
    }
    if (itsCnt != 0) {
        intersection = nrstPnt;
        o_leafIdx = leafIdx;
        return true;
    } else {
        return false;
    }
}

CComplexObject::CComplexObject()
{
    stRoot = NULL;
    leafptrs = NULL;
    numofleaves = 0;
}

CComplexObject::~CComplexObject()
{
	Clear(); // for simplicity and consistency
}

void CComplexObject::Clear(void)
{
    ReleaseShapeTree(stRoot);
    stRoot = NULL;
    if(leafptrs != NULL) delete [] leafptrs;
    leafptrs = NULL;
    numofleaves = 0;
}

bool CComplexObject::CombineWith(CComplexObject &objOther, char op)
{
    // NOTE: After the combination, the other object is cleared.
    if (this == &objOther) { return false; }
    tagShapeTree *newroot = new tagShapeTree;
    tagShapeTree **newleafptrs; // new leaf pointers
    int newnumofleaves; // new number of leaves
    int i; // loop variable
    int j; // loop variable
	ASSERT(op == '*' || op == '+' || op == '-');
    newroot->lchild = stRoot;
    newroot->op = op;
    newroot->rchild = objOther.stRoot;
    newroot->parent = NULL;
    stRoot->parent = newroot;
    objOther.stRoot->parent = newroot;
    objOther.stRoot = NULL;
    stRoot = newroot; // root set
    // The new tree has be constructed. Reconstruct leaf pointers.
	newnumofleaves = numofleaves + objOther.numofleaves;
    if (newnumofleaves > 0) {
        newleafptrs = new tagShapeTree *[newnumofleaves];
        for(i = 0, j = 0; i < numofleaves; i++, j++){
            newleafptrs[j] = leafptrs[i];
        }
        delete [] leafptrs;
        for(i = 0; i < objOther.numofleaves; i++, j++){
            newleafptrs[j] = objOther.leafptrs[i];
        }
        delete [] objOther.leafptrs;
        objOther.leafptrs = NULL;
        objOther.numofleaves = 0;
        leafptrs = newleafptrs; // leaf pointers set
        numofleaves = newnumofleaves; // number of leaves set
    } // else leafptrs must be NULL and numofleaves must be zero
    return true;
}

CComplexObject *CComplexObject::Disassemble(void)
{
    // Returns a newly allocated object.
    CComplexObject *resultObject = NULL; // result object
    int i; // loop variable
    int j; // loop variable
    int lnum; // number of the left subtree
    int rnum; // number of the right subtree
    tagShapeTree **newleafptrs; // new leave pointers
    tagShapeTree *lchild; // left child
    tagShapeTree *rchild; // right child
	if (numofleaves <= 1) { return NULL; }
    resultObject = new CComplexObject();
    lnum = ShapeTreeNodeCount(stRoot->lchild);
    rnum = numofleaves - lnum;
    newleafptrs = new tagShapeTree *[lnum];
    resultObject->leafptrs = new tagShapeTree *[rnum]; // result leafptrs set
    for (i = 0, j = 0; i < lnum; i++, j++) {
        newleafptrs[i] = leafptrs[j];
    }
    for(i = 0; i < rnum; i++, j++){
        resultObject->leafptrs[i] = leafptrs[j];
    } // result pointers assigned
    delete [] leafptrs;
    leafptrs = newleafptrs; // leafptrs set and pointers assigned
    numofleaves = lnum; // numofleaves set
    resultObject->numofleaves = rnum; // result numofleaves set
	// update the trees
    lchild = stRoot->lchild;
    rchild = stRoot->rchild;
    resultObject->stRoot = rchild; // result tree constructed
    delete stRoot; // node released
    stRoot = lchild; // new tree constructed
    return resultObject;
}

void CComplexObject::Draw2(CDC *pDC, const CScreenViewer &scrnVwr, POINT basePos, int width) const
{
    int i;
    for(i = 0; i < numofleaves; i++){
        leafptrs[i]->pShape->Draw2(pDC, scrnVwr, basePos, width);
    }
}

CLight CComplexObject::FilterDRLight(const CLight &light) const
{
    return CLight(light.GetRed() * drredrate, 
		light.GetGreen() * drgrnrate, 
		light.GetBlue() * drblurate);
}

CLight CComplexObject::FilterMRLight(const CLight &light) const
{
    return CLight(light.GetRed() * mrredrate, 
		light.GetGreen() * mrgrnrate, 
		light.GetBlue() * mrblurate);
}

CLight CComplexObject::FilterRfrLight(const CLight &light, double distance) const
{
    double red;
    double green;
    double blue;

	red = light.GetRed() * buffer.rfbase;
	green = light.GetGreen() * buffer.rfbase;
	blue = light.GetBlue() * buffer.rfbase;
	red *= exp(distance * 10.0 * buffer.rfredk);
	green *= exp(distance * 10.0 * buffer.rfgrnk);
	blue *= exp(distance * 10.0 * buffer.rfbluk);
    return CLight(red, green, blue);
}

bool CComplexObject::GetDRRay(const CRay &rayIn, const C3DVector &intersection, int leafIdx, CRay &rayOut) const
{
	if (buffer.max_drrate == 0.0) { return false; }
    rayOut.SetStartPoint(intersection);
    if (!IsNegativeShape(leafIdx)) {
        rayOut.SetDirection(leafptrs[leafIdx]->pShape->GetNormal(intersection));
    } else {
        rayOut.SetDirection(-leafptrs[leafIdx]->pShape->GetNormal(intersection));
    }
    return true;
}

bool CComplexObject::GetMRRay(const CRay &rayIn, const C3DVector &intersection, int leafIdx, CRay &rayOut) const
{
    C3DVector normal; // normal of the shape; vector as direction
    C3DVector ridir; // direction of the ray in
    C3DVector rodir; // direction of the ray out

	if (buffer.max_mrrate == 0.0 ) { return false; }
    normal = leafptrs[leafIdx]->pShape->GetNormal(intersection);
    if (IsNegativeShape(leafIdx)) { normal = -normal; }
    ridir = rayIn.GetDirection();
	if (ridir * normal >= -C3DVector::EPSILON)
	{
		return false;
	}
    rodir = 2.0 * (ridir - (ridir * normal) * normal) - ridir;
    rayOut.SetStartPoint(intersection);
    rayOut.SetDirection(rodir);
    return true;
}

int CComplexObject::GetRfrRay(const CRay &rayIn, const C3DVector &intersection, int leafIdx, CRay &rayOut) const
{
	if (buffer.max_drrate + buffer.max_mrrate >= 
		1 - C3DVector::EPSILON)
	{ // about to be no light - ignore it
		return 0;
	}
	// NOTE: many variables here are defined as "C3DVector" because 
	// this allows free calculation on them
    C3DVector normal; // normal of the shape
    C3DVector ridir; // direction of the ray in
    C3DVector rodir; // direction of the ray out
    double dbl1; // temporary double
    double sineval; // sine value
    double refang; // refraction angle
    C3DVector paradir; // direction parallel to the surface of the shape
	int cond; // condition: in or out

    normal = leafptrs[leafIdx]->pShape->GetNormal(intersection); // the model of "normal" is one
    if (IsNegativeShape(leafIdx)) { normal = -normal; }
    ridir = rayIn.GetDirection(); // the model of "ridir" is one
    paradir = ridir - (ridir * normal) * normal;
	paradir.UnitizeVector(); // the model of "paradir" is one
	dbl1 = ridir * normal;
	if (fabs(dbl1) < C3DVector::EPSILON) {
		return 0;
	}
    sineval = ridir.XMultiply(normal).GetModel();
	// dbl1 > 0 means the ray is on the opposite side to the normal
    if (dbl1 > 0)
	{
		sineval *= ior;
		cond = -1; // out
	}
	else
	{
		sineval /= ior;
		cond = 1; // in
	}
    if (sineval > 1.0)
	{ // full reflection is not proceeded
		return 0;
	}
    refang = asin(sineval);
	if (cond < 0) {
		rodir = sineval * paradir + 
			cos(refang) * normal;
	} else {
		rodir = sineval * paradir - 
			cos(refang) * normal;
	}
	rodir.UnitizeVector();
    rayOut.SetStartPoint(intersection);
    rayOut.SetDirection(rodir);
    return cond;
}

bool CComplexObject::GetIntersection(int leafIdx, const CRay &ray, C3DVector &intersection) const
{
	return leafptrs[leafIdx]->pShape->CalculateIntersection(
		ray, intersection, IsNegativeShape(leafIdx));
}

int CComplexObject::GetShapeCount(void) const
{
    return numofleaves;
}

CString CComplexObject::GetShapeType(void) const
{
    if(IsSimpleObject()){
        return stRoot->pShape->GetShapeType();
    }else{
        return CString("");
    }
}

CShape *CComplexObject::GetShapePtr(void)
{
    if(IsSimpleObject()){
        return stRoot->pShape;
    }else{
        return NULL;
    }
}

bool CComplexObject::IsValidIntersection(const C3DVector &intersection) const
{
    if(stRoot != NULL){
        return IsOnObject(stRoot, intersection);
    }else{
        return false;
    }
}

bool CComplexObject::IsInObject(tagShapeTree *root, const C3DVector &pnt) const
{
	ASSERT(root != NULL);
    if(root->lchild != NULL){
        switch(root->op){
        case '*':
            if(IsInObject(root->lchild, pnt) && IsInObject(root->rchild, pnt)){
                return true;
            }
            break;
        case '+':
            if(IsInObject(root->lchild, pnt) || IsInObject(root->rchild, pnt)){
                return true;
            }
            break;
        case '-':
            if(IsInObject(root->lchild, pnt) && !IsInOrOnObject(root->rchild, pnt)){
                return true;
            }
            break;
        }
        return false;
    }else{
        return root->pShape->IsInObject(pnt);
    }
}

bool CComplexObject::IsInOrOnObject(tagShapeTree *root, const C3DVector &pnt) const
{
	ASSERT(root != NULL);
    if(root->lchild != NULL){
        switch(root->op){
        case '*':
            if(IsInOrOnObject(root->lchild, pnt) && IsInOrOnObject(root->rchild, pnt)){
                return true;
            }
            break;
        case '+':
            if(IsInOrOnObject(root->lchild, pnt) || IsInOrOnObject(root->rchild, pnt)){
                return true;
            }
            break;
        case '-':
            if(IsInOrOnObject(root->lchild, pnt) && !IsInObject(root->rchild, pnt)){
                return true;
            }
            break;
        }
        return false;
    }else{
        return root->pShape->IsInOrOnObject(pnt);
    }
}

bool CComplexObject::IsOnObject(tagShapeTree *root, const C3DVector &pnt) const
{
	ASSERT(root != NULL);
    if(root->lchild != NULL){
        switch(root->op){
        case '*':
            if (IsInOrOnObject(root->lchild, pnt) && IsOnObject(root->rchild, pnt) ||
                IsOnObject(root->lchild, pnt) && IsInOrOnObject(root->rchild, pnt)) 
			{
                return true;
            }
            break;
        case '+':
            if (IsOnObject(root->lchild, pnt) && !IsInObject(root->rchild, pnt) ||
                !IsInObject(root->lchild, pnt) && IsOnObject(root->rchild, pnt)) 
			{
                return true;
            }
            break;
        case '-':
            if (IsOnObject(root->lchild, pnt) && !IsInObject(root->rchild, pnt) ||
                IsInOrOnObject(root->lchild, pnt) && IsOnObject(root->rchild, pnt)) 
			{
                return true;
            }
            break;
        }
        return false;
    }else{
        return root->pShape->IsOnObject(pnt);
    }
}

bool CComplexObject::IsNegativeShape(int leafIdx) const
{
    bool neg = false; // negative flag
    tagShapeTree *stleaf; // shape tree leaf

    stleaf = leafptrs[leafIdx];
    while(stleaf->parent != NULL){
        if(stleaf->parent->op == '-'){
            if(stleaf == stleaf->parent->rchild){
                neg = !neg; // a reverse
            }
        }
        stleaf = stleaf->parent;
    }
    return neg;
}

bool CComplexObject::IsSimpleObject(void) const
{
    if(stRoot != NULL){
        if(stRoot->lchild == NULL){
            return true;
        }
    }
    return false;
}

void CComplexObject::ReleaseShapeTree(tagShapeTree *root)
{
    if(root != NULL){
        ReleaseShapeTree(root->lchild);
        ReleaseShapeTree(root->rchild);
        if(root->lchild == NULL) delete root->pShape; // pShape is used for leaves
        delete root;
    }
}

int CComplexObject::ShapeTreeNodeCount(tagShapeTree *root)
{
    int number; // number of nodes
    if (root != NULL) {
        if (root->lchild == NULL) { return 1; }
        number = ShapeTreeNodeCount(root->lchild);
        number += ShapeTreeNodeCount(root->rchild);
    }else{
        return 0;
    }
    return number;
}
