// 3DVector.cpp: implementation of the C3DVector class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <math.h>
#include "GraphicsConclusion.h"
#include "3DVector.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

const double C3DVector::EPSILON = 1E-7;

C3DVector::C3DVector()
{
	// There are no dynamic allocated elements in the object
	// So I can use Clear() to do the initialization
	Clear();
}

C3DVector::C3DVector(const C3DVector &vctSource)
{
    vdata.x = vctSource.vdata.x;
    vdata.y = vctSource.vdata.y;
    vdata.z = vctSource.vdata.z;
    vdata.t = vctSource.vdata.t;
}

C3DVector::C3DVector(double dblX, double dblY, double dblZ)
{
    vdata.x = dblX;
    vdata.y = dblY;
    vdata.z = dblZ;
    vdata.t = 1.0;
}

C3DVector::~C3DVector()
{
}

void C3DVector::Clear(void)
{
    vdata.x = 0.0;
    vdata.y = 0.0;
    vdata.z = 0.0;
    vdata.t = 1.0;
}

double C3DVector::DistanceTo(const C3DVector &vctOther) const
{
    double dblDeltaX;   // difference of the vectors' x coordinate
    double dblDeltaY;   // difference of the vectors' y coordinate
    double dblDeltaZ;   // difference of the vectors' x coordinate

    dblDeltaX = vctOther.vdata.x - vdata.x;
    dblDeltaY = vctOther.vdata.y - vdata.y;
    dblDeltaZ = vctOther.vdata.z - vdata.z;
    return sqrt(dblDeltaX * dblDeltaX + dblDeltaY * dblDeltaY + dblDeltaZ * dblDeltaZ);
}

void C3DVector::Get(double &dblX, double &dblY, double &dblZ) const
{
    dblX = vdata.x;
    dblY = vdata.y;
    dblZ = vdata.z;
}

double C3DVector::GetArrayItem(int intAxis) const
{
    ASSERT(0 <= intAxis && intAxis <= 3);
    return adata[intAxis];
}

double C3DVector::GetModel(void) const
{
    return sqrt(vdata.x * vdata.x + vdata.y * vdata.y + vdata.z * vdata.z);
}

double C3DVector::GetX(void) const
{
    return vdata.x;
}

double C3DVector::GetY(void) const
{
    return vdata.y;
}

double C3DVector::GetZ(void) const
{
    return vdata.z;
}

double C3DVector::operator*(const C3DVector &vctOther) const
{
    double dblResult;

    dblResult = vdata.x * vctOther.vdata.x + vdata.y * vctOther.vdata.y + vdata.z * vctOther.vdata.z;
    return dblResult;
}

C3DVector operator*(double dblScale, const C3DVector &vctSource)
{
    C3DVector vctDestination;
    
    // NOTE: t-value will be lost after operation
    vctDestination.vdata.x = dblScale * vctSource.vdata.x;
    vctDestination.vdata.y = dblScale * vctSource.vdata.y;
    vctDestination.vdata.z = dblScale * vctSource.vdata.z;
    vctDestination.vdata.t = 1.0;
    return vctDestination;
}

C3DVector C3DVector::operator+(const C3DVector &vctOther) const
{
    C3DVector vctDestination;

    // NOTE: t-value will be lost after operation
    vctDestination.vdata.x = vdata.x + vctOther.vdata.x;
    vctDestination.vdata.y = vdata.y + vctOther.vdata.y;
    vctDestination.vdata.z = vdata.z + vctOther.vdata.z;
    vctDestination.vdata.t = 1.0;
    return vctDestination;
}

C3DVector C3DVector::operator-(void) const
{
    return C3DVector(-vdata.x, -vdata.y, -vdata.z);
}

C3DVector C3DVector::operator-(const C3DVector &vctOther) const
{
    C3DVector vctDestination;

    // NOTE: t-value will be lost after operation
    vctDestination.vdata.x = vdata.x - vctOther.vdata.x;
    vctDestination.vdata.y = vdata.y - vctOther.vdata.y;
    vctDestination.vdata.z = vdata.z - vctOther.vdata.z;
    vctDestination.vdata.t = 1.0;
    return vctDestination;
}

C3DVector &C3DVector::operator-=(const C3DVector &vctOther)
{
	// NOTE: t-value will be lost after operation
    if (&vctOther == this) { return *this; }
    vdata.x -= vctOther.vdata.x;
    vdata.y -= vctOther.vdata.y;
    vdata.z -= vctOther.vdata.z;
	vdata.t = 1.0;
    return *this;
}

C3DVector &C3DVector::operator+=(const C3DVector &vctOther)
{
	// NOTE: t-value will be lost after operation
    if (&vctOther == this) { return *this; }
    vdata.x += vctOther.vdata.x;
    vdata.y += vctOther.vdata.y;
    vdata.z += vctOther.vdata.z;
	vdata.t = 1.0;
    return *this;
}

C3DVector &C3DVector::operator=(const C3DVector &vctSource)
{
    if(this == &vctSource)
    {   // duplicating
        return *this;
    }
    vdata.x = vctSource.vdata.x;
    vdata.y = vctSource.vdata.y;
    vdata.z = vctSource.vdata.z;
    vdata.t = vctSource.vdata.t;
    return *this;
}

bool C3DVector::operator==(const C3DVector &vctOther) const
{
    // NOTE: unstandard vectors may not be correctly compared
    if (vdata.x == vctOther.vdata.x && 
		vdata.y == vctOther.vdata.y && 
		vdata.z == vctOther.vdata.z) 
	{
        return true;
    } else {
        return false;
    }
}

void C3DVector::Set(const C3DVector &vctSource)
{
	operator=(vctSource); // for simplicity and consistency
}

void C3DVector::Set(double dblX, double dblY, double dblZ)
{
    vdata.x = dblX;
    vdata.y = dblY;
    vdata.z = dblZ;
    vdata.t = 1.0;
}

void C3DVector::SetX(double dblX)
{
    vdata.x = dblX;
}

void C3DVector::SetY(double dblY)
{
    vdata.y = dblY;
}

void C3DVector::SetZ(double dblZ)
{
	vdata.z = dblZ;
}

void C3DVector::StandardizePoint(void)
{
    double dblScale;

    // This function standardizes the point according to the t-value
	ASSERT(vdata.t != 0.0);
    dblScale = 1.0 / vdata.t;
    vdata.x = vdata.x * dblScale;
    vdata.y = vdata.y * dblScale;
    vdata.z = vdata.z * dblScale;
    vdata.t = 1.0;
}

void C3DVector::UnitizeVector(void)
{
    double dblBuffer;

    dblBuffer = sqrt(vdata.x * vdata.x + vdata.y * vdata.y + vdata.z * vdata.z);
    ASSERT(dblBuffer != 0.0);
    dblBuffer = 1.0 / dblBuffer;
    vdata.x = vdata.x * dblBuffer;
    vdata.y = vdata.y * dblBuffer;
    vdata.z = vdata.z * dblBuffer;
    vdata.t = 1.0;
}

C3DVector C3DVector::XMultiply(const C3DVector &vctOther) const
{
    C3DVector vctDestination;

    // NOTE: t-value will be lost after operation
    vctDestination.vdata.x = vdata.y * vctOther.vdata.z - vdata.z * vctOther.vdata.y;
    vctDestination.vdata.y = vdata.z * vctOther.vdata.x - vdata.x * vctOther.vdata.z;
    vctDestination.vdata.z = vdata.x * vctOther.vdata.y - vdata.y * vctOther.vdata.x;
    vctDestination.vdata.t = 1.0;
    return vctDestination;
}
