// 3DMatrix.cpp: implementation of the C3DMatrix class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GraphicsConclusion.h"
#include "3DMatrix.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

C3DMatrix::C3DMatrix(const double dblArray[])
{
    int i;  // loop variable

    for(i = 0; i < 16; i++)
    {
        data[i] = dblArray[i]; // for speed
    }
}

C3DMatrix::C3DMatrix(const C3DMatrix &mtxSource)
{
    int i;  // loop variable

    for(i = 0; i < 16; i++)
    {
        data[i] = mtxSource.data[i];
    }
}

C3DMatrix::C3DMatrix()
{
	// NOTE: There are no dynamically allocated elements in the object
	// So I can use Clear() to do the initialization
	Clear(); // for simplicity and consistency
}

C3DMatrix::~C3DMatrix()
{
}

void C3DMatrix::Clear(void)
{
    int i;  // loop variable

    for(i = 0; i < 16; i++)
    {
        data[i] = 0;
    }
}

void C3DMatrix::Get(double dblArray[]) const
{
    ASSERT(dblArray != NULL);
    int i;  // loop variable

    for(i = 0; i < 16; i++)
    {
        dblArray[i] = data[i];
    }
}

bool C3DMatrix::Inverse( void )
{
    C3DMatrix mtxInversed;  // the inversed matrix
    int i;  // loop variable
    int j;  // loop variable
    double double1;     // temporary value
    int intLineNum; // line number
    int intStep;    // step of treatment
    double dblTempData[16];    // temporary data

	mtxInversed.SetToUnit( false );
    // copy the data
    for( i = 0; i < 16; i++ )
    {
        dblTempData[i] = data[i];
    }
    // stepped treatment
    for( intStep = 0; intStep < 4; intStep++ )
    {
        // first find a line whose first item is non-zero
        for( i = intStep; i < 4; i++ )
        {
            if( dblTempData[i * 4 + intStep] != 0.0 )
            {
                break;
            }
        }
        if( i == 4 )
        {   // failed
            return false;
        }
        intLineNum = i;
        // swap the line with the first line
        // operate on the original matrix
        for( i = intStep; i < 4; i++ ) // here i only need to start from intStep
        { // because elements before it are zero
            double1 = dblTempData[intStep * 4 + i];
            dblTempData[intStep * 4 + i] = dblTempData[intLineNum * 4 + i];
            dblTempData[intLineNum * 4 + i] = double1;
        }
        // synchronize the target matrix
        for(i = 0; i < 4; i++) // here i should start from 0
        {
            double1 = mtxInversed.data[intStep * 4 + i];
            mtxInversed.data[intStep * 4 + i] = mtxInversed.data[intLineNum * 4 + i];
            mtxInversed.data[intLineNum * 4 + i] = double1;
        }
        // for each line below, eliminate the first number
        for(i = intStep + 1; i < 4; i++)
        {
            // operate on the original matrix
			// dblTempData[intStep * 4 + intStep] mustn't be zero
            double1 = dblTempData[i * 4 + intStep] / dblTempData[intStep * 4 + intStep];
            dblTempData[i * 4 + intStep] = 0.0;
            for(j = intStep + 1; j < 4; j++)
            {
                dblTempData[i * 4 + j] -= dblTempData[intStep * 4 + j] * double1;
            }
            // synchronize the target matrix
            for(j = 0; j < 4; j++)
            {
                mtxInversed.data[i * 4 + j] -= mtxInversed.data[intStep * 4 + j] * double1;
            }
        }
    }
    // eleminate the rest numbers
    for(intStep = 3; intStep >= 0; intStep--)
    {
        for(i = intStep - 1; i >= 0; i--)
        {
            // operate on the original matrix
			// dblTempData[intStep * 4 + intStep] mustn't be zero
            double1 = dblTempData[i * 4 + intStep] / dblTempData[intStep * 4 + intStep];
            dblTempData[i * 4 + intStep] = 0.0;
            // synchronize the target matrix
            for(j = 0; j < 4; j++)
            {
                // a line minus another
                mtxInversed.data[i * 4 + j] -= mtxInversed.data[intStep * 4 + j] * double1;
            }
        }
    }
    // unitize the matrix
    for(i = 0; i < 4; i++)
    {
        // operate on the original matrix
        double1 = dblTempData[i * 4 + i];
        dblTempData[i * 4 + i] = 1;
        //synchronize the target matrix
        for(j = 0; j < 4; j++)
        {
			if (double1 == 0.0) { return false; }
            mtxInversed.data[i * 4 + j] /= double1;
        }
    }
    // copy the data
    for(i = 0; i < 16; i++)
    {
        data[i] = mtxInversed.data[i];
    }
    return true; // inverse successful
}

C3DMatrix C3DMatrix::MoveMatrix(double dblIncrX, double dblIncrY, double dblIncrZ)
{
    C3DMatrix mtxResult;

    mtxResult.SetToUnit(false);
    mtxResult.data[3 * 4 + 0] = dblIncrX;
    mtxResult.data[3 * 4 + 1] = dblIncrY;
    mtxResult.data[3 * 4 + 2] = dblIncrZ;
    return mtxResult;
}

C3DVector operator*(const C3DVector &vector1, const C3DMatrix &matrix2)
{
    C3DVector vctResult;
    int s;  // left x also right y
    int rx; // right x

    vctResult.vdata.t = 0.0; // special: let all numbers be zero
    for(rx = 0; rx < 4; rx++)
    {
        for(s = 0; s < 4; s++)
        {
            vctResult.adata[rx] += vector1.adata[s] * matrix2.data[s * 4 + rx];
        }
    }
    return vctResult;
}

C3DVector &operator*=(C3DVector &vector1, const C3DMatrix &matrix2)
{
	vector1 = operator*(vector1, matrix2); // for simplicity and consistency but ignoring speed
    return vector1;
}

C3DMatrix C3DMatrix::operator*(const C3DMatrix &mtxOther) const
{
    C3DMatrix mtxResult;
    int ly; // left y
    int s;  // left x also right y
    int rx; // right x

    for(ly = 0; ly < 4; ly++)
    {
        for(rx = 0; rx < 4; rx++)
        {   // mtxResult has been reset to zeroes.
            for(s = 0; s < 4; s++)
            {
                mtxResult.data[ly * 4 + rx] += data[ly * 4 + s] * mtxOther.data[s * 4 + rx];
            }
        }
    }
    return mtxResult;
}

C3DMatrix &C3DMatrix::operator=(const C3DMatrix &mtxSource)
{
    int i;  // loop variable

    if(this == &mtxSource)
    {   // duplicating
        return *this;
    }
    for(i = 0; i < 16; i++)
    {
        data[i] = mtxSource.data[i];
    }
    return *this;
}

C3DMatrix C3DMatrix::RotateMatrix(double dblTheta, int intAxis)
{
    C3DMatrix mtxResult;
    int intAxis1;   // the first modified axis
    int intAxis2;   // the second modified axis

    ASSERT(0 <= intAxis && intAxis <= 2);
	// x' = x cos  - y sin  [-]
	// y' = x sin  + y cos  [-]
    mtxResult.SetToUnit(FALSE);
    intAxis1 = (intAxis + 1) % 3;
    intAxis2 = (intAxis + 2) % 3;
    mtxResult.data[intAxis1 * 4 + intAxis1] = cos(dblTheta);
    mtxResult.data[intAxis2 * 4 + intAxis1] = -sin(dblTheta);
    mtxResult.data[intAxis1 * 4 + intAxis2] = sin(dblTheta);
    mtxResult.data[intAxis2 * 4 + intAxis2] = cos(dblTheta);
    return mtxResult;
}

void C3DMatrix::Set(const double dblArray[])
{
    ASSERT(dblArray != NULL);
    int i;  // loop variable

    for(i = 0; i < 16; i++)
    {
        data[i] = dblArray[i];
    }
}

void C3DMatrix::SetItem(int intNumber, double dblValue)
{
    ASSERT(0 <= intNumber && intNumber <= 15);
    data[intNumber] = dblValue;
}

void C3DMatrix::SetToUnit(bool blnClear)
{
    if(blnClear) { Clear(); }
    data[0 * 4 + 0] = 1.0;
    data[1 * 4 + 1] = 1.0;
    data[2 * 4 + 2] = 1.0;
    data[3 * 4 + 3] = 1.0;
}

C3DMatrix C3DMatrix::ShiftMatrix(double dblScaleA, double dblScaleB, int intAxis)
{
    C3DMatrix mtxResult;
    int intAxisA;  // scale A axis
    int intAxisB;  // scale B axis

    // ShiftMatrix function shifts coordinates on axis A and axis B by the value of the coordinate on axis intAxis
    mtxResult.SetToUnit(false);
    intAxisA = (intAxis + 1) % 3;
    intAxisB = (intAxis + 2) % 3;
    mtxResult.data[intAxis * 4 + intAxisA] = dblScaleA;
    mtxResult.data[intAxis * 4 + intAxisB] = dblScaleB;
    return mtxResult;
}

C3DMatrix C3DMatrix::SizingMatrix(double dblScaleX, double dblScaleY, double dblScaleZ)
{
    C3DMatrix mtxResult;

    mtxResult.data[0 * 4 + 0] = dblScaleX;
    mtxResult.data[1 * 4 + 1] = dblScaleY;
    mtxResult.data[2 * 4 + 2] = dblScaleZ;
    mtxResult.data[3 * 4 + 3] = 1.0;
    return mtxResult;
}
