/* ==== parser.c ==== */
/* simplistic command file parser for initial prototype version of patel */

#include <stdio.h>
#include <ctype.h>
#include "parser.h" /* structure def */

#define scream(arg) buffer->type=arg; return(-1);

int getCmd(modeDesc_t *mode,commandPacket_t *buffer) {
   /* read input until we find a line beginning with a command;
      CR and LF are both considered newline symbols, whitespace is ignored.
      If we find a comment line (starts with a #), go into "comment"
      mode until we find a CR or LF character. */
   int startline=1,comment=0,argcounter=0;
   int sym,nybble;
   varDesc_t varSpec;
   unsigned long argaccum=0;
   char cmdchar=' ';
   while(cmdchar==' ') {
      sym=getchar();
      switch(sym) {
         case EOF:
                scream(PARSED_EOF)
                /* break?! */
         case '#': /* enter comment mode */
                if(!startline) {
                   /* ERROR! comments may only start at START of a line */
                   scream(PARSED_BADCOM)
                }
                comment=1;
                while(comment) {
                   /* ignore everything until we find a newline or EOF */
                   sym=getchar();
                   if(sym==EOF) {
                     /* comment ends here if the file does... */
                     scream(PARSED_EOF)
                   }
                   if(sym=='\n' || sym=='\r') {
                      comment=0;
                      startline=1;
                   }
                }
                break;
         case '\n':/* end of line, just like....*/
         case '\r':startline=1;break;
         case ' ': /*whitespace, normally ignored */
         case '\t': /* or others? */
                   startline=0;break;
         default: if(!startline) {
                     /* ERROR! command characters may only appear
                        at *START* of a line */
                     scream(PARSED_BADCMD)
                  }
                  cmdchar=sym;
      }
   } /* end of loop */
   /* we now have a command character, right? */
   if(!isalpha(cmdchar)) {
     /* ERROR! not valid command character */
     scream(PARSED_BADCMD)
   }
   if(isupper(cmdchar)) {
     /* no argument needed */
     /* Umm... As arguments might be in *hex*, should we check to see if
       the following character is a newline/whitespace/EOF (as it should be)?*/
     sym=getchar();
     if(EOF==sym || isspace(sym)) {
       buffer->cmd=cmdchar;
       buffer->type=PARSED_CMD;
       return(0);
     } else {
       /* ERROR! upper-case command followed by other characters! */
       scream(PARSED_BUSYLINE)
     }
   }
   /* must have a lower-case command, which needs some arg */
   /* Now, question is... how many chars of argument data will follow?
      We might have variable amounts, depending on command type! */
   varSpec=mode->var[cmdchar-'a'];
   argcounter=varSpec.bytes;
   while(sym=getchar(),(EOF!=sym && !isspace(sym)) ) {
     if(!isxdigit(sym) || islower(sym) ) {
        /* ERROR! Non-hex or lower-case char in argument */
        scream(PARSED_BADHEX)
     }
     nybble=(sym<'A')?(sym-'0'):(sym-('A'-10));
     argaccum<<=4; argaccum+=nybble;

     argcounter--;
     if(argcounter<0) {
       /* ERROR! Reading too many chars! */
       scream(PARSED_BADARG)
     }
   }
   if(argcounter!=0) {
      /* ERROR! Wrong number of chars read in argument! */
      scream(PARSED_BADARG)
   }
   if(argaccum<varSpec.minval || argaccum>varSpec.maxval) {
      /* ERROR! Argument out of range! */
      scream(PARSED_OUTOFRANGE)
   }
   /* A variable-setting command at last */
   buffer->cmd=cmdchar;
   buffer->type=PARSED_VARSET;
   buffer->vars[cmdchar-'a']=argaccum;
   /*buffer->val=argaccum; */
   return(0);
}

void displayError(cmdTypeEnum error) {
   char *errorTable[]={
      "Variable parsed- not an error",
      "Command parsed- not an error",
      "EOF found- not really an error",
      "Malformed comment found",
      "Malformed command found",
      "Extra characters found on line after command",
      "Invalid characters in argument (uppercase hexadecimal required)",
      "Bad length of argument",
      "Variable found out of permitted range"
   };
   fprintf(stderr,"ERROR parsing commands:\n %s\n",errorTable[error]);
}

/* === end of parser.c === */
