/* Copyright (C) 1999-2001,2004 by Peter Eastman

   This program is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT ANY 
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
   PARTICULAR PURPOSE.  See the GNU General Public License for more details. */

package artofillusion.ui;

import artofillusion.*;
import buoy.event.*;
import buoy.widget.*;
import java.awt.*;
import java.awt.image.*;
import java.net.*;

/** EditingTool is the superclass of tools for editing objects or scenes.  An EditingTool
    has an image which appears in a tool palette, allowing the tool to be selected.  When
    selected, the editing tool responds to events in the scene or object viewer. */

public abstract class EditingTool
{
  protected EditingWindow theWindow;
  protected BFrame theFrame;
  
  public EditingTool(EditingWindow win)
  {
    theWindow = win;
    if (win != null)
      theFrame = win.getFrame();
  }
  
  /** Get the EditingWindow to which this tool belongs. */
  
  public EditingWindow getWindow()
  {
    return theWindow;
  }
  
  /** Try to load image named name. On failure, return fallback instead */
  protected Image loadImage(String name, Image fallback)
  {
    ImageProducer ip;
    Image im;
    
    if (theFrame == null)
      return fallback;
    try
    {
      ip = (ImageProducer) getClass().getResource("/artofillusion/Icons/"+name).getContent();
    }
    catch (Exception e)
    {
      new BStandardDialog("", Translate.text("errorLoadingIcon", name), BStandardDialog.ERROR).showMessageDialog(theFrame);
      return fallback;
    }
    im = theFrame.getComponent().createImage(ip);
    MediaTracker mt = new MediaTracker(theFrame.getComponent());
    mt.addImage(im, 0);
    try
    {
      mt.waitForID(0);
    }
    catch (InterruptedException ex)
    {
      new BStandardDialog("", Translate.text("errorLoadingIcon", name), BStandardDialog.ERROR).showMessageDialog(theFrame);
      return fallback;
    }
    return im;
  }
  
  /** Try to load image named name. */
  protected Image loadImage(String name)
  {
    return loadImage(name, null);
  }
  
  protected Image createSelectedImage()
  {
    ImageProducer ip = new FilteredImageSource(getIcon().getSource(), new HighlightFilter());
    Image im = theFrame.getComponent().createImage(ip);
    MediaTracker mt = new MediaTracker(theFrame.getComponent());
    mt.addImage(im, 0);
    try
    {
      mt.waitForID(0);
    }
    catch (InterruptedException ex)
    {
      return null;
    }
    return im;
  }
  
  /** Get the icon to display for this tool. */
  
  public abstract Image getIcon();

  /** Get the icon to display for this tool when it is selected. */
  
  public abstract Image getSelectedIcon();
  
  /** Get the tool tip text to display for this tool (or null if it does not have a tool tip). */
  
  public String getToolTipText()
  {
    return null;
  }
  
  public static final int ALL_CLICKS = 0;
  public static final int OBJECT_CLICKS = 1;
  public static final int HANDLE_CLICKS = 2;

  public int whichClicks()
  {
    return ALL_CLICKS;
  }
  
  public boolean hilightSelection()
  {
    return true;
  }
  
  public void drawOverlay(Graphics g, ViewerCanvas view)
  {
  }
  
  public void mousePressed(WidgetMouseEvent e, ViewerCanvas view)
  {
  }

  public void mousePressedOnObject(WidgetMouseEvent e, ViewerCanvas view, int obj)
  {
  }

  public void mousePressedOnHandle(WidgetMouseEvent e, ViewerCanvas view, int obj, int handle)
  {
  }

  public void mouseReleased(WidgetMouseEvent e, ViewerCanvas view)
  {
  }
  
  public void mouseDragged(WidgetMouseEvent e, ViewerCanvas view)
  {
  }
  
  public void mouseMoved(WidgetMouseEvent e, ViewerCanvas view)
  {
  }
  
  public void keyPressed(KeyPressedEvent e, ViewerCanvas view)
  {
  }

  public void activate()
  {
    theWindow.setTool(this);
  }
  
  public void deactivate()
  {
  }

  public void iconDoubleClicked()
  {
  }
  
  class HighlightFilter extends RGBImageFilter
  {
    public HighlightFilter()
    {
      super();
      canFilterIndexColorModel = true;
    }
    
    public int filterRGB(int x, int y, int rgb)
    {
      int r = (rgb>>16) & 0xFF;
      int g = (rgb>>8) & 0xFF;
      int b = rgb & 0xFF;
      r = (r+g+b)/3;
      g = b = g/3;
      return (rgb & 0xFF000000) + (r<<16) + (g<<8) + b;
    }
  }
}