/* Copyright (C) 2002 by Peter Eastman

   This program is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT ANY 
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
   PARTICULAR PURPOSE.  See the GNU General Public License for more details. */


/** This interface represents a script which can create a new state for
  a new time from a previous state and the initial state.
  The type of these states is not specified.
 
  @author Michael Butscher
*/


package artofillusion.script;

import java.io.*;
import artofillusion.*;

public interface IterAnimationScript
{
  
  /** Creates the initial state from information in the current scene
    and/or displays a dialog to allow configuring the script.
   
    @return the initial state or null if the state could not be created.
            in the latter case the script should have displayed a dialog
            to inform the user about the reason of the failure.
  */
  public Serializable createInitialState(LayoutWindow window);
  

  /** Take <CODE>initial</CODE> state and one <CODE>previous</CODE> state
    and create a new state which is appropriate if time goes the
    <CODE>timestep</CODE> interval further.
    
    <P>The <CODE>timestep</CODE> may be negative iff
    {@link #isBackwardTimeAllowed()} returns
    true for the same combination of parameters.<BR>
    <CODE>initial</CODE> and <CODE>previous</CODE> may be identical (if this
    is the first step).<BR>
    The <CODE>scene</CODE> must not be modified by this method.
    
    <P>The <CODE>initial</CODE> state is given because it may contain
    additional configuration data which does not need to be stored in
    every state object.
    
    <P>The current implementation always uses a timestep of 1/fps or
    -1/fps where fps are the frames per second of the current scene. (TODO(MB)!)
   
    @return a new (or recycled) state object or null if no state could be
            created.
  */
  public Object executeTimeStep(Serializable initial, Object previous,
      double timestep, Scene scene);

  
  /** Returns true iff a new state can be created from an old state which
    lies in the future of the new one.<BR>
    This is only needed in interactive mode. During animation rendering
    time always goes forward.<BR>
    The parameters have the same meaning as for {@link #executeTimeStep()}.
  */
  public boolean isBackwardTimeAllowed(Serializable initial, Object previous,
      double timestep, Scene scene);
  
  
  /** Returns the minimal timestep allowed for these states.
    The parameters have the same meaning as for {@link #executeTimeStep()}.
    <P>Ignored by current implementation!  (TODO(MB)!)
  */
  public double getMinimalTimeStep(Serializable initial, Object previous,
      Scene scene);

  
  /** Returns the maximal timestep allowed for these states.
    The parameters have the same meaning as for {@link #executeTimeStep()}.
    <P>Ignored by current implementation!  (TODO(MB)!)
  */
  public double getMaximalTimeStep(Serializable initial, Object previous,
      Scene scene);
      
  
  /** Realizes the state, meaning to set the scene according to the state
    information in <CODE>state</CODE> and <CODE>initial</CODE>.
  */
  public void realizeState(Serializable initial, Object state, Scene scene);

}


