//
// BNBOT.CPP
//
// Implementation of a virtual base class for Battle.net bots.
//
// by Scott Coleman 3/31/98
//



#include "pch.h"

#include "bnbot.h"
#include <assert.h>



// event IDs
#define EID_SHOWUSER            1001
#define EID_JOIN                1002
#define EID_LEAVE               1003
#define EID_WHISPER             1004
#define EID_TALK                1005
#define EID_BROADCAST           1006
#define EID_CHANNEL             1007
#define EID_USERFLAGS           1009
#define EID_WHISPERSENT         1010
#define EID_CHANNELFULL         1013
#define EID_CHANNELDOESNOTEXIST 1014
#define EID_CHANNELRESTRICTED   1015
#define EID_INFO                1018
#define EID_ERROR               1019
#define EID_EMOTE               1023
#define EID_UNIQUENAME          2010


const unsigned int SOCKET_MESSAGE = WM_USER + 100;

//============================================================================
BnBot::BnBot() {

	log = 0;
	m_nBufLen=0;
	m_nBufPos=0;
  s = INVALID_SOCKET;
  nServerPort = 6112;
  szLoginName[0] = '\0';
  szUniqueName[0] = '\0';
  szPassword[0] = '\0';
  szServerAddr[0] = '\0';
  szHomeChannel[0] = '\0';
  szCurrentChannel[0] = '\0';
}


//============================================================================
BnBot::~BnBot() {

  LogClose();
}


//============================================================================
void BnBot::SetLogonInfo(char *szUserName,
                         char *szUserPass,
                         char *szServer,
                         short nPort) {

  strcpy(szLoginName, szUserName);
  strcpy(szUniqueName, szUserName);
  strcpy(szPassword, szUserPass);
  strcpy(szServerAddr, szServer);
  nServerPort = nPort;
}


//============================================================================
void BnBot::SetHomeChannel(char *szChannelName) {

  strcpy(szHomeChannel, szChannelName);
}


//============================================================================
int BnBot::Connect(HWND hWnd) {
  struct sockaddr_in name;
  struct hostent *hp;

  memset(&name, '\0', sizeof(name));
  name.sin_family = AF_INET;
  name.sin_port = htons(nServerPort);

  // if this is a hostname and not an IP address, resolve it
  char *p = szServerAddr;
  while (*p && (isdigit(*p) || (*p == '.'))) {
    p++;
  }

  // non-digit found - assume hostname
  if (*p) {
    hp = gethostbyname(szServerAddr);
    if (hp == 0)
      return 0;         // can't resolve hostname
    memcpy(&name.sin_addr, hp->h_addr, hp->h_length);
  }
  else {
    name.sin_addr.s_addr = inet_addr(szServerAddr);
  }

  s = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (s == INVALID_SOCKET) {
    return 0;
  }

  if (connect(s, (struct sockaddr *)&name, sizeof(name))) {
    return 0;
  }

  // OK kids: this is how to do WinSock PROPERLY. Windows is an async
  // environment, lets treat it appropriately. Now, instead of continually
  // checking to see if there's data, we let Windows let us know when it's
  // time to read, and when it's time to re-connect. Simple:
  if (WSAAsyncSelect(s, hWnd, SOCKET_MESSAGE, FD_READ|FD_CLOSE))
  {
	  return 0;
  }

  return 1;
}


//============================================================================
int BnBot::Logon() {

  // send ^C, ^D, username, and password
  // ^C gets the server's attention for logon; ^D turns off ECHO
  Send("%c%c%s\r\n%s\r\n", 0x03, 0x04, szLoginName, szPassword);

  return 1;
}


//============================================================================
void BnBot::Disconnect() {

  if (s == INVALID_SOCKET)
    return;

  closesocket(s);
  s = INVALID_SOCKET;
}


//============================================================================
int BnBot::Reconnect(HWND hWnd) {
	m_nBufLen=0;
	m_nBufPos=0;

  LogWrite("Attempting to connect...");
  for (int tries=0; tries < 10; ++tries) {
    if (!Connect(hWnd)) {
      LogWrite("Connect() failed.");
      Sleep(5000);
      continue;
    }
    if (!Logon()) {
      LogWrite("Logon() failed.");
      continue;
    }
    Sleep(1000);
    Send("/join %s\r\n", szHomeChannel);
    return 1;
  }
  LogWrite("Unable to log on after %d tries - giving up.", tries);
  return 0;
}


//============================================================================
int __cdecl BnBot::Send(char *lpszFmt, ...) {
  char szOutStr[MAXTEXTLENGTH];
  va_list argptr;

  va_start(argptr, lpszFmt);
  vsprintf(szOutStr, lpszFmt, argptr);
  va_end(argptr);

  if (send(s, szOutStr, strlen(szOutStr), 0) < 0)
    return 0;

  return 1;
}


//============================================================================
int BnBot::ParseEvent(char *pszEvent,
                      int *pnEventId,
                      char *pszSpeaker,
                      u_long *puFlags,
                      char *pszEventText) {

  if (!pszEvent || !pnEventId || !pszSpeaker || !puFlags || !pszEventText)
    return 1;

  *pszSpeaker = '\0';
  *pszEventText = '\0';
  *puFlags = 0;

  *pnEventId = atoi(pszEvent);

  // some event messages have no speaker or flag fields
  if ((*pnEventId != EID_INFO) && (*pnEventId != EID_CHANNELFULL) &&
      (*pnEventId != EID_CHANNEL) &&
      (*pnEventId != EID_CHANNELDOESNOTEXIST) &&
      (*pnEventId != EID_CHANNELRESTRICTED) && (*pnEventId != EID_ERROR)) {

    char szJunk[MAXSTRINGLENGTH];
    sscanf(pszEvent, "%d %s %s %x", pnEventId, szJunk, pszSpeaker, puFlags);
  }

  // the event text is enclosed in quotes
  char *p = strchr(pszEvent, '"');
  if (p) {
    strncpy(pszEventText, p+1, MAXTEXTLENGTH);
    pszEventText[MAXTEXTLENGTH-1] = '\0';

    // nix the trailing quote
    p = strrchr(pszEventText, '"');
    if (p)
      *p = '\0';
  }

  return 1;
}


//============================================================================
int BnBot::Dispatch(char *szEventMsg) {
  int nEventId;
  int nRC = 1;
  u_long uFlags;
  char szSpeaker[MAXSTRINGLENGTH] = "";
  char szEventText[MAXTEXTLENGTH] = "";

  if (!ParseEvent(szEventMsg,
                  &nEventId,
                  szSpeaker,
                  &uFlags,
                  szEventText)) {
    return 0;
  }

  // dispatch to the appropriate event handler
  switch (nEventId) {
    case EID_SHOWUSER:
      OnShowUser(szSpeaker, uFlags, szEventText);
    break;

    case EID_JOIN:
      OnJoin(szSpeaker, uFlags, szEventText);
    break;

    case EID_USERFLAGS:
      OnUserFlags(szSpeaker, uFlags, szEventText);
    break;

    case EID_LEAVE:
      OnLeave(szSpeaker, uFlags, szEventText);
    break;

    case EID_TALK:
      OnTalk(szSpeaker, uFlags, szEventText);
    break;

    case EID_BROADCAST:
      OnBroadcast(szSpeaker, uFlags, szEventText);
    break;

    case EID_CHANNEL:
      OnChannel(szSpeaker, uFlags, szEventText);
    break;

    case EID_WHISPER:
      nRC = OnWhisper(szSpeaker, uFlags, szEventText);
    break;

    case EID_WHISPERSENT:
      OnWhisperSent(szSpeaker, uFlags, szEventText);
    break;

    case EID_EMOTE:
      OnEmote(szSpeaker, uFlags, szEventText);
    break;

    case EID_CHANNELFULL:
      OnChannelFull(szSpeaker, uFlags, szEventText);
    break;

    case EID_CHANNELDOESNOTEXIST:
      OnChannelDoesNotExist(szSpeaker, uFlags, szEventText);
    break;

    case EID_CHANNELRESTRICTED:
      OnChannelRestricted(szSpeaker, uFlags, szEventText);
    break;

    case EID_INFO:
      OnInfo(szSpeaker, uFlags, szEventText);
    break;

    case EID_ERROR:
      OnError(szSpeaker, uFlags, szEventText);
    break;

    case EID_UNIQUENAME:
      OnUniqueName(szSpeaker, uFlags, szEventText);
    break;

    default:
      LogWrite("Unhandled event ID %d", nEventId);
      return 0;
    break;
  }

  return nRC;
}


//============================================================================
// I really cleaned this function up, and boy did it need it...
int BnBot::MsgLoop() {
	int nRC = 1;

	if (s == INVALID_SOCKET)
		return 0;

// First, move data so that buffer size is maximized (if buffer is not empty)

	if (m_nBufLen > 0)
	{
		char tempBuf[BUFSIZE];
		memmove(tempBuf, m_stageBuf+m_nBufPos, m_nBufLen);
		memmove(m_stageBuf, tempBuf, m_nBufLen);
	}

// Okay, now try to fill buffer
	fd_set fds;
	FD_ZERO(&fds);
	FD_SET(s, &fds);
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;

	int n = select(s+1, &fds, 0, 0, &tv);
	if (n) 
	{
		int nNumToRead = BUFSIZE-m_nBufLen;

		n = recv(s, m_stageBuf+m_nBufLen, nNumToRead, 0);
		if (n < 0) 
		{
			char ts[256];
			wsprintf(ts, "recv() returned %d, error code %u", n, WSAGetLastError());
			return 0;
		}

		m_nBufLen += n;
	}

// Next, try to dispatch all messages currently in the staging buffer:
	while (m_nBufLen > 0) 
	{
		char *m = m_stageBuf+m_nBufPos;
		int nMsgLen=0;
		while (nMsgLen < m_nBufLen) 
		{
			if (m[nMsgLen] == '\n')
				break;
			nMsgLen++;
		}

		nMsgLen++;
		if (nMsgLen <= m_nBufLen)
		{

			m[nMsgLen-1] = '\0';

			if (isdigit(*m))
				Dispatch(m);

			m_nBufLen -= nMsgLen;
			m_nBufPos += nMsgLen;

			if (m_nBufLen == 0)
				m_nBufPos = 0;
		}
		else // something wrong.. just reset everything for next time
		{
			m_nBufLen = 0;
			m_nBufPos = 0;
		}
	}

	// Notice: this next function will only get called when a message
	// has come in and we are through processing it. If you want to do
	// periodic Idle processing, create a timer and do it. Don't rely on
	// this function.  I only left it in for compatibility...
	IdleHook(); 

	return nRC;
}


//=============================================================================
int BnBot::LogOpen(char *lpszFileName) {

  if (log) {
    return 0;
  }

  log = fopen(lpszFileName, "ab");
  if (!log)
    return 0;

  return 1;
}


//=============================================================================
int __cdecl BnBot::LogWrite(char *lpszFmt, ...) {
  va_list argptr;
  char szOutStr[1024];

  if (!log)
    return 0;

  va_start(argptr, lpszFmt);
  vsprintf(szOutStr, lpszFmt, argptr);
  va_end(argptr);

  fprintf(log, "%s\r\n", szOutStr);
  return 1;
}


//=============================================================================
void BnBot::LogClose() {

  if (log)
    fclose(log);
  log = 0;
}


/*****************************************************************************
*
* Virtual Event Handlers
*
******/


//============================================================================
inline int BnBot::OnShowUser(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnJoin(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnUserFlags(char *szSpeaker,
                              u_long uFlags,
                              char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnLeave(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnTalk(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnBroadcast(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnChannel(char *szSpeaker, u_long uFlags, char *szEventText) {

  strcpy(szCurrentChannel, szEventText);
  return 1;
}

//============================================================================
inline int BnBot::OnWhisper(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnWhisperSent(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnEmote(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnChannelFull(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnChannelDoesNotExist(char *szSpeaker, u_long uFlags,
                                 char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnChannelRestricted(char *szSpeaker, u_long uFlags,
                               char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnInfo(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}

//============================================================================
inline int BnBot::OnError(char *szSpeaker, u_long uFlags, char *szEventText) {

  return 1;
}


//============================================================================
inline int BnBot::OnUniqueName(char *szSpeaker, u_long uFlags, char *szEventText) {

  strcpy(szUniqueName, szSpeaker);
  return 1;
}

