unit DNDObjects;

interface

Uses contnrs, Dialogs, SysUtils, Classes, DNDStream, UniDNDListBox;

Const MAX_ATTACKS  =  5;
Const MAX_LEVELS   = 20;
Const MAX_SPELL_LEVEL = 9;

Type DNDException = Class(Exception);

Const AbTxt : Array[0..5] of String = ('STR','DEX',
'CON',
'INT',
'WIS',
'CHA');

Const
     XP : Array[1..20] of Integer =
     ( 0, 1000, 3000, 6000, 10000, 15000, 21000, 28000, 36000, 45000,
       55000, 66000, 78000, 91000, 105000, 120000, 136000, 153000, 171000, 190000);

type
     TAffected = Class;
     TTrainedSkill = Class;
     TOtherPoints = Class;
     TSkillList = Class;
     TDNDList = Class;
     TPrerequisites = Class;
     TBaseFeatList = Class;
     TTrainedFeat = Class;
     TLanguageList = Class;
     TFeatList = Class;
     TBaseSkillList = Class;

  TMouseMoveEvent = procedure(Sender: TObject; Shift: TShiftState;
    X, Y: Integer) of object;

TDNDObject = Class(TPrimalDNDObject)
Public
  Procedure StreamWrite(Stream : TDNDStream); Virtual;
  Procedure StreamRead(Stream : TDNDStream); Virtual;
End;

 TDNDList = Class(TPrimalDNDList)
  function    GetObject(Index: Integer): TDNDObject;
  procedure   SetObject(Index: Integer; AObject: TDNDObject);
  property    Items[Index: Integer]: TDNDObject Read GetObject Write SetObject; default;
 End;

TOnChange  = Procedure of Object;

TAbilities = Class
Private
  fStr               : Integer;
  fDex               : Integer;
  fCon               : Integer;
  fInt               : Integer;
  fWis               : Integer;
  fCha               : Integer;
  Procedure          SetStr(newStr : Integer);
  Procedure          SetDex(newDex : Integer);
  Procedure          SetCon(newCon : Integer);
  Procedure          SetInt(newInt : Integer);
  Procedure          SetWis(newWis : Integer);
  Procedure          SetCha(newCha : Integer);
  Procedure          Changed;
Public
  OnChange           : TOnChange;
  Constructor Create; Virtual;
  Destructor  Destroy; Override;
  Procedure   Clear;                     // Reset all values to default.
  Procedure   Add(Source : TAbilities);  // Add 'Source' values to this object
  Property    Str : Integer read fStr write SetStr;
  Property    Dex : Integer read fDex write SetDex;
  Property    Con : Integer read fCon write SetCon;
  Property    Int : Integer read fInt write SetInt;
  Property    Wis : Integer read fWis write SetWis;
  Property    Cha : Integer read fCha write SetCha;
  Procedure   Roll4D4DropLowest;
  Function    MeetsPrerequisites(Character : TAbilities) : Boolean;
  Function    StrModifier : Integer;
  Function    DexModifier : Integer;
  Function    ConModifier : Integer;
  Function    IntModifier : Integer;
  Function    WisModifier : Integer;
  Function    ChaModifier : Integer;
  Procedure StreamWrite(Stream : TDNDStream);
  Procedure StreamRead(Stream : TDNDStream);
End;

TSavingThrows = Class
Private
  fFortitude          : Integer;
  fWillpower          : Integer;
  fReflex             : Integer;
  Procedure          SetFortitude(newFortitude : Integer);
  Procedure          SetWillpower(newWillpower : Integer);
  Procedure          SetReflex(newReflex : Integer);
  Procedure          Changed;
Public
  OnChange           : TOnChange;
  Constructor Create; Virtual;
  Destructor  Destroy; Override;
  Procedure   Clear;                       // Reset all values to default.
  Procedure   Add(Source : TSavingThrows); // Add 'Source' values to this object
  Property    Fortitude : Integer read fFortitude write SetFortitude;
  Property    Willpower : Integer read fWillpower write SetWillpower;
  Property    Reflex    : Integer read fReflex write SetReflex;
  Function    MeetsPrerequisites(Character : TSavingThrows) : Boolean;
  Procedure   StreamWrite(Stream : TDNDStream);
  Procedure   StreamRead(Stream : TDNDStream);
End;

TOtherPoints = Class
Private
  fMaxSkill              : Integer;
  fMaxFeat               : Integer;
  fMaxBonusFeat          : Integer;
  fMaxClassAbilities     : Integer;
  fHit                   : Integer;
  fArmorClass            : Integer;
  fLevel                 : Integer;
  fSpeed                 : Integer;
  fToHit                 : Array[0..MAX_ATTACKS-1] of Integer; // 0 = Primary Attack, 1 = Secondary Attack, 2 = Third Attack, etc.
  fDamage                : Array[0..MAX_ATTACKS-1] of Integer; // 0 = Primary Attack, 1 = Secondary Attack, 2 = Third Attack, etc.
  fBareHandToHit         : Array[0..MAX_ATTACKS-1] of Integer; // 0 = Primary Attack, 1 = Secondary Attack, 2 = Third Attack, etc.
  fArcane                : Array[0..MAX_SPELL_LEVEL] of Integer;
  fKnown                 : Array[0..MAX_SPELL_LEVEL] of Integer;
  fDivine                : Array[0..MAX_SPELL_LEVEL] of Integer;
  fInitiative            : Integer;
  fCasterBonusLevels     : Integer;
{  fMinLength, fMaxLength : Integer;
  fMinWeight, fMaxWeight : Integer; }
  Procedure   SetMaxSkill(newMaxSkill : Integer);
  Procedure   SetMaxFeat(newMaxFeat : Integer);
  Procedure   SetHit(newHit : Integer);
  Procedure   SetSpeed(newSpeed : Integer);
  Procedure   SetMaxClassAbilities(newMaxClassAbilities : Integer);
  Procedure   SetArmorClass(newArmorClass : Integer);
  Procedure   SetLevel(NewLevel : Integer);
  Procedure   SetDamage(Index: Integer; NewDamage : Integer);
  Procedure   SetToHit(Index: Integer; NewToHit : Integer);
  Function    GetDamage(Index: Integer) : Integer;
  Function    GetToHit(Index: Integer) : Integer;
  Procedure   SetBareHandToHit(Index: Integer; NewToHit : Integer);
  Function    GetBareHandToHit(Index: Integer) : Integer;
  Procedure   SetArcane(Index: Integer; NewArcane : Integer);
  Function    GetArcane(Index: Integer) : Integer;
  Procedure   SetKnownArcane(Index: Integer; NewKnownArcane : Integer);
  Function    GetKnownArcane(Index: Integer) : Integer;
  Procedure   SetDivine(Index: Integer; NewDivine : Integer);
  Function    GetDivine(Index: Integer) : Integer;
  Procedure   SetInitiative(NewInitiative: Integer);
  Procedure   SetCasterBonusLevels(NewCasterBonusLevels: Integer);
  Procedure   SetMaxBonusFeat(newMaxBonusFeat : Integer);
  Procedure   Changed;
Public
  OnChange            : TOnChange;
  Constructor Create; Virtual;
  Destructor  Destroy; Override;
  Procedure   Clear;                     // Reset all values to default.
  Procedure   Add(Source : TOtherPoints); // Add 'Source' values to this object
  Property    MaxSkill   : Integer read fMaxSkill   write SetMaxSkill;
  Property    MaxFeat    : Integer read fMaxFeat    write SetMaxFeat;
  Property    MaxBonusFeat    : Integer read fMaxBonusFeat    write SetMaxBonusFeat;
  Property    Hit        : Integer read fHit        write SetHit;
  Property    ArmorClass : Integer read fArmorClass write SetArmorClass;
  Property    Initiative         : Integer read finitiative write SetInitiative;
  Property    CasterBonusLevels  : Integer read fCasterBonusLevels write SetCasterBonusLevels;
  Property    Level              : Integer read fLevel      write SetLevel;
  Property    Speed              : Integer read fSpeed      write SetSpeed;
  Property    MaxClassAbilities  : Integer read fMaxClassAbilities    write SetMaxClassAbilities;
  Property    ToHit[Index: Integer]: Integer read GetToHit write SetToHit;
  Property    BareHandToHit[Index: Integer]: Integer read GetBareHandToHit write SetBareHandToHit;
  Property    Damage[Index: Integer]: Integer read GetDamage write SetDamage;
  Property    Divine[Index: Integer]: Integer read GetDivine write SetDivine;
  Property    Arcane[Index: Integer]: Integer read GetArcane write SetArcane;
  Property    Known[Index: Integer]: Integer read GetKnownArcane write SetKnownArcane;
  Function    MeetsPrerequisites(Character : TOtherpoints) : Boolean;
  Procedure   StreamWrite(Stream : TDNDStream);
  Procedure   StreamRead(Stream : TDNDStream);
End;

TAffected = Class ( TDNDObject )
Public
  Savingthrows        : TSavingthrows;
  Abilities           : TAbilities;
  Skills              : TSkillList;
  Points              : TOtherPoints;
  OnChange            : TOnChange;
  Prerequisites       : TPrerequisites;
  Alignment           : Word;
  Feats               : TBaseFeatList;

  SelectableFeats     : TBaseFeatList;
  SpecialBonusFeats   : TBaseFeatList;

  Languages           : TLanguageList;
  BonusLanguages      : TLanguageList; // Languages they can CHOOSE from

  SelectableWeapons   : TDNDList; // Weapons they can CHOOSE from
  FreeWeapons         : TDNDList; // Weapons
  Procedure   Changed;
  Constructor Create; Override;
  Destructor  Destroy; Override;
  Procedure   Clear; Virtual;         // Reset all values to default.
  Procedure   Add(Source : TAffected); // Add 'Source' values to this object
  Function    MeetsPrerequisites(Character : TAffected) : Boolean; Virtual;
  Procedure StreamWrite(Stream : TDNDStream); Override;
  Procedure StreamRead(Stream : TDNDStream); Override;
End;

TPrerequisites = Class ( TAffected )
End;


TBaseSkill = Class (TAffected)
Public
 PrimeAbility      : Integer; // The Ability Bonus that affects this skill. (0=STR,1=DEX,2=CON,3=INT,4=WIS,5=CHA)
 ArmorCheckPenalty : Boolean;
 TrainedOnly       : Boolean;
 Parameters        : TStringList;
 Synergy           : TBaseSkillList;
 Constructor Create; Override;
 Destructor  Destroy; Override;
 Procedure StreamWrite(Stream : TDNDStream); Override;
 Procedure StreamRead(Stream : TDNDStream); Override;
End;

TTrainedSkill = Class (TDNDObject)
Protected
 Function GetName: String; Override;
 Procedure SetValue(NewValue : Real); override;
 Function GetValue : Real; override;
Public
 Points          : Real;
 BaseSkill       : TBaseSkill;
 List            : TSkillList;
 CrossClass      : Boolean;  // If Crossclass then final points is halved.
 Parameter       : String;
 Function DisplayValue : Real; Override;
 Constructor Create(fSkill : TBaseSkill; fList : TSkillList); Reintroduce;
 Destructor Destroy; Override;
 Procedure Clear;
 Procedure StreamWrite(Stream : TDNDStream); Override;
 Procedure StreamRead(Stream : TDNDStream); Override;
End;

TLanguage = Class (TDNDObject)
End;

TLanguageList = Class (TDNDList)
Public
  Procedure   AddList(Source : TLanguageList);  // Add Source items to this list.
  function    GetLanguage(Index: Integer): TLanguage;
  procedure   SetLanguage(Index: Integer; AObject: TLanguage);
  Procedure   AddLanguage(Language : TLanguage);
  property    Items[Index: Integer]: TLanguage Read GetLanguage Write SetLanguage; default;
  Procedure StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Procedure StreamRead(Stream : TDNDStream; StoreContent: Boolean);
End;

TDomain = Class (TDNDObject)
End;

TDomainList = Class (TDNDList)
Public
  Procedure   AddList(Source : TDomainList);  // Add Source items to this list.
  function    GetDomain(Index: Integer): TDomain;
  procedure   SetDomain(Index: Integer; AObject: TDomain);
  Procedure   AddDomain(Domain : TDomain);
  property    Items[Index: Integer]: TDomain Read GetDomain Write SetDomain; default;
  Procedure StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Procedure StreamRead(Stream : TDNDStream; StoreContent: Boolean);
End;

TSkillList = Class (TDNDList)
Private
  Procedure   Changed;
  Procedure   RaiseSkill(Skill : TTrainedSkill);
  Function   GetTotalSkillpoints : Real;
Public
  OnChange            : TOnChange;
  Constructor Create;
  Destructor  Destroy; Override;
  Procedure   AddList(Source : TSkillList);  // Add Source items to this list.
  function    GetSkill(Index: Integer): TTrainedSkill;
  procedure   SetSkill(Index: Integer; AObject: TTrainedSkill);
  property    Items[Index: Integer]: TTrainedSkill Read GetSkill Write SetSkill ; default;
  Function    HasBaseSkill(HasSkill : TBaseSkill) : TTrainedSkill;
  property    TotalSkillpoints : Real read GetTotalSkillpoints;
  Procedure   StreamWrite(Stream : TDNDStream);
  Procedure   StreamRead(Stream : TDNDStream);
End;

TBaseSkillList = Class (TDNDList)
Private
  function    GetSkill(Index: Integer): TBaseSkill;
  procedure   SetSkill(Index: Integer; AObject: TBaseSkill);
Public
  property    Items[Index: Integer]: TBaseSkill Read GetSkill Write SetSkill ; default;
  Procedure   StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Procedure   StreamRead(Stream : TDNDStream; StoreContent: Boolean);
  Procedure   Clear; override;
  Function    HasSkill(zSkill : TBaseSkill) : Boolean;
End;

TBaseFeat = Class (TAffected)
 Parameters      : TStringList;
 Flags           : Integer; { Bit 0 Set: Feat may be selected more than once
                              Bit 1 Set: Feat parameters are all skills
                            }
 NeedsFeat       : String;  { This basefeat draws its parameters from a specific selected feat
                             ( for example, Weapon Specialisation requires the
                               character to have the weapon focus feat of the
                               same type. }
 Constructor Create; Override;
 Destructor  Destroy; Override;
 Function    MeetsPrerequisites(Character : TAffected) : Boolean; Override;
 Procedure   StreamWrite(Stream : TDNDStream); Override;
 Procedure   StreamRead(Stream : TDNDStream); Override;
End;

TBaseFeatList = Class (TDNDList)
Public
  function    GetFeat(Index: Integer): TBaseFeat;
  procedure   SetFeat(Index: Integer; AObject: TBaseFeat);
  property    Items[Index: Integer]: TBaseFeat Read GetFeat Write SetFeat ; default;
  Procedure   AddList(Source : TBaseFeatList);  // Add Source items to this list.
  Procedure   AddFeat(Feat : TBaseFeat);
  Procedure   StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Procedure   StreamRead(Stream : TDNDStream; StoreContent: Boolean);
End;

TTrainedFeat = Class (TDNDObject)
Protected
 Function GetName: String; Override;
Public
 Parameter       : String;
 BaseFeat        : TBaseFeat;
 List            : TFeatList;
 Constructor Create(fFeat : TBaseFeat; fList : TFeatList); Reintroduce;
 Destructor Destroy; Override;
 Procedure   StreamWrite(Stream : TDNDStream); Override;
 Procedure   StreamRead(Stream : TDNDStream); Override;
End;

TFeatList = Class (TDNDList)
Public
  function    GetFeat(Index: Integer): TTrainedFeat;
  procedure   SetFeat(Index: Integer; AObject: TTrainedFeat);
  property    Items[Index: Integer]: TTrainedFeat Read GetFeat Write SetFeat ; default;
  Procedure   StreamWrite(Stream : TDNDStream);
  Procedure   StreamRead(Stream : TDNDStream);
End;

Function Roll4D6() : Integer;
Function Roll5D6() : Integer;
Function UMIN(A, B : Integer) : Integer;

{ Add a line of text on which this addition failed }
Procedure Requires(S :String);

implementation

 Uses DNDCharacter, DNDWorld, FrmMain, DNDWeapons;

 function TDNDList.GetObject(Index: Integer): TDNDObject;
 Begin
   Result := TDNDObject(GetItem(Index));
 End;

 procedure TDNDList.SetObject(Index: Integer; AObject: TDNDObject);
 Begin
   SetItem(index,AObject);
 End;

 Procedure Requires(S :String);
 Begin
{$IFNDEF NOMAIN}
  if (S='') Then Begin
    Main.HintTitle.Caption := '';
    Main.Hint.Visible := False;
    Exit;
  End;
  If Main.HintTitle.Caption = '' Then Begin
    Main.HintTitle.Caption := 'Unmet Requirements';
    Main.HintText.Caption := '';
  End;
  if ( Main.HintText.Caption <> '' ) Then Main.HintText.Caption := Main.HintText.Caption + ', ';
  Main.HintText.Caption :=
  Main.HintText.Caption + S;
  Main.Hint.Visible := True;
{$ENDIF}
 End;

 Function UMIN(A, B : Integer) : Integer;
 Begin
  If A < B Then Result := A Else Result := B;
 end;

  Procedure TDNDObject.StreamWrite(Stream : TDNDStream);
  Begin
    With Stream do Begin
      Stream.WriteInteger(fVersion);
      Stream.WriteInteger(fGroup);
      Stream.WriteReal(fValue);
      Stream.WriteString(fName);
      Stream.WriteString(fDescription);
    End;
  End;

  Procedure TDNDObject.StreamRead(Stream : TDNDStream);
  Begin
    With Stream do Begin
     fVersion     := Stream.ReadInteger;
     fGroup       := Stream.REadINteger;
     fValue       := Stream.ReadReal;
     fName        := Stream.ReadString;
     fDescription := Stream.ReadString;
    End;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////


  Function TAffected.MeetsPrerequisites(Character : TAffected) : Boolean;
  Var I,J,H  :Integer;
  Found : Boolean;
  Begin
   Requires('');
   Result :=
      Prerequisites.Savingthrows.MeetsPrerequisites(Character.SavingThrows) AND
      Prerequisites.Abilities.MeetsPrerequisites(Character.Abilities) AND
      Prerequisites.Points.MeetsPrerequisites(Character.Points);

   If Character.ClassNameIs('TCharacter') Then Begin

       if Prerequisites.Points.ToHit[0] > 0 Then Begin
         If TCharacter(Character).MeleeBase(0) < Prerequisites.Points.ToHit[0] Then Begin
           Requires('Base Attack Bonus +'+IntToStr(Prerequisites.Points.ToHit[0]));
           Result := False;
         End;
       End;

       if Prerequisites.Points.fBareHandToHit[0] > 0 Then Begin
         If TCharacter(Character).UnarmedBase(0) < Prerequisites.Points.ToHit[0] Then Begin
           Requires('Base Unarmed Attack Bonus +'+IntToStr(Prerequisites.Points.ToHit[0]));
           Result := False;
         End;
       End;

       for I := 0 To Prerequisites.Feats.Count-1 Do Begin
         Found := False;
         for J := 0 To TCharacter(Character).LearnedFeats.Count-1 Do
           If TCharacter(Character).LearnedFeats[J].BaseFeat =
              Prerequisites.Feats[I] Then Found := True;

         For H := 0 To TCharacter(Character).Classes.Count-1 Do
         for J := 0 To TCharacter(Character).Classes[H].LearnedBonusFeats.Count-1 Do
           If TCharacter(Character).Classes[H].LearnedBonusFeats[J].BaseFeat =
              Prerequisites.Feats[I] Then Found := True;

         for J := 0 To TCharacter(Character).Feats.Count-1 Do
           If TCharacter(Character).Feats[J] =
              Prerequisites.Feats[I] Then Found := True;
         if (not Found) Then Begin
           Result := False;
           Requires(Prerequisites.Feats[I].Name);
         End;
       End;

       for I := 0 To Prerequisites.Skills.Count-1 Do Begin
         Found := False;
         for J := 0 To TCharacter(Character).Skills.Count-1 Do
           If (TCharacter(Character).Skills[J].BaseSkill = Prerequisites.Skills[I].BaseSkill ) and
              (TCharacter(Character).Skills[J].Parameter = Prerequisites.Skills[I].Parameter ) and
              (TCharacter(Character).Skills[J].Points >= Prerequisites.Skills[I].Points )
           Then Found := True;

         if (not Found) Then Begin
           Result := False;
           Requires(Prerequisites.Skills[I].Name+'('+FloatToStr(Prerequisites.Skills[I].Points)+')');
         End;
       End;


   End;
   {      <<<}
  End;

  Procedure TAffected.Changed;
  Begin
    if Assigned(OnChange) Then OnChange;
  End;

  Constructor TAffected.Create;
  Begin
   Inherited;
   Savingthrows          := TSavingthrows.Create;
   SavingThrows.OnChange := Changed;
   Abilities          := TAbilities.Create;
   Abilities.OnChange := Changed;
   Skills             := TSkillList.Create;
   Skills.OnChange    := Changed;
   Points             := TOtherPoints.Create;
   Points.OnChange    := Changed;
   Feats              := TBaseFeatList.Create;
   SelectableFeats    := TBaseFeatList.Create;
   SpecialBonusFeats  := TBaseFeatList.Create;
   Languages          := TLanguageList.Create;
   BonusLanguages     := TLanguageList.Create;
   FreeWeapons        := TWeaponList.Create;
   SelectableWeapons  := TWeaponList.Create;
   If not ClassNameIs('TPrerequisites') Then Prerequisites := TPrerequisites.Create;
  End;

  Destructor TAffected.Destroy;
  Begin
   If not ClassNameIs('TPrerequisites') Then Prerequisites.Free;
   BonusLanguages.Free;
   Languages.Free;
   Feats.Free;
   SelectableFeats.Free;
   SpecialBonusFeats.Free;
   Points.Free;
   Skills.Free;
   Abilities.Free;
   Savingthrows.Free;
   FreeWeapons.Free;
   SelectableWeapons.Free;
   Inherited;
  End;

  Procedure TAffected.Clear;
  Begin
   Points.Clear;
   Skills.Clear;
   Abilities.Clear;
   Savingthrows.Clear;
   Feats.Clear;
   SelectableFeats.Clear;
   SpecialBonusFeats.Clear;
   Languages.Clear;
   BonusLanguages.Clear;
   FreeWeapons.Clear;
   SelectableWeapons.Clear;
  End;

  Procedure TAffected.Add(Source : TAffected);
  Begin
   Points.Add(Source.Points);
   Languages.AddList(Source.Languages);
   Skills.AddList(Source.Skills);
   Feats.AddList(Source.Feats);
   SelectableFeats.AddList(Source.SelectableFeats);
   SpecialBonusFeats.AddList(Source.SpecialbonusFeats);
   Abilities.Add(Source.Abilities);
   SavingThrows.Add(Source.SavingThrows);
   BonusLanguages.AddList(Source.BonusLanguages);
   TWeaponList(FreeWeapons).AddList(TWeaponList(Source.FreeWeapons));
   TWeaponList(SelectableWeapons).AddList(TWeaponList(Source.SelectableWeapons));
  End;

  Procedure TAffected.StreamWrite(Stream : TDNDStream);
  Begin
   Inherited;
   Points.StreamWrite(Stream);
   Abilities.StreamWrite(Stream);
   SavingThrows.StreamWrite(Stream);
   Languages.StreamWrite(Stream,False);
   Feats.StreamWrite(Stream,False);
   SelectableFeats.StreamWrite(Stream,False);
   SpecialBonusFeats.StreamWrite(Stream,False);
   Skills.StreamWrite(Stream);
   BonusLanguages.StreamWrite(Stream,False);
   TWeaponList(FreeWeapons).StreamWrite(Stream,False);
   TWeaponList(SelectableWeapons).StreamWrite(Stream,False);
   If not ClassNameIs('TPrerequisites') Then Prerequisites.StreamWrite(Stream);
  End;

  Procedure TAffected.StreamRead(Stream : TDNDStream);
  Begin
   Inherited;
   Points.StreamRead(Stream);
   Abilities.StreamRead(Stream);
   SavingThrows.StreamRead(Stream);
   Languages.StreamRead(Stream,False);
   Feats.StreamRead(Stream,False);
   SelectableFeats.StreamRead(Stream,False);
   SpecialBonusFeats.StreamRead(Stream,False);
   Skills.StreamRead(Stream);
   BonusLanguages.StreamRead(Stream, False);
   TWeaponList(FreeWeapons).StreamRead(Stream,False);
   TWeaponList(SelectableWeapons).StreamRead(Stream,False);
   If not ClassNameIs('TPrerequisites') Then Prerequisites.StreamRead(Stream);
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

    Function Roll4D6() : Integer;
    Var Dice : Array[1..4] of Integer;
        I : Integer;
        LowestVal,LowestDice : Integer;
    Begin
     For I := 1 To 4 do Dice[I] := Random(6)+1;

     LowestVal  := 255;
     LowestDice := 0;
     For I := 1 To 4 Do If ( Dice[I] < LowestVal ) Then Begin
       LowestVal := Dice[I];
       LowestDice := I;
     End;

     Result := 0;
     For I := 1 To 4 Do If ( LowestDice <> I ) Then Begin
      Result := Result + Dice[I];
     End;

    End;

  /////////////////////////////////////////////////////////

    Function Roll5D6() : Integer;
    Var Dice : Array[1..5] of Integer;
        I : Integer;
        LowestVal,LowestDice : Integer;
        LastLowestDice : Integer;
    Begin
     For I := 1 To 5 do Dice[I] := Random(6)+1;

     LowestVal  := 255;
     LowestDice := 0;
     For I := 1 To 5 Do If ( Dice[I] < LowestVal ) Then Begin
       LowestVal := Dice[I];
       LowestDice := I;
     End;

     LowestVal := 255;
     LastLowestDice := 0;
     For I := 1 To 5 Do If ( Dice[I] < LowestVal ) and ( I <> LowestDice ) Then Begin
       LowestVal := Dice[I];
       LastLowestDice := I;
     End;

     Result := 0;
     For I := 1 To 5 Do If ( LowestDice <> I ) and ( LastLowestDice <> I) Then Begin
      Result := Result + Dice[I];
     End;

    End;

  Function ToModifier(Val : Integer) : Integer;
  Begin
     Result := ((Val Div 2)-5);
  End;

  Procedure TAbilities.StreamWrite(Stream : TDNDStream);
  Begin
    With Stream Do Begin
      Stream.WriteInteger(fStr);
      Stream.WriteInteger(fDex);
      Stream.WriteInteger(fCon);
      Stream.WriteInteger(fInt);
      Stream.WriteInteger(fWis);
      Stream.WriteInteger(fCha);
    End;
  End;

  Procedure TAbilities.StreamRead(Stream : TDNDStream);
  Begin
    With Stream Do Begin
      fStr := Stream.ReadInteger;
      fDex := Stream.ReadInteger;
      fCon := Stream.ReadInteger;
      fInt := Stream.ReadInteger;
      fWis := Stream.ReadInteger;
      fCha := Stream.ReadInteger;
    End;
  End;

  Function  TAbilities.StrModifier : Integer;
  Begin
   Result := ToModifier(fStr);
  End;

  Function  TAbilities.DexModifier : Integer;
  Begin
   Result := ToModifier(fDex);
  End;

  Function  TAbilities.ConModifier : Integer;
  Begin
   Result := ToModifier(fCon);
  End;

  Function  TAbilities.IntModifier : Integer;
  Begin
   Result := ToModifier(fInt);
  End;

  Function  TAbilities.WisModifier : Integer;
  Begin
   Result := ToModifier(fWis);
  End;

  Function  TAbilities.ChaModifier : Integer;
  Begin
   Result := ToModifier(fCha);
  End;

  Procedure TAbilities.Roll4D4DropLowest;

  Begin
    fStr := Roll4D6();
    fDex := Roll4D6();
    fCon := Roll4D6();
    fInt := Roll4D6();
    fWis := Roll4D6();
    fCha := Roll4D6();
    Changed;
  End;

  Function TAbilities.MeetsPrerequisites(Character : TAbilities) : Boolean;
  Begin
    Result := False;
    if (fStr > Character.Str ) Then Requires('Str '+IntToStr(fStr)+'+')
    Else if (fDex > Character.Dex ) Then Requires('Dex '+IntToStr(Dex)+'+')
    Else if (fCon > Character.Con ) Then Requires('Con '+IntToStr(Con)+'+')
    Else if (fInt > Character.Int ) Then Requires('Int '+IntToStr(Int)+'+')
    Else if (fWis > Character.Wis ) Then Requires('Wis '+IntToStr(Wis)+'+')
    Else if (fCha > Character.Cha ) Then Requires('Cha '+IntToStr(Cha)+'+')
    Else Result := True;
  End;

  Constructor TAbilities.Create;
  Begin
   Clear;
  End;

  Procedure  TAbilities.Changed;
  Begin
    if Assigned(OnChange) Then OnChange;
  End;

  Procedure  TAbilities.SetStr(newStr : Integer);
  Begin
     if fStr <> newStr Then Begin
       fStr := newStr;
       Changed;
     End
  End;

  Procedure  TAbilities.SetDex(newDex : Integer);
  Begin
     if fDex <> newDex Then Begin
       fDex := newDex;
       Changed;
     End;
  End;

  Procedure  TAbilities.SetCon(newCon : Integer);
  Begin
     if fCon <> newCon Then Begin
       fCon := newCon;
       Changed;
     End;
  End;

  Procedure  TAbilities.SetInt(newInt : Integer);
  Begin
     if fInt <> newInt Then Begin
       fInt := newInt;
       Changed;
     End;
  End;

  Procedure  TAbilities.SetWis(newWis : Integer);
  Begin
     if fWis <> newWis Then Begin
       fWis := newWis;
       Changed;
     End;
  End;

  Procedure  TAbilities.SetCha(newCha : Integer);
  Begin
     if fCha <> newCha Then Begin
       fCha := newCha;
       Changed;
     End;
  End;

  Destructor TAbilities.Destroy;
  Begin
   Inherited;
  End;

  Procedure TAbilities.Clear;
  Begin
    fStr := 0; fDex := 0; fCon := 0;
    fInt := 0; fWis := 0; fCha := 0;
  End;

  Procedure TAbilities.Add(Source : TAbilities);
  Begin
    fStr := Source.Str + fStr;
    fDex := Source.Dex + fDex;
    fCon := Source.Con + fCon;
    fInt := Source.Int + fInt;
    fWis := Source.Wis + fWis;
    fCha := Source.Cha + fCha;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TSavingThrows.StreamWrite(Stream : TDNDStream);
  Begin
    With Stream Do Begin
      Stream.WriteInteger(fFortitude);
      Stream.WriteInteger(fReflex);
      Stream.WriteInteger(fWillpower);
    End;
  End;

  Procedure TSavingThrows.StreamRead(Stream : TDNDStream);
  Begin
    With Stream Do Begin
      fFortitude := Stream.ReadInteger;
      fReflex    := Stream.ReadInteger;
      fWillPower := Stream.ReadInteger;
    End;
  End;

  Function TSavingThrows.MeetsPrerequisites(Character : TSavingThrows) : Boolean;
  Begin
    Result := (Character.Fortitude >= fFortitude) and
              (Character.Reflex >= fReflex) and
              (Character.Willpower >= fWillpower);

    If (Character.Willpower < fWillpower) Then
       Requires('Willpower '+IntToStr(fWillpower));
    If (Character.Fortitude < fFortitude) Then
       Requires('Fortitude '+IntToStr(fFortitude));
    If (Character.Reflex < fReflex) Then
       Requires('Refex '+IntToStr(fReflex));

  End;

  Procedure   TSavingThrows.Changed;
  Begin
    if Assigned(OnChange) Then OnChange;
  End;

  Constructor TSavingThrows.Create;
  Begin
   Clear;
  End;

  Destructor TSavingThrows.Destroy;
  Begin
  End;

  Procedure  TSavingThrows.SetFortitude(newFortitude : Integer);
  Begin
     if fFortitude <> newFortitude Then Begin
       fFortitude := newFortitude;
       Changed;
     End;
  End;

  Procedure  TSavingThrows.SetWillpower(newWillpower : Integer);
  Begin
     if fWillpower <> newWillpower Then Begin
       fWillpower := newWillpower;
       Changed;
     End;
  End;

  Procedure  TSavingThrows.SetReflex(newReflex : Integer);
  Begin
     if fReflex <> newReflex Then Begin
       fReflex := newReflex;
       Changed;
     End;
  End;

  Procedure TSavingThrows.Clear;
  Begin
    fFortitude := 0;
    fWillpower := 0;
    fReflex    := 0;
  End;

  Procedure TSavingThrows.Add(Source : TSavingThrows);
  Begin
    fFortitude := Source.Fortitude + fFortitude;
    fWillpower := Source.Willpower + fWillpower;
    fReflex    := Source.Reflex    + fReflex;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TOtherPoints.StreamWrite(Stream : TDNDStream);
  Var I : Integer;
  Begin
    With Stream Do Begin
      Stream.WriteInteger(fMaxSkill);
      Stream.WriteInteger(fMaxFeat);
      Stream.WriteInteger(fMaxBonusFeat);
      Stream.WriteInteger(fHit);
      Stream.WriteInteger(fArmorClass);
      Stream.WriteInteger(fLevel);
      Stream.WriteInteger(fSpeed);
      Stream.WriteInteger(fMaxClassAbilities);
      Stream.WriteInteger(fInitiative);
      Stream.WriteInteger(fcasterBonusLevels);

      Stream.WriteInteger(MAX_ATTACKS);
      For I := 0 To MAX_ATTACKS-1 Do Begin
         Stream.WriteInteger(fToHit[I]);
         Stream.WriteInteger(fDamage[I]);
         Stream.WriteInteger(fBareHandToHit[I]);
      End;

      Stream.WriteInteger(MAX_SPELL_LEVEL);
      For I := 0 To MAX_SPELL_LEVEL Do Begin
         Stream.WriteInteger(fArcane[I]);
         Stream.WriteInteger(fKnown[I]);
      End;
    End;
  End;

  Procedure TOtherPoints.StreamRead(Stream : TDNDStream);
  Var I, Max : Integer;
  Begin
    With Stream Do Begin
      fMaxSkill   := Stream.ReadInteger;
      fMaxFeat    := Stream.ReadInteger;
      fMaxBonusFeat    := Stream.ReadInteger;
      fHit        := Stream.ReadInteger;
      fArmorClass := Stream.ReadInteger;
      fLevel      := Stream.ReadInteger;
      fSpeed      := Stream.ReadInteger;
      fMaxClassAbilities := Stream.ReadInteger;
      fInitiative := Stream.ReadInteger;
     fCasterBonusLevels := Stream.ReadInteger;

      Max := Stream.ReadInteger;
      For I := 0 To Max-1 Do Begin
         fToHit[I]  := Stream.ReadInteger;
         fDamage[I] := Stream.ReadInteger;
         fBareHandToHit[I] := Stream.ReadInteger;
      End;

      Max := Stream.ReadInteger;
      For I := 0 To Max Do Begin
         fArcane[I] := Stream.ReadInteger;
         fKnown[I] := Stream.ReadInteger;
      End;
    End;
  End;

  Constructor TOtherPoints.Create;
  Begin
   Clear;
  End;

  Destructor TOtherPoints.Destroy;
  Begin
  End;

  Function  TOtherPoints.MeetsPrerequisites(Character : TOtherpoints) : Boolean;
  Begin
    Result := (Character.MaxSkill >= fmaxSkill ) and
              (Character.MaxFeat >= fMaxFeat ) and
              (Character.MaxBonusFeat >= fMaxBonusFeat ) and
              (Character.Speed >= fSpeed ) and
              (Character.Hit >= fHit ) And
              (Character.ArmorClass >= fArmorClass ) And
              (Character.Level >= fLevel ) And
              (Character.Speed >= fSpeed );
  End;

  Procedure TOtherPoints.SetMaxSkill(newMaxSkill : Integer);
  Begin
     if fMaxSkill <> newMaxSkill Then Begin
       fMaxSkill  := newMaxSkill;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetMaxFeat(newMaxFeat : Integer);
  Begin
     if fMaxFeat <> newMaxFeat Then Begin
       fMaxFeat  := newMaxFeat;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetMaxBonusFeat(newMaxBonusFeat : Integer);
  Begin
     if fMaxBonusFeat <> newMaxBonusFeat Then Begin
       fMaxBonusFeat  := newMaxBonusFeat;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetMaxClassAbilities(newMaxClassAbilities : Integer);
  Begin
     if fMaxClassAbilities <> newMaxClassAbilities Then Begin
       fMaxClassAbilities  := newMaxClassAbilities;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetSpeed(newSpeed : Integer);
  Begin
     if fSpeed <> newSpeed Then Begin
       fSpeed  := newSpeed;
       Changed;
     End;
  End;

  Procedure   TOtherPoints.Changed;
  Begin
    if Assigned(OnChange) Then OnChange;
  End;

  Procedure TOtherPoints.SetHit(newHit : Integer);
  Begin
     if fHit <> newHit Then Begin
       fHit  := newHit;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetArmorClass(newArmorClass : Integer);
  Begin
     if fArmorClass <> newArmorClass Then Begin
       fArmorClass  := newArmorClass;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetDamage(Index: Integer; NewDamage : Integer);
  Begin
     if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetDamage out of Range');
        Exit;
     End;
     if fDamage[Index] <> newDamage Then Begin
       fDamage[Index]  := newDamage;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetInitiative(NewInitiative: Integer);
  Begin
     if fInitiative <> newInitiative Then Begin
       fInitiative  := newInitiative;
       Changed;
     End;
  End;
  
  Procedure TOtherPoints.SetCasterBonusLevels(NewCasterBonusLevels: Integer);
  Begin
     if fCasterBonusLevels <> newCasterBonusLevels Then Begin
       fCasterBonusLevels  := newCasterBonusLevels;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetToHit(Index: Integer; NewToHit : Integer);
  Begin
     if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetToHit out of Range');
        Exit;
     End;
     if fToHit[Index] <> newToHit Then Begin
       fToHit[Index]  := newToHit;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetBareHandToHit(Index: Integer; NewToHit : Integer);
  Begin
     if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetBareHandToHit out of Range');
        Exit;
     End;
     if fBareHandToHit[Index] <> newToHit Then Begin
       fBareHandToHit[Index]  := newToHit;
       Changed;
     End;
  End;

  Function TOtherPoints.GetBareHandToHit(Index: Integer) : Integer;
  Begin
    if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetBareHandToHit out of Range');
       Exit;
    End;
    Result := fBareHandToHit[Index];
  End;

  Procedure TOtherPoints.SetLevel(newLevel : Integer);
  Begin
    if fLevel <> newLevel Then Begin
      fLevel  := newLevel;
      Changed;
    End;
  End;

  Procedure TOtherPoints.SetArcane(Index: Integer; NewArcane : Integer);
  Begin
     if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetArcane out of Range');
        Exit;
     End;
     if fArcane[Index] <> newArcane Then Begin
       fArcane[Index]  := newArcane;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetDivine(Index: Integer; NewDivine : Integer);
  Begin
     if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetDivine out of Range');
        Exit;
     End;
     if fDivine[Index] <> newDivine Then Begin
       fDivine[Index]  := newDivine;
       Changed;
     End;
  End;

  Procedure TOtherPoints.SetKnownArcane(Index: Integer; NewKnownArcane : Integer);
  Begin
     if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
        Raise DNDException.Create('SetKnownArcane out of Range');
        Exit;
     End;
     if fKnown[Index] <> newKnownArcane Then Begin
       fKnown[Index]  := newKnownArcane;
       Changed;
     End;
  End;

  Function TOtherPoints.GetKnownArcane(Index: Integer) : Integer;
  Begin
    if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetKnownArcane out of Range');
       Exit;
    End;
    Result := fKnown[Index];
  End;

  Function TOtherPoints.GetDamage(Index: Integer) : Integer;
  Begin
    if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetDamage out of Range');
       Exit;
    End;
    Result := fDamage[Index];
  End;

  Function TOtherPoints.GetArcane(Index: Integer) : Integer;
  Begin
    if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetArcane out of Range');
       Exit;
    End;
    Result := fArcane[Index];
  End;

  Function TOtherPoints.GetDivine(Index: Integer) : Integer;
  Begin
    if ( (Index > MAX_SPELL_LEVEL) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetDivine out of Range');
       Exit;
    End;
    Result := fDivine[Index];
  End;

  Function TOtherPoints.GetToHit(Index: Integer) : Integer;
  Begin
    if ( (Index >= MAX_ATTACKS) or (Index < 0) ) Then Begin
       Raise DNDException.Create('GetToHit out of Range');
       Exit;
    End;
    Result := fToHit[Index];
  End;

  Procedure TOtherPoints.Clear;
  Var I : Integer;
  Begin
    fMaxSkill   := 0;
    fMaxFeat    := 0;
    fMaxBonusFeat := 0;
    fHit        := 0;
    fLevel      := 0;
    fInitiative := 0;
    fArmorClass := 0;
    fMaxClassAbilities := 0;
    For I := 0 To MAX_ATTACKS-1 Do Begin
       fToHit[I] := 0;
       fBareHandToHit[I] := 0;
       fDamage[I] := 0;
    End;
    For I := 0 To MAX_SPELL_LEVEL Do Begin
       fArcane[I] := 0;
       fDivine[I] := 0;
       fKnown[I]  := 0;
    End;
    fSpeed := 0;
  End;

  Procedure TOtherPoints.Add(Source : TOtherPoints);
  Var I : Integer;
  Begin
    fCasterBonusLevels := Source.CasterBonusLevels + fCasterBonusLevels;
    fInitiative := Source.Initiative + fInitiative;
    fMaxSkill   := Source.MaxSkill   + fMaxSkill;
    fMaxFeat    := Source.MaxFeat    + fMaxFeat;
    fMaxBonusFeat := Source.MaxBonusFeat + fMaxBonusFeat;
    fHit        := Source.Hit        + fHit;
    fArmorClass := Source.ArmorClass + fArmorClass;
    fSpeed      := Source.Speed      + fSpeed;
    fLevel      := Source.Level      + fLevel;
    fMaxClassAbilities  := Source.MaxClassAbilities + fMaxClassAbilities;
    For I := 0 To MAX_ATTACKS-1 Do Begin
       fToHit[I] := Source.ToHit[I] + fToHit[I];
       fBarehandToHit[I] := Source.BareHandToHit[I] + fBareHandToHit[I];
       fDamage[I] := Source.Damage[I] + fDamage[I];
    End;

    For I := 0 To MAX_SPELL_LEVEL Do Begin
       fArcane[I] := Source.Arcane[I] + fArcane[I];
       fKnown[I]  := Source.Known[I]  + fKnown[I];
       fDivine[I] := Source.Divine[I] + fDivine[I];
    End;

  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TTrainedSkill.StreamWrite(Stream : TDNDStream);
  Begin
    Inherited;
    Stream.WriteReal(Points);
    Stream.WriteBaseSkill(BaseSkill);
    Stream.WriteBoolean(CrossClass);
    Stream.WriteString(Parameter);
  End;

  Procedure TTrainedSkill.StreamRead(Stream : TDNDStream);
  Begin
    Inherited;
    Points := Stream.ReadReal;
    BaseSkill := TBaseSkill(Stream.ReadBaseSkill);
    CrossClass := Stream.ReadBoolean;
    Parameter := Stream.ReadString;
  End;

  Function TTrainedSkill.GetName : String;
  Begin
   If Assigned(BaseSkill) Then Begin
     Result := BaseSkill.Name;
     If Parameter <> '' Then Result := Result + ' ('+Parameter+')';
   End Else Result := '';
  End;

  Constructor TTrainedSkill.Create(fSkill : TBaseSkill; fList : TSkillList);
  // Create a new Trained skill based on fSkill, and add it to a Skill List.
  Begin
   Inherited Create();
   BaseSkill := fSkill;
   List  := fList;
   List.Add(Self);
   Points := 1;
   Min := 0;
   Max := 100;
  End;

  Destructor TTrainedSkill.Destroy;
  // Kill self from the available Skill list.
  Begin
   List.Delete(List.IndexOf(Self));
   Inherited;
  End;

  Procedure TTrainedSkill.Clear;
  Begin
   Points := 0;
   Inherited;
  End;

  Procedure TTrainedSkill.SetValue(NewValue : Real);
  Begin
    Points := NewValue;
  End;

  Function TTrainedSkill.GetValue : Real;
  Begin
    Result := Points;
  End;

  Function TTrainedSkill.DisplayValue : Real;
  Begin
   If CrossClass Then
     Result := GetValue / 2 Else Result := GetValue;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TLanguageList.StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Var I : Integer;
  Begin
    With Stream Do Begin
      WriteInteger(Count);
      For I := 0 To Count-1 Do Begin
         Stream.WriteString(Items[I].Name);
         if Storecontent Then Items[I].StreamWrite(Stream);
      End;
    End;
  End;

  Procedure TLanguageList.StreamRead(Stream : TDNDStream; StoreContent: Boolean);
  Var I, Max : Integer;
      Language : TLanguage;
      Name : String;
  Begin
    Clear;
    Max := Stream.ReadInteger;
    For I := 0 To Max-1 Do Begin
       Name := Stream.ReadString;
       Language := World.AddLanguage(Name);
       if StoreContent then Language.StreamRead(Stream);
       if World.Languages <> Self Then Add(Language);
    End;
  End;

  Procedure  TLanguageList.AddLanguage(Language : TLanguage);
  Begin
   If(IndexOf(Language) <> -1) Then Exit;
   Add(Language);
  End;

  Procedure  TLanguageList.AddList(Source : TLanguageList);  // Add Source items to this list.
  Var I : Integer;
  Begin
    For I := 0 To Source.Count-1 Do
      AddLanguage(Source.Items[I]);
  End;

  function TLanguageList.GetLanguage(Index: Integer): TLanguage;
  Begin
    Result := TLanguage(GetItem(Index));
  End;

  procedure   TLanguageList.SetLanguage(Index: Integer; AObject: TLanguage);
  Begin
    SetItem(index,AObject);
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TDomainList.StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Var I : Integer;
  Begin
    With Stream Do Begin
      WriteInteger(Count);
      For I := 0 To Count-1 Do Begin
         Stream.WriteString(Items[I].Name);
         if Storecontent Then Items[I].StreamWrite(Stream);
      End;
    End;
  End;

  Procedure TDomainList.StreamRead(Stream : TDNDStream; StoreContent: Boolean);
  Var I, Max : Integer;
      Domain : TDomain;
      Name : String;
  Begin
    Clear;
    Max := Stream.ReadInteger;
    For I := 0 To Max-1 Do Begin
       Name := Stream.ReadString;
       Domain := World.AddDomain(Name);
       if StoreContent then Domain.StreamRead(Stream);
       if World.Domains <> Self Then Add(Domain);
    End;
  End;

  Procedure  TDomainList.AddDomain(Domain : TDomain);
  Begin
   If(IndexOf(Domain) <> -1) Then Exit;
   Add(Domain);
  End;

  Procedure  TDomainList.AddList(Source : TDomainList);  // Add Source items to this list.
  Var I : Integer;
  Begin
    For I := 0 To Source.Count-1 Do
      AddDomain(Source.Items[I]);
  End;

  function TDomainList.GetDomain(Index: Integer): TDomain;
  Begin
    Result := TDomain(GetItem(Index));
  End;

  procedure   TDomainList.SetDomain(Index: Integer; AObject: TDomain);
  Begin
    SetItem(index,AObject);
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TSkillList.StreamWrite(Stream : TDNDStream);
  Var I : Integer;
  Begin
   Inherited;
   Stream.WriteInteger(Count);
   For I := 0 To Count-1 Do
      Items[I].StreamWrite(Stream);
  End;

  Procedure TSkillList.StreamRead(Stream : TDNDStream);
  Var I, Max : Integer;
      Skill : TTrainedSkill;
  Begin
     Inherited;
     Clear;
     Max := Stream.ReadInteger;
     For I := 0 To Max-1 Do Begin
         Skill := TTrainedSkill.Create(nil,Self);
         Skill.StreamRead(Stream);
     End;
  End;

  Constructor TSkillList.Create;
  Begin
    Inherited;
  End;

  Procedure   TSkillList.Changed;
  Begin
    if Assigned(OnChange) Then OnChange;
  End;

  Destructor TSkillList.Destroy;
  Begin
   Inherited;
  End;

  Procedure TSkillList.RaiseSkill(Skill : TTrainedSkill);
  Var I : Integer;
      Sk : TTrainedSkill;
  Begin
    For I := 0 To Count-1 Do
      If (Skill.BaseSkill = Items[I].BaseSkill) and
         (Skill.Parameter = Items[I].Parameter) Then Begin
         Items[I].Points := Items[I].Points + Skill.DisplayValue;
         Exit;
      End;
    Sk := TTrainedSkill.Create(Skill.BaseSkill,Self);
    Sk.Parameter := Skill.Parameter;
    If Skill.CrossClass Then
      Sk.Points := Skill.DisplayValue
    Else
      Sk.Points := Skill.DIsplayValue;
  End;

  Procedure TSkillList.AddList(Source : TSkillList);
  // Added Trained Skill Objects are not shared with the source list. Where
  // Possible, Skillpoints are merged or new Trained Skill Objects created.
  Var I : Integer;
  Begin
    For I := 0 To Source.Count-1 Do
        RaiseSkill(Source.Items[I]);
    If Source.Count > 0 Then Changed;
  End;

  Procedure TSkillList.SetSkill(Index: Integer; AObject: TTrainedSkill);
  Begin
    SetItem(index,AObject);
  End;

  Function TSkillList.GetSkill(Index: Integer): TTrainedSkill;
  Begin
   Result := TTrainedSkill(GetItem(Index));
  End;

  Function TSkillList.HasBaseSkill(HasSkill : TBaseSkill) : TTrainedSkill;
  Var I : Integer;
  Begin
   Result := Nil;
   For I := 0 To Count-1 Do
     If Items[i].BaseSkill = HasSkill Then Begin
       Result := Items[I];
       Exit;
     End;
  End;

  Function TSkillList.GetTotalSkillpoints : Real;
  Var I : Integer;
  Begin
   Result := 0;
   For I := 0 To Count-1 Do
         Result := Result + Items[i].Points;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Function  TBaseSkilLList.HasSkill(zSkill : TBaseSkill) : Boolean;
  Var I : Integer;
  Begin
   Result := False;
   For I := 0 To Count-1 Do
    If Items[I] = zSkill Then REsult := True;
  End;

  Procedure TBaseSkillList.Clear;
  Begin
   Inherited;
  End;

  Procedure TBaseSkillList.StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Var I : Integer;
  Begin
     Stream.WriteInteger(Count);
     For I := 0 To Count-1 Do Begin
       Stream.WriteString(Items[I].Name);
       If Storecontent Then Items[I].StreamWrite(Stream);
     End;
  End;

  Procedure TBaseSkillList.StreamRead(Stream : TDNDStream; StoreContent: Boolean);
  Var I,Max : Integer;
      BaseSkill  : TBaseSkill;
      Name  : String;
  Begin
     Max := Stream.ReadInteger;
     Clear;
     For I := 0 To Max-1 Do Begin
       Name := Stream.ReadString;
       BaseSkill := World.AddSkill(Name);
       If StoreContent Then BaseSkill.StreamRead(Stream);
       if World.Skills <> Self Then Add(BaseSkill);
     End;
  End;

  Procedure TBaseSkillList.SetSkill(Index: Integer; AObject: TBaseSkill);
  Begin
    SetItem(index,AObject);
  End;

  Function TBaseSkillList.GetSkill(Index: Integer): TBaseSkill;
  Begin
   Result := TBaseSkill(GetItem(Index));
  End;
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////


  Procedure TBaseFeatList.StreamWrite(Stream : TDNDStream; StoreContent: Boolean);
  Var I : Integer;
  Begin
    Stream.WriteInteger(Count);
    For I := 0 To Count-1 Do Begin
       Stream.WriteString(Items[I].Name);
       if StoreContent Then Items[I].StreamWrite(Stream);
    End;
  End;

  Procedure TBaseFeatList.StreamRead(Stream : TDNDStream; StoreContent: Boolean);
  Var I, Max : Integer;
      BaseFeat : TBaseFeat;
      Name : String;
  Begin
   Max := Stream.ReadInteger;
   Clear;
   For I := 0 To Max-1 Do Begin
     Name := Stream.ReadString;
     BaseFeat := World.AddFeat(Name);
     If StoreContent Then BaseFeat.StreamRead(Stream);
     if World.Feats <> Self Then Add(BaseFeat);
   End;
  End;

  Procedure TBaseFeatList.SetFeat(Index: Integer; AObject: TBaseFeat);
  Begin
    SetItem(index,AObject);
  End;

  Function TBaseFeatList.GetFeat(Index: Integer): TBaseFeat;
  Begin
   Result := TBaseFeat(GetItem(Index));
  End;

  Procedure  TBaseFeatList.AddList(Source : TBaseFeatList);  // Add Source items to this list.
  Var I : Integer;
  Begin
    For I := 0 To Source.Count-1 Do
      AddFeat(Source.Items[I]);
  End;

  Procedure TBaseFeatList.AddFeat(Feat : TBaseFeat);
  Begin
    If(IndexOf(Feat) <> -1) Then Exit;
    Add(Feat);
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TTrainedFeat.StreamWrite(Stream : TDNDStream);
  Begin
    Inherited;
    Stream.WriteString(Parameter);
    Stream.WriteBaseFeat(BaseFeat);
  End;

  Procedure TTrainedFeat.StreamRead(Stream : TDNDStream);
  Begin
    Inherited;
    Parameter := Stream.ReadString;
    BaseFeat := TBaseFeat(Stream.ReadBaseFeat);
  End;

  Function TTrainedFeat.GetName : String;
  Begin
   If Assigned(BaseFeat) Then Begin
     Result := BaseFeat.Name;
     If Parameter <> '' Then Result := Result + ' ('+Parameter+')';
   End Else Result := '';
  End;

  Constructor TTrainedFeat.Create(fFeat : TBaseFeat; fList : TFeatList);
  // Create a new Trained Feat based on fFeat, and add it to a Feat List.
  Begin
   Inherited Create();
   BaseFeat := fFeat;
   List  := fList;
   List.Add(Self);
  End;

  Destructor TTrainedFeat.Destroy;
  // Kill self from the available Feat list.
  Begin
   List.Delete(List.IndexOf(Self));
   Inherited;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TFeatList.StreamWrite(Stream : TDNDStream);
  Var I : Integer;
  Begin
    Inherited;
    Stream.WriteInteger(Count);
    For I := 0 To Count-1 Do Begin
      Items[I].StreamWrite(Stream);
    End;
  End;

  Procedure TFeatList.StreamRead(Stream : TDNDStream);
  Var I, Max : Integer;
      Feat : TTrainedFeat;
  Begin
    Inherited;
    Max := Stream.ReadInteger;
    Clear;
    For I := 0 To Max-1 Do Begin
      Feat := TTrainedFeat.Create(Nil,Self);
      Feat.StreamRead(Stream);
    End;
  End;

  Procedure TFeatList.SetFeat(Index: Integer; AObject: TTrainedFeat);
  Begin
    SetItem(index,AObject);
  End;

  Function TFeatList.GetFeat(Index: Integer): TTrainedFeat;
  Begin
   Result := TTRainedFeat(GetItem(Index));
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Procedure TBaseSkill.StreamWrite(Stream : TDNDStream);
  Var I : Integer;
  Begin
   Stream.WriteInteger(PrimeAbility);
   Stream.WriteBoolean(ArmorCheckPenalty);
   Stream.WriteBoolean(TrainedOnly);

   Stream.WriteInteger(Parameters.Count);
   For I := 0 To Parameters.Count-1 Do
     Stream.WriteString(Parameters[I]);
   Synergy.StreamWrite(Stream,False);
  End;

  Procedure TBaseSkill.StreamRead(Stream : TDNDStream);
  Var I, Max : Integer;
  Begin
   PrimeAbility := Stream.ReadInteger;
   ArmorCheckPenalty := Stream.ReadBoolean;
   TrainedOnly := Stream.ReadBoolean;

   Parameters.Clear;
   Max := Stream.ReadInteger;
   For I := 0 To Max-1 Do
     Parameters.Add(Stream.ReadString);
    Synergy.StreamRead(Stream,False);
  End;

  Constructor TBaseSkill.Create;
  Begin
   Inherited;
   Parameters := TStringList.Create;
   Synergy := TBaseSkillList.Create;
  End;

  Destructor  TBaseskill.Destroy;
  Begin
   Synergy.Free;
   Parameters.Free;
   inherited;
  End;

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Function TBaseFeat.MeetsPrerequisites(Character : TAffected) : Boolean;
  Begin
{   If Character.ClassNameIs('TCharacter') Then Begin
       for I := 0 To Prerequisites.Feats.Count-1 Do Begin
         Found := False;
         for J := 0 To TCharacter(Character).LearnedFeats.Count-1 Do
           If TCharacter(Character).LearnedFeats[I].BaseFeat =
              Prerequisites.Feats[I] Then Found := True;
         if (not Found) Then Begin
           Result := False;
           Exit;
         End;
       End;
   End;}
   Result := Inherited MeetsPrerequisites(Character);
  End;

  Constructor TBaseFeat.Create;
  Begin
   Inherited;
   Parameters := TStringList.Create;
  End;

  Destructor  TBaseFeat.Destroy;
  Begin
   Parameters.Free;
   inherited;
  End;

  Procedure   TBaseFeat.StreamWrite(Stream : TDNDStream);
  Var I : Integer;
  Begin
   Inherited;

   Stream.WriteInteger(Parameters.Count);
   For I := 0 To Parameters.Count-1 Do
     Stream.WriteString(Parameters[I]);

   Stream.WriteInteger(Flags);
   Stream.WriteString(NeedsFeat);
  End;

  Procedure   TBaseFeat.StreamRead(Stream : TDNDStream);
  Var I, Max : Integer;
  Begin
   Inherited;

   Parameters.Clear;
   Max := Stream.ReadInteger;
   For I := 0 To Max-1 Do
     Parameters.Add(Stream.ReadString);
   Flags := Stream.ReadInteger;
   NeedsFeat := Stream.ReadString;
  End;


end.

