unit DNDCharacter;

// Included:
// - Assign Race to Player
// - Assign Classes+Class Levels to Player
// - Assign Deity to Player
// - Assign Skills to Player
//
//
// Not Included:
// - Assign Feats to Player
// - Assign Spells to Player
// - Saving Objects to Stream
// - Choosable Traits for a class. (For Example, Thief may choose from several different selections)

interface

Uses contnrs, Dialogs, SysUtils, DNDObjects, DNDClasses, DNDSpells, DNDStream, Classes,
DNDItems;

Const CHARACTER_FILE_ID : String = 'RedbladeCharacterfilev1.00';

const MAX_EQUIPMENT_KITS = 4;

Type

TCharacter = Class (TAffected)
Private
  fDeity              : TBaseDeity;
  Procedure SetRace( NewRace : TBaseRace );
  Procedure SetDeity( NewDeity : TBaseDeity );
  Function GetLevel : Integer;
Public
  WorldFilename       : String;
  fRace               : TBaseRace;
  fClasses            : TClassList;
  Base                : TAffected;
  Alignment           : Integer;
  BonusStat           : Array[1..MAX_LEVELS DIV 4] of Integer;
  LearnedFeats        : TFeatList;
  SpecializedSchools  : String;
  ProHibitedSchools   : String;
  ProhibitedDomains   : String;
  SpecializedDomains  : String;
  Domain1             : String;
  Domain2             : String;
  Gender              : Integer;
  RollMethod          : Integer;
  Money               : TMoney;
  Inventory           : TItemList;

  Equipped            : Array[0..MAX_EQUIPMENT_KITS-1] of TItemList;

  DualWield           : Integer; { Wield two weapons at once }
  TwoHandedWield      : Integer; { Wield a weapon with two hands }
  OtherBladeSide      : Boolean; { Calculate the blade's other side }

  fChanged            : Boolean;

  Procedure Calculate;
  Constructor Create; Override;
  Destructor Destroy; Override;
  Function  AddClass( NewClass : TBaseClass): TTrainedClass;
  Function  AddSkill( ToClass : TBaseClass; NewSkill : TBaseSkill; Parameter : String): TTrainedSkill;
  Function  AddSpell( ToClass : TBaseClass; NewSpell : TBaseSpell; Level : Integer): TTrainedSpell;
  Function  AddFeat( NewFeat : TBaseFeat; Parameter : String): TTrainedFeat;
  Function  AddBonusFeat( NewFeat : TBaseFeat; Parameter : String; Clas : TTrainedClass ): TTrainedFeat;
  Procedure AddLanguage( NewLanguage : TLanguage);
  Procedure RemoveLanguage( NewLanguage : TLanguage );
  Procedure RemoveClass( Remove : TBaseclass );
  Property  Race  : TBaseRace read fRace write SetRace;
  Property  Deity : TBaseDeity read fDeity write SetDeity;
  Function KnowsFeat( NewFeat : TBaseFeat; Parameter : String; Clas : TTrainedClass ) : TTrainedFeat;
  Property  Classes : TClassList read fClasses;
  Property  Level : Integer read GetLevel;
  Function ClassSkillpoints( Clas : TTrainedClass ) : Integer;
  Function ClassSkillpointsUsed( Clas : TTrainedClass ) : Integer;
  Function SkillPointsUsed : Integer;
  Function FeatPointsUsed : Integer;
  Function BonusFeatPointsUsed(Clas : TTrainedClass) : Integer;
  Function SkillPoints : Integer;
  Function BonusAbilities : Integer;
  Function IllegalAlignment(Alignment : Integer) : Boolean;
  Function FeatPoints : Integer;
  Function BonusFeatPoints(Clas : TTrainedClass) : Integer;
  Function ClassFeatpoints( Clas : TTrainedClass ) : INteger;
  Function LanguagePoints : Integer;
  Procedure StreamWrite(Stream : TDNDStream); Override;
  Procedure StreamRead(Stream : TDNDStream); Override;
  Procedure Load(FileName : String);
  Procedure Save(FileName : String);
  Procedure New;
  Function CanSpecialize : Boolean;
  Procedure RollMoney;
  Function CanSelectDomains : Boolean;
  Function HitPoints : Integer;
  Function ArmorClass : Integer;
  Procedure IncludeWierdMultiClassRule;
  Function Initiative : Integer;
  Function Fortitude : Integer;
  Function Reflex : Integer;
  Function Will : Integer;
  Function SaveBonus : Integer;

  Function SkillTotal(Skill : Integer) : Real;

  Function GetAbilityMod(AbMod : Integer): Integer;

  Function SpellDC( Clas : TTrainedClass; Level : Integer ) : Integer;

  Function SizeBonus : Integer;

  Function MeleeBase(Attack : Integer) :  Integer;
  Function RangeBase(Attack : Integer) :  Integer;
  Function UnarmedBase(Attack : Integer) :  Integer;

  Function ClassAbilityPointsUsed( Clas : TTrainedClass ) : Integer;
  Function WisAcBonus : Integer;

  Function SpellsCastPerDay( Clas : TTrainedClass; Level : Integer) : String;
  Function SpellsKnown( Clas : TTrainedClass; Level : Integer) : Integer;

  Function SkillIsCrossClass(Skill : TBaseSkill) : Boolean;
  Procedure RevalidateCrossclass;

  Function MaxSkillRank( Skill : TBaseSkill ) : Real;
  Function CurrentSkillRank( Skill : TBaseSkill ) : Real;

  { Weapon = Index based on equipped index }

  Function GetWeapon(Weapon : Integer) : TDNDItem;

  Function WeaponTotalAttackBonus(Weapon : Integer) : String;
  Function WeaponDamage(Weapon : Integer) : String;
  Function WeaponCritical(Weapon : Integer) : String;
  Function WeaponRange(Weapon : Integer) : String;
  Function WeaponWeight(Weapon : Integer) : String;
  Function WeaponSize(Weapon : Integer) : String;
  Function WeaponType(Weapon : Integer) : String;
  Function WeaponSpecial(Weapon : Integer) : String;

  Function WeaponAttBonusses(Weapon : TDNDItem; WeapNr : Integer) : Integer;
  Function WeaponDamBonusses(Weapon : TDNDItem; WeapNr : Integer) : Integer;
  Function GotFinesse(Weapon : TDNDItem ) : Boolean;
  Function HasFeat( NewFeat : TBaseFeat; Parameter : String ) : TTrainedFeat;
  Function IsProficient(Weapon : TDNDItem ) : Boolean;
  Function HasFocus(Weapon : TDNDItem ) : Boolean;
  Function HasWeaponSpecialization(Weapon : TDNDItem ) : Boolean;

  Function OffhandWeaponLight : Boolean;
  Function Ambidexterious : Boolean;
  Function TwoWeaponFighting : Boolean;
  Function ImprovedTwoWeaponFighting : Boolean;
  Function PrimaryPenalties : Integer;
  Function OffhandPenalties : Integer;

  Function SkillUsable(Skill : TBaseSkill) : Boolean;
  Function SkillAbilityMod(Skill : TBAseSkill) : Integer;
  Function SkillRanks(Skill : TBAseSkill) : Real;
  Function SkillMiscRanks(Skill : TBaseSkill) : Real;
  Function SkillTotalRanks(Skill : TBaseSkill) : Real;

  Function TrainedSkillRanks(Skill : TTrainedSkill) : Real;
  Function TrainedSkillMiscRanks(Skill : TTrainedSkill) : Real;
  Function TrainedSkillTotalRanks(Skill : TTrainedSkill) : Real;

  Function ConHitpoints : Integer;
  Function baseHitPoints : Integer;

{  Function SkillMod    (Skill : TBaseSkill) : String;
  Function SkillRank   (Skill : TBaseSkill) : String;
  Function SkillAbMod  (Skill : TBaseSkill) : String;
  Function SkillMiscMod(Skill : TBaseSkill) : String;}

  Function ArmorBonus : Integer;
  Function ShieldBonus : Integer;
  Function GetArmor : TDNDItem;
  Function GetShield : TDNDItem;
  Function ArmorDexBonus : Integer;

  Function Speed : Integer;

  Function UnarmedDiceSize : Integer;
  Function UnarmedDice : Integer;

  Function InitiativeMisc : Integer;
  Function HasImprovedInitiative : Boolean;

  Function LightLoad : Integer;
  Function MediumLoad : Integer;
  Function HeavyLoad : Integer;

  Function SizeWeightAdjuster : Real;

  Function BAB : String;

  Function BABMeleeBase : Integer;
  Function BABRangeBase : Integer;

  Function TotalLevel : Integer;


{ Function DoubleEdgedWeapon(Weap : TDNDItem; Weapon : Integer) : Boolean;}

End;

implementation

  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////

  Uses FrmMain, DNDWorld;

  Function URANGE(a,b,c : Integer) : Integer;
  Begin
   if b < a Then b := a;
   if b > c Then b := c;
   result := b;
  End;

  Function TCharacter.Speed : Integer;
           { 70 60 50 40 30 20 15 10 5 }
    const Speed : Array[1..7,1..9] of Integer =
       { 12   } (( 70 ,60  ,50 ,40 ,30 ,20, 15, 10, 5 ),
       { 35   } ( 90 ,80  ,65 ,50 ,40 ,25, 20, 15, 10 ),
       { 68   } ( 110, 100, 80,60 ,50 ,35, 25, 20 ,15  ),
       { 911  } ( 130, 120, 95, 70, 60 ,40, 30, 25, 20  ),
       { 1214 } ( 150, 140, 115, 80, 70, 45, 35, 30, 25  ),
       { 1517 } ( 170, 160, 125, 90, 80, 55, 40 ,35, 30   ),
       { 1820 } ( 190, 180, 140, 100, 90, 60, 45 ,40, 35   ));

  Var Ar : Integer;
    Sped : Integer;
    Super : Integer;
    CalcSpecial : Integer;
    I : Integer;
  Begin
    Super := 0;

    CalcSpecial := -1;
    For I := 0 To fClasses.Count-1 Do if fClasses[I].BaseClass.MonkSpeed > 0 Then Begin
     CalcSpecial := I;
     Break;
    End;

    If CalcSpecial > -1 Then Begin
      Sped := fRace.Points.Speed;
      If Sped < 6 Then
        Ar := 9
      Else Case Sped of
        6..10 : Ar := 8;
       11..15 : Ar := 7;
       16..20 : Ar := 6;
       21..30 : Ar := 5;
       31..40 : Ar := 4;
       41..50 : Ar := 3;
       51..60 : Ar := 2;
       Else Ar := 1;
      End;
      Super := Speed[(fClasses[CalcSpecial].Level div 3)+1,Ar]-(Sped);
    End;

    Result := Points.Speed+(Super);

  End;

  Function TCharacter.SpellsCastPerDay( Clas : TTrainedClass; Level : Integer) : String;
  Var mMod : Integer;
   Bonus : Integer;
  Begin
   mMod := 0;
   Case Clas.BaseClass.CastStat Of
     0 :  mMod := Abilities.StrModifier;
     1 :  mMod := Abilities.DexModifier;
     2 :  mMod := Abilities.ConModifier;
     3 :  mMod := Abilities.IntModifier;
     4 :  mMod := Abilities.WisModifier;
     5 :  mMod := Abilities.ChaModifier;
   End;
    Result := IntToStr(Clas.Points.Arcane[Level]);
    Bonus := 0;
    If ( Level > 0 ) and ( mMod > 1 ) Then
      Bonus := BonusSpells[URANGE(1,mMod,17), Level];
    If ( Bonus > 0)  Then Result := Result + '+' + IntToStr(Bonus);
  End;

  Function TCharacter.SpellsKnown( Clas : TTrainedClass; Level : Integer) : Integer;
  Begin
    Result := Clas.Points.Known[Level];
  End;

  Function RollxDy ( x, Y : Integer ) : Integer;
  Var I  :Integer;
  Begin
   Result := 0;
   For I := 1 To X do Result := Result + Random(Y);
  End;

  Procedure TCharacter.RollMoney;
  Begin
   Money.Clear;
   If ( Classes.Count > 0 ) Then Begin
    Money.Gold := RollxDY(Classes[0].BaseClass.MoneyDice,Classes[0].BaseClass.MoneyDiceSize)* Classes[0].Baseclass.MoneyMultiplier;
   End;
  End;

  Function TCharacter.SpellDC( Clas : TTrainedClass; Level : Integer ) : Integer;
  Begin
   Result := 10+Level;
   Case Clas.BaseClass.CastStat Of
     0 :  Result := Result + Abilities.StrModifier;
     1 :  Result := Result + Abilities.DexModifier;
     2 :  Result := Result + Abilities.ConModifier;
     3 :  Result := Result + Abilities.IntModifier;
     4 :  Result := Result + Abilities.WisModifier;
     5 :  Result := Result + Abilities.ChaModifier;
   End;
  End;

  Procedure TCharacter.AddLanguage( NewLanguage : TLanguage);
  Begin
   Base.Languages.AddLanguage(NewLanguage);
  End;

  Procedure TCharacter.RemoveLanguage( NewLanguage : TLanguage);
  Var I : Integer;
  Begin
   I := Base.Languages.IndexOf(NewLanguage);
   if ( I <> -1 ) Then
   Base.Languages.Delete(I);
  End;

  Function UMAX(A,B : Integer): Integer;
  Begin
   if ( A > B ) Then Result := A Else Result := B;
  End;

  Function  TCharacter.LanguagePoints : Integer;
  Begin
   Result := 2 + UMAX(0,Abilities.IntModifier);
  End;


  Procedure TCharacter.IncludeWierdMultiClassRule;
  Const
    BaseAttackBonus : Array[1..15,1..3] of Byte =
     {6}  ((  1, 0, 0 ),
           (  2, 0, 0 ),
           (  3, 0, 0 ),
           (  4, 0, 0 ),
           (  5, 0, 0 ),
           (  6, 1, 0 ),
           (  7, 2, 0 ),
           (  8, 3, 0 ),
           (  9, 4, 0 ),
           ( 10, 5, 0 ),
           ( 11, 6, 1 ),
           ( 12, 7, 2 ),
           ( 13, 8, 3 ),
           ( 14, 9, 4 ),
           ( 15,10, 5 ) );

  Begin
   // Wierd Multiclass rule on page 55 of PHB.
   if Classes.Count >= 2 Then Begin
      If ( Points.ToHit[0] > 6 ) Then Begin
        Points.ToHit[1] := UMAX(Points.ToHit[1],BaseAttackBonus[Points.ToHit[0]-5,1]);
        Points.ToHit[2] := UMAX(Points.ToHit[2],BaseAttackBonus[Points.ToHit[0]-5,2]);
        Points.ToHit[3] := UMAX(Points.ToHit[3],BaseAttackBonus[Points.ToHit[0]-5,3]);
      End;
   End;
  End;

  Var Calculating : Boolean;
  Procedure TCharacter.Calculate;
  Var I : Integer;
      C : Integer;
  Begin
   If not Calculating Then Begin
    Calculating := True;
   Clear;
   MeetsPrerequisites(Self);
   Points.Level :=0;
   Add(Base);
   if Assigned(fRace) Then Add(fRace);
   if Abilities.Int < 3 Then Abilities.Int := 3;
   fClasses.AddTo(Self); // Add all classes to self.

   IncludeWierdMultiClassRule;

   if Abilities.Int < 3 Then Abilities.Int := 3;

   For I := 1 To BonusAbilities Do Begin
      Case BonusStat[I] of
         0 : Abilities.Str := Abilities.Str + 1;
         1 : Abilities.Dex := Abilities.Dex + 1;
         2 : Abilities.Con := Abilities.Con + 1;
         3 : Abilities.Int := Abilities.Int + 1;
         4 : Abilities.Wis := Abilities.Wis + 1;
         5 : Abilities.Cha := Abilities.Cha + 1;
      End;
   End;

   For I := 0 To LearnedFeats.Count-1 Do Begin
      Add(LearnedFeats[I].BaseFeat);
   End;

   For C := 0 To Classes.Count-1 Do
   For I := 0 To Classes[C].LearnedBonusFeats.Count-1 Do
      Add(Classes[C].LearnedBonusFeats[I].BaseFeat);

   For I := 0 To Equipped[0].Count-1 Do
    Add(equipped[0][I]);


   // Add Classes
   // Add Spells
   // Add Deity
   // Add Allowed SpellSchools
   // Add DeniedSpellSchools
   // TrainedSkills
   Calculating := False;
   End;
  End;

  Function TCharacter.ClassSkillpointsUsed( Clas : TTrainedClass ) : Integer;
  Var i :Integer;
  Begin
    Result := 0;
    For I := 0 To Clas.LearnedSkills.Count-1 Do Begin
      Result := Result + Round(Clas.LearnedSkills[I].Points);
    End;
  End;

  Function TCharacter.SizeBonus : Integer;
  Begin
    Case fRace.Size of
       0 : Result := 8; {Fine}
       1 : Result := 4; {Diminutive}
       2 : Result := 2; {Tiny}
       3 : Result := 1; {Small}
       5 : Result :=-1; {Large}
       6 : Result :=-2; {Huge}
       7 : Result :=-4; {Gargantuan}
       8 : Result :=-8; {Colossal}
       Else Result := 0; {Medium-size}
    End;
 End;

  Function TCharacter.ClassAbilityPointsUsed( Clas : TTrainedClass ) : Integer;
  Var i :Integer;
  Begin
    Result := 0;
    For I := 0 To Clas.ChosenClassAbilities.Count-1 Do Inc(Result);
  End;

  Function TCharacter.TotalLevel : Integer;
  Begin
   Result := Level;
   if fRace <> nil Then
     Result := Result+frace.LevelCost;
  End;

  Function TCharacter.ClassSkillpoints( Clas : TTrainedClass ) : INteger;
  Var I : Integer;
  Begin
    If Classes.Count = 0 Then Begin
      Result := 0;
      Exit;
    End;

    Result := UMAX( 1, ( Abilities.IntModifier + Clas.BaseClass.Level[1].Points.MaxSkill ) * 4 );

    { Add the human '1 skillpoint per level' bonus }
    if ( Clas = Classes[0] ) and
       ( Assigned(Race) ) and
       ((Race.Flags and 1) > 0 ) Then Result := Result + (Level-1);

    For I := 2 To Clas.Level Do Begin
     Result := Result + UMAX( 1, Abilities.IntModifier + Clas.BaseClass.Level[I].Points.MaxSkill );
    End;
    If ( Assigned(Race) and ( Clas = Classes[0] ) ) Then Result := Result + Race.Points.MaxSkill;
  End;

  Function TCharacter.ClassFeatpoints( Clas : TTrainedClass ) : INteger;
  Var I : Integer;
  Begin
    If not Assigned(Race) Then Begin
      Result := 0;
      Exit;
    End;
    Result := 0;
    For I := 1 To Clas.Level Do Begin
     Result := Result +Clas.BaseClass.Level[I].Points.MaxFeat;
    End;
{    Result := Result + Race.Points.MaxFeat;}
    If ( Clas = Classes[0] ) Then Result := Result + Race.Points.MaxFeat;
  End;

  Function TCharacter.SkillPointsUsed : Integer;
  Var I : Integer;
  Begin
    Result := 0;
    For I := 0 To Classes.Count-1  Do Begin
      Result := Result + ClassSkillPointsUsed(Classes.Items[I]);
    End;
  End;

  Function TCharacter.FeatPointsUsed : Integer;
  Begin
    Result := LearnedFeats.Count;
  End;

  Function TCharacter.BonusFeatPointsUsed(Clas : TTrainedClass) : Integer;
  Begin
    Result := Clas.LearnedBonusFeats.Count;
  End;

  Function TCharacter.SkillPoints : Integer;
  Var I : Integer;
  Begin
    Result := 0;
    For I := 0 To Classes.Count-1  Do Begin
      Result := Result + ClassSkillPoints(Classes.Items[I]);
    End;
  End;

  Constructor TCharacter.Create;
  Var I :Integer;
  Begin
   Inherited Create;
   Inventory           := TItemList.create;
   For I := 0 To MAX_EQUIPMENT_KITS-1 do
     Equipped[I] := TItemList.Create;
   Base                := TAffected.Create;
   Money               := TMoney.Create;
   Base.OnChange       := Changed;
   Base.Abilities.Str  := 10;
   Base.Abilities.Dex  := 10;
   Base.Abilities.Con  := 10;
   Base.Abilities.Int  := 10;
   Base.Abilities.Wis  := 10;
   Base.Abilities.Cha  := 10;
   fRace               := nil;
   fDeity              := nil;
   fClasses            := TClassList.Create;
   fClasses.OnChange   := Changed;
   LearnedFeats        := TFeatList.Create;
  End;

  Destructor TCharacter.Destroy;
  var I : Integer;
  Begin
   Money.Free;
   For I := 0 To MAX_EQUIPMENT_KITS-1 do
     Equipped[I].Free;
   Inventory.Free;
   Inherited Destroy;
  End;

  Procedure TCharacter.SetRace( NewRace : TBaseRace );
  Begin
    If fRace <> NewRace Then Begin
      fRace := NewRace;
      Changed;
    End;
  End;

  Procedure TCharacter.SetDeity( NewDeity : TBaseDeity );
  Begin
    If fDeity <> NewDeity Then Begin
      fDeity := NewDeity;
      Changed;
    End;
  End;

  Function TCharacter.AddClass( NewClass : TBaseClass ): TTrainedClass;
  Begin
    Result := fClasses.HasBaseClass(Newclass);
    If not Assigned(Result) Then Begin
      Result := TTrainedClass.Create( NewClass , fClasses );
      if ( Classes.Count = 1 ) Then RollMoney;
      Changed;
    End;
  End;

  Function TCharacter.AddSpell( ToClass : TBaseClass; NewSpell : TBaseSpell; Level : Integer): TTrainedSpell;
  Var C : TTrainedClass;
      I : Integer;
  Begin
    Result := Nil;
    C := fClasses.HasBaseClass(ToClass);
    If not Assigned(C) Then Exit;

    For I := 0 To C.LearnedSpells.Count-1 Do
    If ( TSpellList(C.LearnedSpells).Items[I].BaseSpell = NewSpell ) Then Result := TSpellList(C.LearnedSpells).Items[I];

    if result = Nil Then Begin
      Result := TTrainedSpell.Create( NewSpell ,TSpellList( C.LearnedSpells) );
      Result.Level := Level;
      Changed;
    End;
  End;

  Function TCharacter.SkillIsCrossClass(Skill : TBaseSkill) : Boolean;
  Var I : Integer;
  Begin
   Result := True;
   For I := 0 To classes.Count-1 Do Begin
     If not Classes[I].IsCrossClass(Skill) Then Result := False;
   End;
  End;

  Procedure TCharacter.RevalidateCrossclass;
  Var I,J : Integer;
  Begin
   For I := 0 To classes.Count-1 Do Begin
     For J := 0 To Classes[I].LearnedSkills.Count-1 Do
        Classes[I].LearnedSkills.Items[J].CrossClass :=
            SkillIsCrossClass(Classes[I].LearnedSkills.Items[J].BaseSkill)
   End;
  End;

  { What is the current total rank for this skill }
  Function TCharacter.CurrentSkillRank( Skill : TBaseSkill ) : Real;
  Var
   T : TTrainedSkill;
  Begin
   T := Skills.HasBaseSkill(Skill);
   If ( Assigned(T) ) Then Result := T.Points Else REsult := 0;
  End;

  { What is the maximum rank for this skill (including the +3 bonus) }
  Function TCharacter.MaxSkillRank( Skill : TBaseSkill ) : Real;
  Var I : Integer;
      OnlyCrossclass:Boolean;
  Begin
   Result := 0;
   OnlyCrossClass := true;
   For I := 0 To classes.Count-1 Do Begin
     If Classes[I].BaseClass.CrossClassSkills.HasSkill(Skill) Then Result := Result + Classes[I].Level;
     if Classes[I].BaseClass.MainSkills.HasSkill(Skill) Then Begin
        Result := Result + Classes[I].Level;
        OnlyCrossClass := False;
     End;
   End;
   If Result > 0 Then Result := Result + 3;
   If OnlyCrossClass Then Result := Result / 2;
  End;

  Function TCharacter.AddSkill( ToClass : TBaseClass; NewSkill : TBaseSkill; Parameter : String): TTrainedSkill;
  Var C : TTrainedClass;
      I : Integer;
  Begin

    Result := Nil;
    C := fClasses.HasBaseClass(ToClass);
    If not Assigned(C) Then Exit;

     For I := 0 To C.LearnedSkills.Count-1 Do Begin
       If ( ( C.LearnedSkills.Items[I].BaseSkill = NewSkill ) AND
            ( C.LearnedSkills.Items[I].Parameter = Parameter ) ) Then Begin
          Result := C.LearnedSkills.Items[I];
       End;
     End;

      If Result = nil Then Begin
        Result := TTrainedSkill.Create( NewSkill , C.LearnedSkills );
       // If Crossclass then set skill as 'crossclass'
        Result.CrossClass := SkillIsCrossClass(NewSkill);
        Result.Parameter := Parameter;
        Changed;
      End;
  End;

  Function TCharacter.ArmorBonus : Integer;
  Var Armor : TDNDItem;
  Begin
     Armor := GetArmor;
     Result := 0;
     If Armor <> nil Then Begin
       Result := Armor.AcBonus;
     End;
  End;

  Function TCharacter.ArmorDexBonus : Integer;
  Var Armor : TDNDItem;
  Begin
     Armor := GetArmor;
     Result := Abilities.DexModifier;
     If Armor <> nil Then Begin
       If Armor.MaxDexBonus < Result Then Result := Armor.MaxDexBonus;
     End;
  End;

  Function TCharacter.ShieldBonus : Integer;
  Var Shield : TDNDItem;
  Begin
     Shield := GetShield;
     Result := 0;
     If Shield <> nil Then Begin
       Result := Shield.AcBonus;
     End;
  End;

  Function TCharacter.ArmorClass : Integer;
  Begin
   Result := Points.ArmorClass+ArmorDexBonus+WisAcBonus+ShieldBonus+ArmorBonus+SizeBonus;
  End;

  Function TCharacter.Initiative : Integer;
  Begin
   Result := {Points.initiative+}1+Abilities.DexModifier+InitiativeMisc;
  End;

  Function TCharacter.InitiativeMisc : Integer;
  Begin
   Result := Points.Initiative;
{   If HasImprovedInitiative Then Result := 4;}
  End;

  Function Tcharacter.SaveBonus : Integer;
  Var AddCharisma : Boolean;
      I : Integer;
  Begin
   AddCharisma := False;
   For I := 0 To Classes.Count-1 Do If Classes[I].BaseClass.AddChaToSaves Then Begin

    AddCharisma := true;
   End;
   If AddCharisma Then Result := Abilities.ChaModifier Else Result := 0;
  End;

  Function Tcharacter.WisAcBonus : Integer;
  Var AddWis : Boolean;
      I : Integer;
  Begin
   AddWis := False;
   For I := 0 To Classes.Count-1 Do If Classes[I].BaseClass.AddWisToAc Then Begin
    AddWis := true;
   End;
   If AddWis Then Result := Abilities.WisModifier Else Result := 0;
  End;

  Function TCharacter.Fortitude : Integer;
  Begin
   Result := SavingThrows.Fortitude+Abilities.ConModifier+SaveBonus;
  End;

  Function TCharacter.Reflex : Integer;
  Begin
   Result := SavingThrows.Reflex+Abilities.DexModifier+SaveBonus;
  End;

  Function TCharacter.Will : Integer;
  Begin
   Result := SavingThrows.Willpower+Abilities.WisModifier+SaveBonus;
  End;

  Procedure TCharacter.RemoveClass( Remove : TBaseClass );
  Var I : Integer;
  Begin
    I := fClasses.IndexOf(Remove);
    If ( I <> -1 ) Then Begin
      FClasses[I].Free;
      Changed;
    End;
  End;

 Function TCharacter.GetLevel : Integer;
  Var I : Integer;
  Begin
    if ( Classes.Count = 0 ) Then Begin
      Result := 0;
      Exit;
    End;
    Result := 0;
    For I := 0 To Classes.Count-1  Do Begin
      Result := Result + Classes.Items[I].Level;
    End;
    If Assigned(Race) Then Result := Result + Race.Points.Level;
  End;

 Function TCharacter.IllegalAlignment(Alignment : Integer) : Boolean;
 Var   I : Integer;
 Begin
   Result := False;
   For I := 0 To Classes.Count-1  Do Begin
     If ( Classes.Items[I].BaseClass.IllegalAlignment and (1 SHL Alignment) > 0 ) Then Begin
       Result := true;
       Exit;
     End;
   End;
 End;

 Function TCharacter.BonusAbilities : Integer;
 Begin
  Result := (Level) Div 4;
 End;

 Var AutoFeat : Boolean;

 Function TCharacter.KnowsFeat( NewFeat : TBaseFeat; Parameter : String; Clas : TTrainedClass ) : TTrainedFeat;
 Var I : Integer;
     C : Integer;
 Begin
   Result := nil;
   AutoFeat := False;
   For I := 0 To Feats.Count-1 Do
    If ( ( Feats[I] = NewFeat )  and
         { Also check if it the feat is flagged as 'does not has to be unique' }
         ( ( Newfeat.Flags and 1 ) = 0 ) ) Then Begin
            AutoFeat := True;
            Result := nil;
            Exit;
         End;

   For I := 0 To LearnedFeats.Count-1 Do
    If ( ( LearnedFeats.Items[I].BaseFeat = NewFeat ) AND
         ( LearnedFeats.Items[I].Parameter = Parameter ) AND
         { Also check if it the feat is flagged as 'does not has to be unique' }
         ( ( Newfeat.Flags and 1 ) = 0 ) ) Then Result := LearnedFeats.Items[I];

   For C := 0 To Classes.Count-1 Do
   For I := 0 To Classes[C].LearnedBonusFeats.Count-1 Do
    If ( ( Classes[C].LearnedBonusFeats.Items[I].BaseFeat = NewFeat ) AND
         ( Classes[C].LearnedBonusFeats.Items[I].Parameter = Parameter ) AND
         { Also check if it the feat is flagged as 'does not has to be unique' }
         ( ( Newfeat.Flags and 1 ) = 0 ) ) Then Result := Classes[C].LearnedBonusFeats.Items[I];
 End;

 Function TCharacter.AddFeat( NewFeat : TBaseFeat; Parameter : String): TTrainedFeat;
 Begin
   Result := KnowsFeat(NewFeat,Parameter,Nil);

   If Result = nil Then Begin
     if (AutoFeat) Then Exit;
     Result := TTrainedFeat.Create(NewFeat,LearnedFeats);
     Result.Parameter := Parameter;
   End;
 End;

 Function TCharacter.AddBonusFeat( NewFeat : TBaseFeat; Parameter : String; Clas : TTrainedClass ): TTrainedFeat;
 Begin
   Result := KnowsFeat(NewFeat,Parameter,Nil);

   If Result = nil Then Begin
     if (AutoFeat) Then Exit;
     Result := TTrainedFeat.Create(NewFeat,Clas.LearnedBonusFeats);
     Result.Parameter := Parameter;
   End;
 End;

 Function TCharacter.FeatPoints : Integer;
 Var I : Integer;
 Begin
  Result := Points.MaxFeat + 1;
  Result := Result + UMAX(0,((Level) Div 3));
{  Result := Result + Race.Points.MaxFeat;}
  For I := 0 To Classes.Count-1  Do
      Result := Result + ClassFeatpoints( Classes.Items[I] );
  If (Classes.Count > 0) and (fRace<>nil) Then Result := Result - Race.Points.MaxFeat;
 End;

 Function TCharacter.BonusFeatPoints(Clas : TTrainedClass) : Integer;
 Begin
  Result := Clas.Points.MaxBonusFeat;
 End;

 Procedure TCharacter.StreamWrite(Stream : TDNDStream);
 Var I : Integer;
 Begin
   Stream.WriteString(WorldFileName);
   Inherited;
   if Assigned(frace) Then Stream.WriteString(fRace.Name) Else Stream.WriteString('');
   if Assigned(fDeity) Then Stream.WriteString(fDeity.Name) Else Stream.WriteString('');
   fClasses.StreamWrite(Stream);
   Base.StreamWrite(Stream);
   Stream.WriteInteger(Alignment);
   Stream.WriteInteger(MAX_LEVELS DIV 4);
   For I := 1 To MAX_LEVELS DIV 4 do Stream.WriteInteger(BonusStat[I]);
   LearnedFeats.StreamWrite(Stream);
   Stream.WriteString(SpecializedSchools);
   Stream.WriteString(ProhibitedSchools);
   Stream.WriteString(Domain1);
   Stream.WriteString(Domain2);
   Stream.WriteString(SpecializedDomains);
   Stream.WriteString(ProhibitedDomains);
{$IFNDEF NOMAIN}
   Stream.WriteString(Main.PersonalityTraits.Text);
   Stream.WriteString(Main.HistoryTxt.Text);
   Stream.WriteString(Main.Goals.Text);
{$ENDIF}
   Stream.WriteInteger(Gender);
   Stream.WriteInteger(RollMethod);
   Inventory.StreamWrite(Stream,True);
   For I := 0 To MAX_EQUIPMENT_KITS-1 do
     Equipped[I].StreamWrite(Stream,TRue);
   Stream.WriteInteger(DualWield);
   Stream.WriteInteger(TwoHandedWield);
 End;

 Procedure TCharacter.StreamRead(Stream : TDNDStream);
 Var I, Max : Integer;
   Campaign : String;
 Begin
   WorldFileName := Stream.ReadString;
   Campaign := ExtractFilePath(ParamStr(0))+'campaigns\'+WorldFileNAme;
   If ( not FileExists(Campaign) ) Then Begin
     ShowMessage('Cannot find '+Campaign+'. I need this campaign file to correctly load this character. Please locate the file and place it at the specified location.');
     Exit;
   End;
   Inherited;
   if World.WorldFileName <> Campaign Then
     World.Load(ExtractFilePath(ParamStr(0))+'campaigns\'+WorldFileNAme);

   fRace := World.AddRace(Stream.ReadString);
   fDeity := World.AddDeity(Stream.ReadString);
   fClasses.StreamRead(Stream);
   Base.StreamRead(Stream);
   Alignment := Stream.ReadInteger;
   Max := Stream.ReadInteger;
   For I := 1 To Max do BonusStat[I] := Stream.ReadInteger;
   LearnedFeats.StreamRead(Stream);
   SpecializedSchools := Stream.ReadString;
   ProhibitedSchools := Stream.ReadString;
   Domain1 := Stream.ReadString;
   Domain2 := Stream.ReadString;
   SpecializedDomains := Stream.ReadString;
   ProhibitedDomains := Stream.ReadString;
{$IFNDEF NOMAIN}
   Main.PersonalityTraits.Text := Stream.ReadString;
   Main.HistoryTxt.Text := Stream.ReadString;
   Main.Goals.Text := Stream.ReadString;
{$ENDIF}
   Gender := Stream.ReadInteger;
   RollMethod := Stream.ReadInteger;
   Inventory.StreamRead(Stream,True);
   For I := 0 To MAX_EQUIPMENT_KITS-1 do
     Equipped[I].StreamRead(Stream,TRue);
   DualWield := Stream.ReadInteger;
   TwoHandedWield := Stream.ReadInteger;
 End;

Procedure TCharacter.Load(FileName : String);
Var Stream : TDNDStream;
Begin
 Stream := TDNDStream.Create(FileName,fmOpenRead);
 if ( Stream.ReadString = CHARACTER_FILE_ID ) Then Begin
   StreamRead(Stream);
   Changed;
 End Else ShowMessage('Not a valid Character file.');
 Stream.Free;
End;

Procedure TCharacter.Save(FileName : String);
Var Stream : TDNDStream;
Begin
 Stream := TDNDStream.Create(FileName,fmOpenWrite or fmCreate);
 Stream.WriteString(CHARACTER_FILE_ID);
 StreamWrite(Stream);
 Stream.Free;
End;

Function TCharacter.CanSpecialize : Boolean;
 Var I : Integer;
 Begin
   For I := 0 To Classes.Count-1  Do Begin
     If Classes.Items[I].BaseClass.CanSpecialize Then Begin Result := True; Exit; End;
   End;
   Result := False;
 End;

 Function TCharacter.CanSelectDomains : Boolean;
 Var I : Integer;
 Begin
   For I := 0 To Classes.Count-1  Do Begin
     If Classes.Items[I].BaseClass.CanSelectDomains Then Begin Result := True; Exit; End;
   End;
   Result := False;
 End;

// Hitpoints. Adds all hitpoints of all classes together, Make First level of
// first class Full Hitpoints. includes con bonuses.
Function TCharacter.baseHitPoints : Integer;
var Cl, I : Integer;
    First : Boolean;
Begin
   Result := 0;
   First := True;
   For Cl := 0 To Classes.Count-1 Do Begin
      For I := 1 to UMIN(Classes[Cl].Level,MAX_LEVELS) Do Begin
         If First Then Begin
           Result := UMAX(1,Result + Classes[Cl].BaseClass.HpPerLevel);
           First := False;
         End Else
           Result := Result + UMAX(1,Classes[Cl].HPLevel[I]);
      End;
   End;
End;

Function TCharacter.HitPoints : Integer;
var Cl, I : Integer;
    First : Boolean;
Begin
   Result := 0;
   First := True;
   For Cl := 0 To Classes.Count-1 Do Begin
      For I := 1 to UMIN(Classes[Cl].Level,MAX_LEVELS) Do Begin
         If First Then Begin
           Result := UMAX(1,Result + Classes[Cl].BaseClass.HpPerLevel + Abilities.ConModifier);
           First := False;
         End Else
           Result := Result + UMAX(1,Classes[Cl].HPLevel[I] + Abilities.ConModifier);
      End;
   End;
End;

Function TCharacter.ConHitpoints : Integer;
var Cl, I : Integer;
Begin
   Result := 0;
   For Cl := 0 To Classes.Count-1 Do Begin
      For I := 1 to UMIN(Classes[Cl].Level,MAX_LEVELS) Do Begin
         Result := Result + Abilities.ConModifier;
      End;
   End;
End;

Function TCharacter.MeleeBase(Attack : Integer) :  Integer;
Begin
  Result := Points.ToHit[Attack];
End;

Function TCharacter.UnarmedBase(Attack : Integer) :  Integer;
Begin
  If ( Points.BareHandToHit[0] = -1 ) Then
    Result := Points.ToHit[Attack]
  Else
    Result := Points.BareHandToHit[Attack];
End;

Function TCharacter.RangeBase(Attack : Integer) :  Integer;
Begin
  Result := Points.ToHit[Attack];
End;

Procedure TCharacter.New;
Var I : Integer;
Begin
  Clear;
  RollMethod := 0;
  Name := '';
  Description := '';
  Gender := 0;
  Base.Clear;
  SpecializedSchools  :='';
  ProhibitedSchools   :='';
  SpecializedDomains  :='';
  ProhibitedDomains   :='';
  Money.Clear;
  Inventory.Clear;
  For I := 0 To MAX_EQUIPMENT_KITS-1 do
    Equipped[I].Clear;
  Domain1 := World.Domains[1].Name;
  Domain2 := World.Domains[2].Name;
  For I := Classes.Count-1 DownTo 0 Do Classes.Items[I].Free;
  For I := LearnedFeats.Count-1 DownTo 0 Do LearnedFeats.Items[I].Free;
  fRace := Nil;
  fDeity := Nil;
  alignment := 0;
  For I := 1 To (MAX_LEVELS DIV 4) Do BonusStat[I] := 0;
   Base.Abilities.Str  := 10;
   Base.Abilities.Dex  := 10;
   Base.Abilities.Con  := 10;
   Base.Abilities.Int  := 10;
   Base.Abilities.Wis  := 10;
   Base.Abilities.Cha  := 10;
  Changed;
End;

Function TCharacter.GetAbilityMod(AbMod : Integer): Integer;
Begin
 Case Abmod of
   0 : Result := Abilities.StrModifier;
   1 : Result := Abilities.DexModifier;
   2 : Result := Abilities.ConModifier;
   3 : Result := Abilities.IntModifier;
   4 : Result := Abilities.WisModifier;
   Else Result := Abilities.ChaModifier;
 End;
End;

Function TCharacter.SkillTotal(Skill : Integer) : Real;
Begin
 Result := Skills[Skill].Points + GetAbilityMod(Skills[Skill].BaseSkill.PrimeAbility);
End;

Function TCharacter.SkillAbilityMod(Skill : TBAseSkill) : Integer;
Begin
 Result := GetAbilityMod(Skill.PrimeAbility);
End;

Function TCharacter.TrainedSkillRanks(Skill : TTrainedSkill) : Real;
Begin
 Result := Skill.Points;
End;

Function TCharacter.TrainedSkillMiscRanks(Skill : TTrainedSkill) : Real;
Begin
 Result := 0;
End;

Function TCharacter.TrainedSkillTotalRanks(Skill : TTrainedSkill) : Real;
Begin
 Result := TrainedSkillRanks(Skill) + SkillAbilityMod(Skill.BaseSkill) + TrainedSkillMiscRanks(Skill);
End;

Function TCharacter.SkillMiscRanks(Skill : TBaseSkill) : Real;
Begin
 Result := 0;
End;

Function TCharacter.SkillTotalRanks(Skill : TBaseSkill) : Real;
Begin
 Result := SkillRanks(Skill) + SkillAbilityMod(Skill) + SkillMiscRanks(Skill);
End;

Function TCharacter.SkillRanks(Skill : TBaseSkill) : Real;
Begin
 { We have no ranks! *SNIF* }
 Result := 0;
End;

Function TCharacter.SkillUsable(Skill : TBaseSkill) : Boolean;
Var I,J : Integer;
Begin
 Result := False;

 { First lets check if we're trained at the skill }
 For I := 0 To Skills.Count-1 Do If Skills[I].BaseSkill = Skill Then Begin
  { We are trained! Well, we can use it then }
  Result := True;
  Exit;
 End;

 { If it can only be used when trained, this is no help! }
 If Skill.TrainedOnly Then Exit;

 For I := 0 To Classes.Count-1 Do Begin
   For J := 0 To Classes[I].BaseClass.MainSkills.Count-1 Do If Classes[I].BaseClass.MainSkills[J] = Skill Then Begin
     Result := True;
     Exit;
   End;
   For J := 0 To Classes[I].BaseClass.CrossClassSkills.Count-1 Do If Classes[I].BaseClass.CrossClassSkills[J] = Skill Then Begin
     Result := True;
     Exit;
   End;
 End;
End;

Function TCharacter.GetWeapon(Weapon : Integer) : TDNDItem;
var I : Integer;
   Index : Integer;
Const Unarmed : TDNDItem = nil;
Begin
 Result := nil; Index := 0;
 OtherBladeSide := False;

 If Weapon = -1 Then Begin
   if ( Unarmed = nil ) Then
     Unarmed := TDNDItem.Create;
   Unarmed.Size := fRace.Size;
   Unarmed.Kind := ItWeapon;
   Unarmed.WeaponProfNeeded := 'Strike, unarmed';
   Unarmed.Att1DamageDice := UnarmedDice;
   Unarmed.Att1DamageDiceSize := UnarmedDiceSize;
   Unarmed.Att1CritMultiplier := 2;
   Unarmed.name := 'Strike, unarmed';
   Unarmed.CriticalRange := 20;
   Unarmed.WeaponFlags := 256;
   Result := Unarmed;
   Exit;
 End;

 For I := 0 To Equipped[0].Count-1 Do Begin
   If Equipped[0].Items[I].Kind = ItWeapon Then Begin

     { If we are at the first weapon, and its a double sided weapon,
       and its wielded twohanded, and a request is made for the second
       weapon, we set the 'OtherBladeSize' to true and return the first
       blade. (Well, wasn't that confusing? :P) }
     If (I=0) and
        (Weapon = 1 ) and
        (TwoHandedWield > 0) and
       ((Equipped[0].Items[I].WeaponFlags and 1) > 0) Then Begin
       OtherBladeSide := True;
       Result := Equipped[0].Items[I];
       Exit;
     End;

     If Index = Weapon Then Begin
       Result := Equipped[0].Items[I];
       { If the second blade is requested, and we're  }
       Exit;
     End;

     Inc(Index);

   End;
 End;
End;

Function TCharacter.GetArmor : TDNDItem;
var I : Integer;
Begin
 Result := nil;
 For I := 0 To Equipped[0].Count-1 Do Begin
   if Equipped[0].Items[I].Kind = ItArmor Then Begin
    If ( Equipped[0].Items[I].WeaponFlags and 32 ) = 0 THen Begin { If not a shield }
     Result := Equipped[0].Items[I];
     Exit;
    End;
   End;
 End;
End;

Function TCharacter.GetShield : TDNDItem;
var I : Integer;
Begin
 Result := nil;
 For I := 0 To Equipped[0].Count-1 Do Begin
   if Equipped[0].Items[I].Kind = ItArmor Then
    If ( Equipped[0].Items[I].WeaponFlags and 32 ) > 0 THen Begin { If not a shield }
     Result := Equipped[0].Items[I];
     Exit;
   End;
 End;
End;

Function TCharacter.BAB : String;
Var I :Integer;
Begin
  Result := '';
  For I := 0 To MAX_ATTACKS-1 Do if MeleeBase(I) > -1 Then Begin
    If ( I > 0 ) Then Result := Result + '/';
    Result := Result + IntToStr(MeleeBase(I));
  End;
End;

Function TCharacter.BABMeleeBase : Integer;
Begin
 Result := MeleeBase(0)+Abilities.StrModifier+SizeBonus;
End;

Function TCharacter.BABRangeBase : Integer;
Begin
 Result := RangeBase(0)+Abilities.DExModifier+SizeBonus;
End;

Function TCharacter.WeaponTotalAttackBonus(Weapon : Integer) : String;
Var Weap : TDNDItem;
   Bonus : Integer;
    I : Integer;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin

   Bonus := WeaponAttBonusses(Weap,Weapon);

   If ( Weapon = 1 ) and ( DualWield > 0 ) Then Begin
    { Aha! we're dual wielding... This means weapon nr 1 is used for extra
      attacks only! }

      Result := IntToStr(MeleeBase(0) + Bonus);

      { If character has improved two weapon fighting she gets an extra attack
        with it albeit with a -5 penalty. }
      If ImprovedTwoWeaponFighting Then Begin
       Result := Result + '/' + IntToStr(MeleeBase(0) + Bonus -5);
      End;
   End Else If ( Weap.WeaponFlags and 16 ) > 0 Then Begin
     { Is this a ranged weapon? }
     For I := 0 To MAX_ATTACKS-1 Do if RangeBase(I) > -1 Then Begin
        If ( I > 0 ) Then Result := Result + '/';
        Result := Result + IntToStr(RangeBase(I) + Bonus);
     End;

   End Else Begin

    { Use unarmed attackrolls with unarmed attacks }
    If ( Weap.WeaponFlags and 256 ) > 0 Then Begin
       For I := 0 To MAX_ATTACKS-1 Do if UnarmedBase(I) > -1 Then Begin
          If ( I > 0 ) Then Result := Result + '/';
          Result := Result + IntToStr(UnarmedBase(I) + Bonus);
       End;
    End Else Begin
       For I := 0 To MAX_ATTACKS-1 Do if MeleeBase(I) > -1 Then Begin
          If ( I > 0 ) Then Result := Result + '/';
          Result := Result + IntToStr(MeleeBase(I) + Bonus);
       End;
    End;

   End;
 End;
End;

Function TCharacter.GotFinesse(Weapon : TDNDItem ) : Boolean;
Var I : Integer;
   BaseFeat : TBaseFeat;
Begin
 Result := False;
 BaseFeat := nil;
 For I := 0 To World.Feats.Count-1 Do If Lowercase(World.Feats[I].Name) = 'weapon finesse' Then Begin
   BaseFeat := World.Feats[I];
   Break;
 End;
 if BaseFeat <> nil Then Begin
   if HasFeat(BaseFeat, Weapon.WeaponProfNeeded ) <> nil Then Begin
    Result := True;
   End;
 End;
End;

Function TCharacter.IsProficient(Weapon : TDNDItem ) : Boolean;

Function FindFeat(Name : String) : TBaseFeat;
Var I : Integer;
Begin
 Result := nil;
 For I := 0 To World.Feats.Count-1 Do
   If Pos(Name,Lowercase(World.Feats[I].Name))<>0 Then
     Result := World.Feats[I];
End;

Var I : Integer;
   BaseFeat : TBaseFeat;
Begin
 Result := False;
 For I := 0 To FreeWeapons.Count-1 Do If Weapon.WeaponProfNeeded = FreeWeapons[I].NAme Then Begin
    Result := True;
    Break;
 End;

 If Result = False Then
 For I := 0 To World.Feats.Count-1 Do If Pos('weapon proficiency',Lowercase(World.Feats[I].Name))<>0 Then Begin
   BaseFeat := World.Feats[I];
   if HasFeat(BaseFeat, Weapon.WeaponProfNeeded ) <> nil Then Begin
    Result := True;
    Break;
   End;
 End;

 { First check the base feat needed for this weapon. }
{ If Result = False Then
 For I := 0 To World.Weapons.Count-1 Do If Lowercase(Weapon.WeaponProfNeeded)=Lowercase(World.Weapons[I].Name) Then Begin
   BaseFeat := nil;
   CAse Weapon.Kind of
    0 : BaseFeat := FindFeat('simple weapon proficiency');
    1 : BaseFeat := FindFeat('martial weapon proficiency');
    2 : BaseFeat := FindFeat('exotic weapon proficiency');
   End;
   ShowMessage('Basefeat searched! :)'+BaseFeat.Name);
   if HasFeat( BaseFeat, '' ) <> nil Then Begin
    ShowMessage('Basefeat found! :)');
    Result := True;
    Break;
   End;
 End;}

End;

Function TCharacter.HasFocus(Weapon : TDNDItem ) : Boolean;
Var I : Integer;
   BaseFeat : TBaseFeat;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('weapon focus',Lowercase(World.Feats[I].Name))<>0 Then Begin
   BaseFeat := World.Feats[I];
   if HasFeat(BaseFeat, Weapon.WeaponProfNeeded ) <> nil Then Begin
    Result := True;
    Break;
   End;
 End;
End;

Function TCharacter.HasWeaponSpecialization(Weapon : TDNDItem ) : Boolean;
Var I : Integer;
   BaseFeat : TBaseFeat;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('weapon specialization',Lowercase(World.Feats[I].Name))<>0 Then Begin
   BaseFeat := World.Feats[I];
   if HasFeat(BaseFeat, Weapon.WeaponProfNeeded ) <> nil Then Begin
    Result := True;
    Break;
   End;
 End;
End;

 Function TCharacter.HasFeat( NewFeat : TBaseFeat; Parameter : String ) : TTrainedFeat;
 Var I : Integer;
     C : Integer;
 Begin
   Result := nil;

   For I := 0 To LearnedFeats.Count-1 Do
    If ( LearnedFeats.Items[I].BaseFeat = NewFeat ) AND
         ( LearnedFeats.Items[I].Parameter = Parameter ) Then Result := LearnedFeats.Items[I];

   For C := 0 To Classes.Count-1 Do
   For I := 0 To Classes[C].LearnedBonusFeats.Count-1 Do
    If ( Classes[C].LearnedBonusFeats.Items[I].BaseFeat = NewFeat ) AND
         ( Classes[C].LearnedBonusFeats.Items[I].Parameter = Parameter )Then Result := Classes[C].LearnedBonusFeats.Items[I];
 End;

Function TCharacter.WeaponDamBonusses(Weapon : TDNDItem; WeapNr : Integer) : Integer;
LAbel Skip;
Begin
 { Weapon finesse, focus or specialization? }
 Result := 0;

 If Weapon.SpecialBonus > 1 Then Begin
{   0=No Special Bonus
    1=Item is masterwork
    2=Item is enchanted +1 }
   Result := Result + (Weapon.Specialbonus-1);
 End;

 if (( Weapon.WeaponFlags and 128 ) > 0) Then Begin
   { This weapon has been set to ignore strength penalties. }
 End Else if (( Weapon.WeaponFlags and 16 ) > 0) and { Is this a ranged weapon? }
             (( Weapon.WeaponFlags and 64 ) = 0) Then Begin { Make sure this isn't a thrown weapon.. those use melee bonusses }

   { We only substract Strength penalties (no bonusses) to range weapons
     when there actually IS a penalty to substract ( PHB 119 ) }
   if ( Abilities.StrModifier > -1 ) then Begin
     if ( Weapon.MaxCompositeBonus > 0 ) Then Begin
       { Its a composite weapon, we allow bonusses to composite weapons! :) }
       REsult := Result + URANGE(0,Abilities.StrModifier,Weapon.MaxCompositeBonus);
     End;
     goto skip;
   End;

   Result := Result + Abilities.StrModifier;

 End Else Begin
   { Strength bonus to melee attacks }

   { We only add Strength bonusses (no penalties) to melee weapons when there
     actually IS a bonus to add ( PHB 119 ) }
   if ( Abilities.StrModifier < 1 ) then goto skip;

   If ( DualWield > 0 ) and ( WeapNr = 1 ) Then Begin
     { We are dual wielding, and this is the off hand, so only 50% the str bonus }
     Result := (Abilities.StrModifier Div 2);
   End Else Begin
     If Weapon.Size > fRace.Size Then
       { The weapon is being held with two hands! Add 150% str bonus }
       Result := Abilities.StrModifier+(Abilities.StrModifier Div 2)
     Else
       { The weapon is held in the main hand, so 100% str bonus }
       Result := Abilities.StrModifier;
   End;
 End;
 skip:

 if ( HasWeaponSpecialization(Weapon) ) Then Begin
  Result := Result + 2;
 End;

End;

Function TCharacter.WeaponAttBonusses(Weapon : TDNDItem; Weapnr : Integer) : Integer;
Begin
 { Weapon finesse, focus or specialization? }
 if ( GotFinesse(Weapon) ) Then Begin
   Result := Abilities.DexModifier;
 End Else Begin
   Result := Abilities.StrModifier;
 End;

 If Weapon.SpecialBonus > 0 Then Begin
   Result := Result + Weapon.Specialbonus;
{   0=No Special Bonus
    1=Item is masterwork
    2=Item is enchanted +1 }
   If Weapon.Specialbonus > 1 Then Dec(Result);
 End;

 if HasFocus(Weapon) Then Begin
   Result := Result + 1;
 End;

 IF not IsProficient(Weapon) Then Begin
  { Character isn't proficient in this weapon, and gets a -4 penalty. }
  Result := Result - 4;
 End;

 { Calculate Character Size Bonus }
 Result := Result + SizeBonus;

 if DualWield > 0 Then Begin
   If WeapNr = 0 Then Begin
     { Main hand weapon }
     Result := Result + PrimaryPenalties;
   End Else If WeapNr = 1 Then begin
     { Offhand weapon }
     Result := Result + OffHandPenalties;
   End Else Begin
     { Third weapon... not counted in dual wielding }
   End;
 End;

End;

Function TCharacter.WeaponDamage(Weapon : Integer) : String;
Var Weap : TDNDItem;
   Bonus : Integer;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin

    If OtherBladeSide Then
      Result := IntToStr(Weap.Att2DamageDice)+'d'+IntToStr(Weap.Att2DamageDiceSize)
    Else
      Result := IntToStr(Weap.Att1DamageDice)+'d'+IntToStr(Weap.Att1DamageDiceSize);

    Bonus := WeaponDamBonusses(Weap,Weapon);
    if ( Bonus < 0 ) Then Result := Result + IntToStr(Bonus);
    if ( Bonus > 0 ) Then Result := Result + '+' + IntToStr(Bonus);
 End;
End;

Function TCharacter.OffhandWeaponLight : Boolean;
Var Weap : TDNDItem;
Begin
 Result := False;
 Weap := GetWeapon(1);
 If Assigned(Weap) Then Begin
   if ( Weap.Size = 0 ) or ( Weap.Size < fRace.Size ) Then Begin
     Result := True;
   End;
 End;
End;

Function TCharacter.ImprovedTwoWeaponFighting : Boolean;
Var I : Integer;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('improved two-weapon fighting',Lowercase(World.Feats[I].Name))<>0 Then Begin
   Result := HasFeat( World.Feats[I], '') <> nil;
   Break;
 End;
End;

Function TCharacter.TwoWeaponFighting : Boolean;
Var I : Integer;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('two-weapon fighting',Lowercase(World.Feats[I].Name))<>0 Then Begin
   Result := HasFeat( World.Feats[I], '') <> nil;
   Break;
 End;
End;

Function TCharacter.Ambidexterious : Boolean;
Var I : Integer;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('ambidexterity',Lowercase(World.Feats[I].Name))<>0 Then Begin
   Result := HasFeat( World.Feats[I], '') <> nil;
   break;
 End;
End;

Function TCharacter.HasImprovedInitiative : Boolean;
Var I : Integer;
Begin
 Result := False;
 For I := 0 To World.Feats.Count-1 Do If Pos('improved initiative',Lowercase(World.Feats[I].Name))<>0 Then Begin
   Result := HasFeat( World.Feats[I], '') <> nil;
   break;
 End;
End;

Function TCharacter.PrimaryPenalties : Integer;
Begin
 Result := -6;
 If OffHandWeaponLight Then Result := Result + 2;
 If TwoWeaponFighting Then Result := Result + 2;
End;

Function TCharacter.OffhandPenalties : Integer;
Begin
 Result := -10;
 If OffHandWeaponLight Then Result := Result + 2;
 If Ambidexterious Then Result := Result + 4;
 If TwoWeaponFighting Then Result := Result + 2;
End;

Function TCharacter.WeaponCritical(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
     if Weap.CriticalRange < 20 Then Result := IntToStr(Weap.CriticalRange)+'-' Else Result := '';

     If OtherBladeSide Then
       Result := Result + '20/x'+IntToStr(Weap.Att2CritMultiplier)
     Else
       Result := Result + '20/x'+IntToStr(Weap.Att1CritMultiplier);
 End;
End;

Function TCharacter.WeaponRange(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
   if ( Weap.RangeIncrement > 0 ) Then
     Result := IntToStr(Weap.RangeIncrement)
   Else
     Result := '-';
 End;
End;

Function TCharacter.WeaponWeight(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
  Result := IntToStr(Weap.Pounds);
 End;
End;

Function TCharacter.WeaponSize(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
   Case Weap.Size of
     0 : Result := 'Diminutive';
     1 : Result := 'Fine';
     2 : Result := 'Tiny';
     3 : Result := 'Small';
     4 : Result := 'Medium-size';
     5 : Result := 'Large';
     6 : Result := 'Huge';
     7 : Result := 'Gargantuan';
     8 : Result := 'Colossal';
   End;
 End;
End;

Function TCharacter.WeaponType(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
 End;
End;

Function TCharacter.WeaponSpecial(Weapon : Integer) : String;
Var Weap : TDNDItem;
Begin
 Result := '';
 Weap := GetWeapon(Weapon);
 If Assigned(Weap) Then Begin
 End;
End;

Function TCharacter.UnarmedDice : Integer;
Const Dice : array[0..8,0..4] of Integer =
        (( 1,1,1,1,1 ),
         ( 1,1,1,1,1 ),
         ( 1,1,1,1,1 ),
         ( 1,1,1,1,2 ),
         ( 1,1,1,1,1 ),
         ( 1,1,1,2,2 ),
         ( 1,1,2,2,2 ),
         ( 1,2,2,2,4 ),
         ( 2,2,2,4,4 ));
Var I, CalcSpecial : Integer;
Begin
 CalcSpecial := -1;
 For I := 0 To fClasses.Count-1 Do if fClasses[I].BaseClass.MonkUnarmedAttacks > 0 Then Begin
   CalcSpecial := I;
   Break;
 End;

 if ( CalcSpecial <> -1 ) Then Begin
   Case URANGE(0,fClasses[CalcSpecial].Level,20) of
     4..7  : I := 1;
     8..11 : I := 2;
    12..15 : I := 3;
    16..20 : I := 4;
    Else I := 0;
   End;
   Result := Dice[URANGE(0,fRace.Size,8),I];
 End Else Begin
   Result := 1;
 End;
End;

Function TCharacter.SizeWeightAdjuster : Real;
Begin
  Case fRace.Size of
    0 : Result := 0.125; {Fine}
    1 : Result := 0.25; {Diminutive}
    2 : Result := 0.5; {Tiny}
    3 : Result := 0.75; {Small}
    5 : Result := 2; {Large}
    6 : Result := 4; {Huge}
    7 : Result := 8; {Gargantuan}
    8 : Result := 16; {Colossal}
    Else Result := 1; {Medium-size}
  End;
End;

Function TCharacter.UnarmedDiceSize : Integer;
Const Dice : array[0..8,0..4] of Integer =
        ((  1, 2, 3, 4,6 ),
         (  2, 3, 4, 6,8 ),
         (  3, 4, 6, 8,10),
         (  4, 6, 8,10,6 ),
         (  6, 8,10,12,20),
         (  8,10,12, 8,10),
         ( 10,12, 8,10,12),
         ( 12, 8,10,12,8 ),
         (  8,10,12, 8,10));
Var I, CalcSpecial : Integer;
Begin
 CalcSpecial := -1;
 For I := 0 To fClasses.Count-1 Do if fClasses[I].BaseClass.MonkUnarmedAttacks > 0 Then Begin
   CalcSpecial := I;
   Break;
 End;

 if ( CalcSpecial <> -1 ) Then Begin
   Case URANGE(0,fClasses[CalcSpecial].Level,20) of
     4..7  : I := 1;
     8..11 : I := 2;
    12..15 : I := 3;
    16..20 : I := 4;
    Else I := 0;
   End;
   Result := Dice[URANGE(0,fRace.Size,8),I];
 End Else Begin
   { Give Medium creatures 3 damage, other races 2 }
   If fRace.Size < 4 Then Result := 2 Else Result := 3;
 End;
End;

Function TCharacter.LightLoad : Integer;
Const LL : Array[1..29] of Integer = ( 3,6,10,13,16,20,23,26,30,33,38,43,50,58,66,76,87,100,116,133,153,173,200,233,255,306,346,400,466 );

Var Base : Integer;
TmpStr : Integer;
Begin
 if ( Abilities.Str < 30 ) Then Begin
   Result := LL[URANGE(1,Abilities.Str,29)];
 End Else Begin
   Base := LL[20+(Abilities.Str Mod 10)];
{   ShowMessage(IntToStr(20+(Abilities.Str Mod 10)));}
   TmpStr := Abilities.Str;
   While TmpStr > 29 Do Begin
    Dec(TmpStr,10);
    Base := Base * 4;
   End;
   Result := Base;
 End;
 Result := Round(SizeWeightAdjuster*Result);
End;

Function TCharacter.MediumLoad : Integer;
Const LL : Array[1..29] of Integer = ( 6,13,20,26,33,40,46,53,60,66,76,86,100,116,133,153,173,200,233,266,306,346,400,466,533,613,693,800,933 );

Var Base : Integer;
TmpStr : Integer;
Begin
 if ( Abilities.Str < 30 ) Then Begin
   Result := LL[URANGE(1,Abilities.Str,29)];
 End Else Begin
   Base := LL[20+(Abilities.Str Mod 10)];
{   ShowMessage(IntToStr(20+(Abilities.Str Mod 10)));}
   TmpStr := Abilities.Str;
   While TmpStr > 29 Do Begin
    Dec(TmpStr,10);
    Base := Base * 4;
   End;
   Result := Base;
 End;
 Result := Round(SizeWeightAdjuster*Result);
End;

Function TCharacter.HeavyLoad : Integer;
Const LL : Array[1..29] of Integer = ( 10,20,30,40,50,60,70,80,90,100,115,130,150,175,200,230,260,300,350,400,460,520,600,700,800,920,1040,1200,1400 );

Var Base : Integer;
TmpStr : Integer;
Begin
 if ( Abilities.Str < 30 ) Then Begin
   Result := LL[URANGE(1,Abilities.Str,29)];
 End Else Begin
   Base := LL[20+(Abilities.Str Mod 10)];
{   ShowMessage(IntToStr(20+(Abilities.Str Mod 10)));}
   TmpStr := Abilities.Str;
   While TmpStr > 29 Do Begin
    Dec(TmpStr,10);
    Base := Base * 4;
   End;
   Result :=Base;
 End;
 Result := Round(SizeWeightAdjuster*Result);
End;

end.

