/*
    (C) 2001 Anders Holmberg

    This file is part of PocketEmulator.

    PocketClive is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    PocketClive is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with PocketClive; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Author contact information : aagholmberg@yahoo.se

    PocketClive uses code and inspiration from the following sources 
    (see AUTHORS.TXT for more details) :

    FUSE - The Free Unix Spectrum Emulator - (C) 1999-2001 Philip Kendall
		
    EasyCE - PocketPC game programming library - (C) 2001 Jacco Bikker

    MESS - Multi-Emulator Super System - (C) 1998-2001 MESS Team

    Note : Source code files present in the sub directory PocketEmulator\EasyCE
           does not comply with the GPL license. These files are (C) Jacco Bikker
           with the following attachment : 

           "If you want to use EasyCE for commercial applications
            you should let me know. I will ask you a small fee,
            since I spend a lot of time developing this software.
            For freeware products, EasyCE can be used freely."
*/

#include "pocket_clive.h"
#include "pocket_clive_types.h"
#include "pocket_clive_files.h"
#include "spectrum_128k.h"
#include "snapshot.h"
#include "FUSE\Z80\z80.h"
#include "FUSE\Z80\z80_macros.h"
#include <stdio.h>
#include "FUSE\libspectrum\libspectrum.h"

#define BUFFER_SIZE 200000

typedef enum snapshot_type {
  SNAPSHOT_TYPE_SNA,
  SNAPSHOT_TYPE_Z80,
} snapshot_type;

snapshot_type snapshot_identify( const char *filename );
int snapshot_copy_from( libspectrum_snap *snap );

void read_snapshot( char *filename )
{
	FILE *fd;
	libspectrum_snap snap;
	BYTE *buffer;
	int nr_read, error;

	libspectrum_snap_initalise( &snap );

	fd = fopen( filename, "rb" );
	if( fd == NULL )
		return;

	buffer = (BYTE *)malloc( BUFFER_SIZE );
	if( buffer == NULL )
		return;

	nr_read = fread( buffer, 1, BUFFER_SIZE, fd );
	if( nr_read == BUFFER_SIZE )
	{
		/* Buffer is big enough to hold the largest possible
		   snapshot - so if the fread function says that it
		   was able to fill the entire buffer the snapshot
		   file is _not_ a snapshot file */
		free( buffer );
		return;
	}

	fclose( fd );

	switch( snapshot_identify( filename ) )
	{
		case SNAPSHOT_TYPE_SNA:
			error = libspectrum_sna_read( buffer, nr_read, &snap );
			if( error != LIBSPECTRUM_ERROR_NONE )
			{
				show_error_dialog( libspectrum_error_message( libspectrum_error( error ) ) );
				free( buffer );
				return;
			}
			break;

		case SNAPSHOT_TYPE_Z80:
			error = libspectrum_z80_read( buffer, nr_read, &snap );
			if( error != LIBSPECTRUM_ERROR_NONE )
			{
				show_error_dialog( libspectrum_error_message( libspectrum_error( error ) ) );
				free( buffer );
				return;
			}
			break;

		default:
			free( buffer );
			return;
	}

	error = snapshot_copy_from( &snap );
	error = libspectrum_snap_destroy( &snap );
	free( buffer );
}

snapshot_type snapshot_identify( const char *filename )
{
	char filetype[5];
	int i;
 
	/* Emergency exit - Z80 is more common than SNA so return it */
	if( strlen( filename ) < 4 )
		return SNAPSHOT_TYPE_Z80;
	
	strncpy( filetype, filename + ( strlen( filename ) - 4 ), 4 );
	filetype[4] = '\0';

	/* Home made tolower (not supported by M$) */
	for( i=1; i<4; i++ )
		if( ( filetype[i] >= 'A' ) && ( filetype[i] <= 'Z' ) )
			filetype[i] += 'a' - 'A';

	if( strcmp( filetype, ".sna" ) )
		   return SNAPSHOT_TYPE_Z80;
	else 
    return SNAPSHOT_TYPE_SNA;
}

int snapshot_copy_from( libspectrum_snap *snap )
{
  int i; 
	
	switch( snap->machine )
	{
		case LIBSPECTRUM_MACHINE_48:
		  setup_machine( SPECTRUM_48K );
		  break;

		case LIBSPECTRUM_MACHINE_128:
			setup_machine( SPECTRUM_128K );
			break;
	}

  z80_reset();

  z80.halted = 0;

  A  = snap->a ; F  = snap->f ;
  A_ = snap->a_; F_ = snap->f_;

  BC  = snap->bc ; DE  = snap->de ; HL  = snap->hl ;
  BC_ = snap->bc_; DE_ = snap->de_; HL_ = snap->hl_;

  IX = snap->ix; IY = snap->iy; I = snap->i; R = snap->r;
  SP = snap->sp; PC = snap->pc;

  IFF1 = snap->iff1; IFF2 = snap->iff2; IM = snap->im;

  /*spectrum_ula_write( 0x00fe, snap->out_ula );
	*/

  if( machine.machine == SPECTRUM_128K ) 
	{
    spectrum_128k_write_port( 0x7ffd, snap->out_128_memoryport );
  /*  ay_registerport_write( 0xfffd, snap->out_ay_registerport );
    for( i=0; i<15; i++ )
      machine_current->ay.registers[i] = snap->ay_registers[i];
  */
	}

  tstates = snap->tstates;

  for( i=0; i<8; i++ ) {
    if( snap->pages[i] != NULL ) memcpy( ram[i], snap->pages[i], 0x4000 );
  }

  return 0;
}

