/*
    (C) 2001 Anders Holmberg

    This file is part of PocketClive.

    PocketClive is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    PocketClive is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with PocketClive; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Author contact information : aagholmberg@yahoo.se

    PocketClive uses code and inspiration from the following sources 
    (see AUTHORS.TXT for more details) :

    FUSE - The Free Unix Spectrum Emulator - (C) 1999-2001 Philip Kendall
		
    EasyCE - PocketPC game programming library - (C) 2001 Jacco Bikker

    MESS - Multi-Emulator Super System - (C) 1998-2001 MESS Team

    Note : Source code files present in the sub directory PocketClive\EasyCE
           does not comply with the GPL license. These files are (C) Jacco Bikker
           with the following attachment : 

           "If you want to use EasyCE for commercial applications
            you should let me know. I will ask you a small fee,
            since I spend a lot of time developing this software.
            For freeware products, EasyCE can be used freely."
*/

/* --- Includes --------------------------------------------------------------- */

#include "pocket_clive.h"
#include "pocket_clive_files.h"
#include "pocket_clive_types.h"
#include "pocket_clive_display.h"

#include "EasyCE\EasyCE.h"
#include "FUSE\Z80\z80.h"
#include "FUSE\keyboard.h"

#include "Winbase.h"
#include "tchar.h"
#include "Afx.h" 

#include "spectrum_48k.h"
#include "spectrum_128k.h"
#include "snapshot.h"

/* --- Defines ---------------------------------------------------------------- */

#define KEYBOARD								L"keyboard.tga"
#define LOGO										L"CliveLogo.tga"
#define LOAD										L"load.tga"
#define CONFIGURE								L"configure.tga"
#define JOYSTICK							  L"virt_joy.tga"

#define HOMEDIR									 "Roms\\"
#define SPECTRUM_128K_ROM1       "128-0.rom"
#define SPECTRUM_128K_ROM2       "128-1.rom"
#define SPECTRUM_48K_ROM         "48.rom"

#define EMULATION_MODE      1
#define CONFIGURATION_MODE  2
#define DISASSEMBLER_MODE   3
#define LOAD_MODE	          4

#define JOYSTICK_EMULATION 1
#define KEYBOARD_EMULATION 2

#define TARGET_SUB_SCREEN 1
#define FRAME_SUB_SCREEN  2

#define AREA_NONE           0
#define AREA_KEYBOARD				1
#define AREA_CONFIG_SCREEN	2
#define AREA_CONFIG_UP			3
#define AREA_CONFIG_DOWN		4
#define AREA_CONFIG_KEY_1		5
#define AREA_CONFIG_KEY_2		6
#define AREA_CONFIG_KEY_3		7
#define AREA_CONFIG_KEY_4		8
#define AREA_JOY_N					9	
#define AREA_JOY_NE					10
#define AREA_JOY_E					11	
#define AREA_JOY_SE					12
#define AREA_JOY_S					13
#define AREA_JOY_SW					14
#define AREA_JOY_W					15
#define AREA_JOY_NW					16
#define AREA_JOY_N_FIRE			17	
#define AREA_JOY_NE_FIRE		18
#define AREA_JOY_E_FIRE			19	
#define AREA_JOY_SE_FIRE		20
#define AREA_JOY_S_FIRE			21
#define AREA_JOY_SW_FIRE		22
#define AREA_JOY_W_FIRE			23
#define AREA_JOY_NW_FIRE		24
#define AREA_JOY_FIRE       25
#define AREA_SUB_CONFIG_SCREEN	26

/* --- Structs and typedefs --------------------------------------------------- */

struct dir_element_t 
{
		char   name[ 32 ];
		struct dir_element_t *next;
};

typedef struct dir_element_t *dir_element_ptr;

/* --- Global variables ------------------------------------------------------- */

dir_element_ptr first_dir_element = NULL;
dir_element_ptr current_dir_element = NULL;
dir_element_ptr last_dir_element = NULL;

int input_type = KEYBOARD_EMULATION;
int screen_rot = SCREEN_PORTRAIT;

/* CallBack functions */
void cb_config_screen( int xpos, int ypos, int xchar, int ychar );
void cb_sub_config_screen( int xpos, int ypos, int xchar, int ychar );
void cb_load_screen( int xpos, int ypos, int xchar, int ychar );
void cb_config_up( int xpos, int ypos, int xchar, int ychar );
void cb_config_down( int xpos, int ypos, int xchar, int ychar );
void cb_config_key_1( int xpos, int ypos, int xchar, int ychar );
void cb_config_key_2( int xpos, int ypos, int xchar, int ychar );
void cb_config_key_3( int xpos, int ypos, int xchar, int ychar );
void cb_config_key_4( int xpos, int ypos, int xchar, int ychar );
void cb_keyboard( int xpos, int ypos, int xchar, int ychar );
void cb_joy_n( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_ne( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_e( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_se( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_s( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_sw( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_w( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_nw( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_n_fire( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_ne_fire( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_e_fire( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_se_fire( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_s_fire( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_sw_fire( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_w_fire( int xpos, int ypos, int xchar, int ychar );   
void cb_joy_nw_fire( int xpos, int ypos, int xchar, int ychar );  
void cb_joy_fire( int xpos, int ypos, int xchar, int ychar );  

/* Keyboard area layout definitions */
area_t sub_configuration_areas[1] = {
	{ AREA_SUB_CONFIG_SCREEN, 0, 0, 0, 0, cb_sub_config_screen, CLICK_TYPE, 0, 0, 0, 0 }
};

area_t load_areas[4] = { 
	{ AREA_CONFIG_SCREEN,  10,  11, 217, 181, cb_load_screen  , CLICK_TYPE, 0, 0, 12, 10 }, 
	{ AREA_CONFIG_UP,     221, 147, 237, 163, cb_config_up    , CLICK_TYPE, 0, 0, 0, 0 },
	{ AREA_CONFIG_DOWN,   221, 166, 237, 182, cb_config_down  , CLICK_TYPE, 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_1,    1, 286,  32, 319, cb_config_key_1 , CLICK_TYPE, 0, 0, 0, 0 },
};

area_t configuration_areas[2] = { 
	{ AREA_CONFIG_SCREEN,  10,  11, 217, 181, cb_config_screen, CLICK_TYPE, 0, 0, 12, 12 }, 
	{ AREA_CONFIG_KEY_3,   65, 286,  96, 319, cb_config_key_3 , CLICK_TYPE, 0, 0, 0, 0 },
};

area_t emul_keyboard_areas[14] = { 
	{ AREA_KEYBOARD,        1, 201, 236, 277, cb_keyboard    , PENDOWN_TYPE, 0, 0, 0, 0 }, 
	{ AREA_KEYBOARD,      145, 281, 236, 296, cb_keyboard    , PENDOWN_TYPE, 0, 0, 0, 0 }, 
	{ AREA_JOY_N,					 80,   0, 160, 64,  cb_joy_n       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_NE,			  160,   0, 240, 64,  cb_joy_ne      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_E,					160,  64, 240, 128, cb_joy_e       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_SE,				160, 128, 240, 192, cb_joy_se      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_S,					 80, 128, 160, 192, cb_joy_s       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_SW,				  0, 128,  80, 192, cb_joy_sw      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_W,					  0,  64,  80, 128, cb_joy_w       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_NW,				  0,   0,  80, 64,  cb_joy_nw      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_1,    1, 286,  32, 319, cb_config_key_1, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_2,   33, 286,  64, 319, cb_config_key_2, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_3,   65, 286,  96, 319, cb_config_key_3, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_4,   97, 286, 128, 319, cb_config_key_4, CLICK_TYPE  , 0, 0, 0, 0 },
};

area_t emul_joystick_areas[21] = { 
	{ AREA_JOY_N,					150, 197, 189, 236, cb_joy_n       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_NE,			  189, 197, 228, 236, cb_joy_ne      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_E,					189, 236, 228, 275, cb_joy_e       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_SE,				189, 275, 228, 314, cb_joy_se      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_S,					150, 275, 189, 314, cb_joy_s       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_SW,				111, 275, 150, 314, cb_joy_sw      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_W,					111, 236, 150, 275, cb_joy_w       , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_NW,				111, 197, 150, 236, cb_joy_nw      , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_JOY_FIRE,      150, 236, 189, 275, cb_joy_fire    , PENDOWN_TYPE, 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_1,    1, 286,  32, 319, cb_config_key_1, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_2,   33, 286,  64, 319, cb_config_key_2, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_3,   65, 286,  96, 319, cb_config_key_3, CLICK_TYPE  , 0, 0, 0, 0 },
	{ AREA_CONFIG_KEY_4,   97, 286, 128, 319, cb_config_key_4, CLICK_TYPE  , 0, 0, 0, 0 },
};

int platform = SPECTRUM_48K;
int mode = EMULATION_MODE;

char snapshot[50][32];
int  number_of_snapshots = 0;
int  Terminated = FALSE;

char copyright[5][32] = {
	"                               ", 
	"                               ",
	"           Release 1B          ",
	"   (C) 2001 Anders Holmberg    ", 
	"                               " };

char configure[9][32] = {
	"TARGET     SPECTRUM 48K", 
	"JOYSTICK   (KEMPSTON)",
	"FRAME SKIP (NOT IMPL)",
	"SCREEN ROT (NOT IMPL)",
	"SOUND VOL  (NOT IMPL)", 
	"",
	"RESET EMULATOR", 
	"EXIT EMULATOR",
	"ABOUT"
};

char target_platform[2][32] = {
	"SPECTRUM 48K", 
	"SPECTRUM 128K"
};

char joystick_name[3][32] = {
	"KEMPSTON", 
	"SINCLAIR", 
	"CURSOR"
};

char frameskip_text[][32] = {
	"NONE",
	"1 FRAME",
	"2 FRAMES", 
	"3 FRAMES"
};


BYTE rom[2][16384];
BYTE ram[8][16384];

machine_t machine;

get_area_t get_area;

int top_dir_index = 0;

int conf_frame_skip = 0;

int sub_screen = 0;

int flash_cntr = 0; 
int flash = 0;

#if 0

FSOUND_STREAM *sound_stream;
FSOUND_SAMPLE *sound_sample;

#endif

int joystick_fired = 0;
int joystick_moved = 0;

int high_tstates = 3500;
int low_tstates  = 3500;
int local_high_tstates = 3500;
int local_low_tstates = 3500;

unsigned int sound_freq = 0;
unsigned int sound_interval = 0;

char root_path[ MAX_PATH ];
WORD mod_path[ MAX_PATH ];

/* --- Function declarations -------------------------------------------------- */

void dir_snapshot( void );
void show_copyright( void );

void setup_sub_screen( char items[][32], int nr_items, int ypos );

void add_dir_element( char *name_ptr );
dir_element_ptr get_first_dir_element( void );
dir_element_ptr get_next_dir_element( void );
void delete_first_dir_element( void );
dir_element_ptr get_sel_dir_element( int index );
int nr_dir_elements( void );

void check_keyboard( void );

area_ptr get_emulation_keyboard_area( int xpos, int ypos );
area_ptr get_configuration_area( int xpos, int ypos );
area_ptr get_load_area( int xpos, int ypos );
area_ptr get_emulation_joystick_area( int xpos, int ypos );
area_ptr get_sub_configuration_area( int xpos, int ypos );


void set_emulation_mode( void );

void set_root_path( void );
void lbuild_absolute_filename( WORD *output, WORD *filename );
void build_absolute_filename( char *output, char *filename );

#if 0

void sound_init( void );
void build_sound( signed char  *sound_buffer, 
								  unsigned int size, 
									unsigned int high_interval, 
									unsigned int low_interval );
void update_sound_output( void );

signed char __cdecl sound_callback(FSOUND_STREAM *stream, void *buff, int len, int param);

#endif

/* --- Implementation --------------------------------------------------------- */

void __cdecl main()
{
	int game_nr = 0;
	int sub_menu = -1;
	int elapsed_time = 0;
	int nr_of_frames = 1;
	int i;

	set_root_path();

  show_copyright();

	setup_machine( SPECTRUM_48K );
	
	keyboard_init();
	z80_init();	
	z80_reset();

	all_is_dirty();

	input_type = KEYBOARD_EMULATION;
	get_area   = get_emulation_keyboard_area;

	while( !Terminated )
	{
		check_keyboard();

		if( mode == EMULATION_MODE )
		{
			/*
			_itoa( msecs(), print_string, 10 );
			bar( 220, 310, 239, 319, 0xffff );
			print( print_string, 221, 311, BLACK );
			*/

			resettimer();

			for( i=0; i<(conf_frame_skip+1); i++ )
			{
				z80_do_opcodes( 49888 );
				z80_interrupt();
				z80_do_opcodes( 20000 );
			}
				
		show_spectrum_display();

		/* Release 2 stuff ---
			if( screen_rot == SCREEN_PORTRAIT )
			  show_spectrum_display();
			else
  			show_spectrum_display_landscape_right();
		*/
			update();
				
			flash_cntr += ( conf_frame_skip + 1 );
			if( flash_cntr >= 16 )
			{
				flash_cntr = 0;
				if( flash )
					flash = 0;
				else
					flash = 1;
				flash_is_dirty();
				all_is_dirty();
			}

			while( msecs() < ( 20 *(conf_frame_skip+1)) ) 
				update( FALSE );  /* do not update the display! */

/*			update_sound_output(); */

/*
			high_tstates = local_high_tstates;
			low_tstates  = local_low_tstates;
*/
		}
		else
		{
 			update();
		}
	}

	/* FSOUND_Close(); */

}

/* --- setup_machine ---
 *
 * Setup PocketClive for the targeted machine
 *
 * ---
 */
void setup_machine( int choice )
{
	char filename[ MAX_PATH ];

	switch( choice )	
	{
		case SPECTRUM_48K:
			build_absolute_filename( filename, SPECTRUM_48K_ROM );
			load( filename, 0x4000, rom[ 0 ] );
			machine.read_memory  = spectrum_48k_read_memory;
			machine.write_memory = spectrum_48k_write_memory;
			machine.read_port    = spectrum_48k_read_port;
			machine.write_port   = spectrum_48k_write_port;
			machine.screen_page  = 5;
			machine.machine 		 = SPECTRUM_48K;
			break;		

		case SPECTRUM_128K:
			build_absolute_filename( filename, SPECTRUM_128K_ROM1 );
			load( filename, 0x4000, rom[ 0 ] );
			build_absolute_filename( filename, SPECTRUM_128K_ROM2 );
			load( filename, 0x4000, rom[ 1 ] );
			machine.read_memory  = spectrum_128k_read_memory;
			machine.write_memory = spectrum_128k_write_memory;
			machine.read_port    = spectrum_128k_read_port;
			machine.write_port   = spectrum_128k_write_port;
			machine.ram_page     = 0;
			machine.rom_page     = 0;
			machine.screen_page  = 5;
			machine.machine		   = SPECTRUM_128K;
			break;
	}
}

/* --- show_error_dialog ---
 *
 * Show a "modal" error dialog box
 *
 * ---
 */
void show_error_dialog( char *error_text )
{
  show_dialog( "ERROR MESSAGE", error_text );
}

/* --- show_dialog ---
 *
 * Show a "modal" dialog box
 *
 * ---
 */
void show_dialog( char *title_text, char *text )
{
	unsigned short *p = (unsigned short *)getbuffer();
	unsigned short *buffer;
  unsigned int i;
	char buffer_text[ 32 ];

	buffer = (unsigned short *)malloc( 100 * 240 * sizeof( unsigned short ) );
	p += 240 * 40;
	
	memcpy( buffer, p, 100 * 240 * sizeof( unsigned short ) );

	bar( 30, 40, 210, 139, 0xffff );
	box( 30, 40, 210, 139, BLACK );
	centre( title_text, 50, BLACK );
	
	for( i=0; i<( strlen( text ) / 20) + 1; i++ )
	{
		memcpy( buffer_text, text + 20 * i, 30 );
		buffer_text[ 20 ] = '\0';
		centre( buffer_text, 65 + 12 * i, BLACK );
	}

	centre( "CLICK TO CONTINUE", 125, BLACK );
	
	/* Wait for user click */
	while ( clicked() ) 
		update();
	while ( !clicked() ) 
		update();

	p = (unsigned short *)getbuffer();
	p += 240*40; 
	memcpy( p, buffer, 100 * 240 * sizeof( unsigned short ) );
	free( buffer );

}

/* --- show_copyright ---
 *
 * Show the PocketClive copyright screen
 *
 * ---
 */
void show_copyright( void )
{
	int i;
	WORD filename[ MAX_PATH ];

	/* Show copyright screen */
	clear( BLACK );

	unsigned short * p = (unsigned short*)getbuffer();
	p += 40*240;
  lbuild_absolute_filename( filename, LOGO );
	ATgaFile::load( filename, 240, 65, p );

	for( i=0; i<5; i++ )
		print( copyright[ i ], 0, i * 25 + 70, 0xffff );

	/* Wait for user click */
	while ( clicked() ) 
		update();
	while ( !clicked() ) 
		update();

	/* Load keyboard image */
	clear( 0 );

	p = (unsigned short*)getbuffer();
	p += 192*240;
	lbuild_absolute_filename( filename, KEYBOARD );
  ATgaFile::load( filename, 240, 128, p );

}

/* --- Hardware functions used by the Z80 CPU emulator ---
 *
 * readport      IN 
 * writeport     OUT 
 * readbyte      memory access
 * writebyte     memory access
 *
 * ---
 */
BYTE readport(WORD port)
{
	return machine.read_port( port );
}

void writeport(WORD port, BYTE b)
{
	machine.write_port( port, b );
}

BYTE readbyte(WORD address)
{
	return machine.read_memory( address );
}

void writebyte(WORD address, BYTE b)
{
	machine.write_memory( address, b );

	if( ( address >= 0x4000 ) && ( address < 0x5B00 ) )	
	{
		/* Screen pixel */
		if( address < 0x5800 )
			dirty_rec[ ( ( address & 0x1800 ) >> 8 ) + ( ( address & 0x00e0 ) >> 5 ) ][ address & 0x001f ] = 1;
		/* Screen attribute */
		else
			dirty_rec[ ( address & 0x03ff ) >> 5 ][ address & 0x001f ] = 1;
	}
}

int tape_trap(void)
{
	return 0;
}

/* --- lbuild_absolute_filename ---
 *
 * build an abosulte filename (with path) by 
 * adding the path to the PocketClive executable
 *
 * This is the unicode (long) version
 *
 * ---
 */
void lbuild_absolute_filename( WORD *output, WORD *filename )
{
	lstrcpy( output, mod_path );
	lstrcat( output, filename );
}

/* --- build_absolute_filename ---
 *
 * build an abosulte filename (with path) by 
 * adding the path to the PocketClive executable
 *
 * This is the char version
 *
 * ---
 */
void build_absolute_filename( char *output, char *filename )
{
	WideCharToMultiByte(CP_ACP, 0, 
		mod_path, wcslen(mod_path) + 1, 
		output, MAX_PATH - 1, 
		NULL, NULL);
	strcat( output, filename );
}

/* --- set_root_path ---
 * 
 * Sets to root of the Roms directory
 *
 * ---
 */
void set_root_path( void )
{
	DWORD number_bytes;

	number_bytes = GetModuleFileName( NULL, mod_path, MAX_PATH + 1 );
	/* Remove PocketClive.exe */
	*( mod_path + number_bytes - 15 ) = 0;
}

/* --- dir_snapshot ---
 *
 * Fill the dir_element linked list with all *.z80 and *.sna files 
 * in the \\Program Files\\PocketClive\\Roms\\ directory.
 *
 * ---
 */
void dir_snapshot( void )
{
	WIN32_FIND_DATA FileData;  
	HANDLE					hSearch;            
	char						temp_name[ 32 ];
	WORD						file_name[ MAX_PATH ];

	BOOL bFinished = FALSE;

	/* Free old entries in the list */
	while( first_dir_element != NULL )
		delete_first_dir_element();

	/* Now find *.sna files */
	lstrcpy( file_name, mod_path );
	lstrcat( file_name, TEXT("Roms\\*.sna" ) );
	
	hSearch = FindFirstFile( file_name, &FileData );
	if( hSearch != INVALID_HANDLE_VALUE )
	{
 		WideCharToMultiByte(CP_ACP, 0, FileData.cFileName, -1, temp_name, 32, NULL, NULL);
		add_dir_element( temp_name );

		while( !bFinished )
		{
			if (!FindNextFile (hSearch, &FileData))
			{
				bFinished = TRUE;
			}
			else
			{
				WideCharToMultiByte(CP_ACP, 0, FileData.cFileName, -1, temp_name, 32, NULL, NULL);
				add_dir_element( temp_name );
			}
		}

		FindClose( hSearch );
	}

	/* Now find *.z80 */
	bFinished = FALSE;
		
	lstrcpy( file_name, mod_path );
	lstrcat( file_name, TEXT("Roms\\*.z80" ) );

	hSearch = FindFirstFile( file_name, &FileData );
	if( hSearch == INVALID_HANDLE_VALUE )
	{
		return;
	}

 	WideCharToMultiByte(CP_ACP, 0, FileData.cFileName, -1, temp_name, 32, NULL, NULL);
	add_dir_element( temp_name );

	while( !bFinished )
	{
		if (!FindNextFile (hSearch, &FileData))
		{
			bFinished = TRUE;
		}
		else
		{
	    WideCharToMultiByte(CP_ACP, 0, FileData.cFileName, -1, temp_name, 32, NULL, NULL);
	    add_dir_element( temp_name );
		}
  }

  if (!FindClose (hSearch))
  {
		return;
  }
} 

/* --- Linked list handling routines ---
 *
 * add_dir_element          - add an element at the end of the list
 * get_first_dir_element    - get the first element in the list (also set up
 *                            global variable to be used by get_next... function)
 * get_next_dir_element     - get the "next" element in the list
 * get_sel_dir_element      - get the specified element in the list
 * nr_dir_elements          - how many elements are there in the list
 * delete_first_dir_element - delete the first element in the list
 * 
 * ---
 */
void add_dir_element( char *name_ptr )
{
	dir_element_ptr dir_element;

	dir_element = (dir_element_ptr)malloc( sizeof( dir_element_t ) );
	strcpy( dir_element->name, name_ptr );

	if( first_dir_element == NULL )
	{
		first_dir_element       = dir_element;
		first_dir_element->next = NULL;
		last_dir_element        = dir_element;
	}
	else
	{
		dir_element->next      = NULL;
		last_dir_element->next = dir_element;
		last_dir_element       = dir_element;
	}
}

dir_element_ptr get_first_dir_element( void )
{
	current_dir_element = first_dir_element;

	return first_dir_element;
}

dir_element_ptr get_next_dir_element( void )
{
	if( current_dir_element != NULL )
		current_dir_element = current_dir_element->next;

	return current_dir_element;
}

void delete_first_dir_element( void )
{
	dir_element_ptr temp_dir_element;

	if( first_dir_element != NULL )
	{
		temp_dir_element  = first_dir_element;
		first_dir_element = first_dir_element->next;
		free( temp_dir_element );
	}
}

dir_element_ptr get_sel_dir_element( int index )
{
	dir_element_ptr dir_element;
	int i;

	dir_element = get_first_dir_element();

	if( index == 0 )
		return dir_element;

	for( i=0; i<index; i++ )
	{
		dir_element = get_next_dir_element();
		if( dir_element == NULL )
			return NULL;
	}

	return dir_element;

}

int nr_dir_elements( void )
{
	dir_element_ptr dir_element;
	int nr = 0;

	dir_element = get_first_dir_element();

	while( dir_element != NULL )
	{
		dir_element = dir_element->next;
		nr++;
	}

	return nr;
}

/* --- Screen Area handling routines
 *
 * Routines that handle the conversion between pixel 
 * position and screen area
 *
 */
area_ptr get_sub_configuration_area( int xpos, int ypos )
{
	int i;

	for( i=0; i<2; i++ )
	{
		if( ( xpos >= sub_configuration_areas[ i ].x1 ) && 
			  ( xpos <= sub_configuration_areas[ i ].x2 ) && 
			  ( ypos >= sub_configuration_areas[ i ].y1 ) && 
			  ( ypos <= sub_configuration_areas[ i ].y2 ) )
			return &sub_configuration_areas[ i ]; 
	}

	return NULL;
}

area_ptr get_configuration_area( int xpos, int ypos )
{
	int i;

	for( i=0; i<2; i++ )
	{
		if( ( xpos >= configuration_areas[ i ].x1 ) && 
			  ( xpos <= configuration_areas[ i ].x2 ) && 
			  ( ypos >= configuration_areas[ i ].y1 ) && 
			  ( ypos <= configuration_areas[ i ].y2 ) )
			return &configuration_areas[ i ]; 
	}

	return NULL;
}

area_ptr get_load_area( int xpos, int ypos )
{
	int i;

	for( i=0; i<4; i++ )
	{
		if( ( xpos >= load_areas[ i ].x1 ) && 
			  ( xpos <= load_areas[ i ].x2 ) && 
			  ( ypos >= load_areas[ i ].y1 ) && 
			  ( ypos <= load_areas[ i ].y2 ) )
			return &load_areas[ i ]; 
	}

	return NULL;
}

area_ptr get_emulation_keyboard_area( int xpos, int ypos )
{
	int i;

	for( i=0; i<14; i++ )
	{
		if( ( xpos >= emul_keyboard_areas[ i ].x1 ) && 
			  ( xpos <= emul_keyboard_areas[ i ].x2 ) && 
			  ( ypos >= emul_keyboard_areas[ i ].y1 ) && 
			  ( ypos <= emul_keyboard_areas[ i ].y2 ) )
			return &emul_keyboard_areas[ i ]; 
	}

	return NULL;
}

area_ptr get_emulation_joystick_area( int xpos, int ypos )
{
	int i;

	for( i=0; i<21; i++ )
	{
		if( ( xpos >= emul_joystick_areas[ i ].x1 ) && 
			  ( xpos <= emul_joystick_areas[ i ].x2 ) && 
			  ( ypos >= emul_joystick_areas[ i ].y1 ) && 
			  ( ypos <= emul_joystick_areas[ i ].y2 ) )
			return &emul_joystick_areas[ i ]; 
	}

	return NULL;
}

void set_emulation_mode( void )
{
	mode = EMULATION_MODE;
	if( input_type == JOYSTICK_EMULATION )
		get_area = get_emulation_joystick_area;
	else
		get_area = get_emulation_keyboard_area;
}

/* --- setup_sub_screen ---
 *
 * Setup the sub menu screens used during configuration...
 *
 * ---
 */
void setup_sub_screen( char items[][32], int nr_items, int ypos )
{
	int i;
	
	bar( 95, ypos - 10, 228, ypos + 5 + nr_items * 10 , 0xffff );
	box( 95, ypos - 10, 228, ypos + 5 + nr_items * 10 , BLACK );
	
	for( i=0; i<nr_items; i++ )
	  print( items[i], 104, ypos + i * 10, BLACK );

	get_area = get_sub_configuration_area;
	
	sub_configuration_areas[0].x1       = 95;
	sub_configuration_areas[0].y1       = ypos - 10;
	sub_configuration_areas[0].x2       = 228;
	sub_configuration_areas[0].y2       = ypos + 5 + nr_items * 10;
	sub_configuration_areas[0].y_mult   = 10;
	sub_configuration_areas[0].y_offset = 10;
}

/* --- check_keyboard ---
 *
 * Check if the user has pressed the screen, call the correct
 * callback function if so is the case. 
 *
 * ---
 */
void check_keyboard( void )
{
	area_ptr area;
	int      xchar, ychar;

	joystick_fired = 0;
	joystick_moved = 0;

	if( clicked() )
	{
		area = get_area( clickxpos(), clickypos() );
		if( area != NULL )
		{
			if( area->clicktype == CLICK_TYPE )
			{
				if( area->x_mult != 0 )
					xchar = ( clickxpos() - area->x1 - area->x_offset ) / area->x_mult;
				else
					xchar = 0;

				if( area->y_mult != 0 )
					ychar = ( clickypos() - area->y1 - area->y_offset ) / area->y_mult;
				else
					ychar = 0;

				area->area_clicked( clickxpos(), clickypos(), xchar, ychar );
			}
		}
	}
	
	if( pendown() )
	{
		area = get_area( clickxpos(), clickypos() );
		if( area != NULL )
		{
			if( area->clicktype == PENDOWN_TYPE )
			{

				if( area->x_mult != 0 )
					xchar = ( clickxpos() - area->x1 - area->x_offset ) / area->x_mult;
				else
					xchar = 0;

				if( area->y_mult != 0 )
					ychar = ( clickypos() - area->y1 - area->y_offset ) / area->y_mult;
				else
					ychar = 0;

				area->area_clicked( clickxpos(), clickypos(), xchar, ychar );
			}
		}
	}
	else
	{
		/* default handling when no pen is pressed */

		keyboard_up();

		if( input_type == KEYBOARD_EMULATION )
		{

			if( symbol() )
				bar( 230, 194, 235, 199, 0xffff );
			else
				bar( 230, 194, 235, 199, BLACK );

			if( caps() )
				bar( 220, 194, 225, 199, 0xffff );
			else
				bar( 220, 194, 225, 199, BLACK );
		}
	}

	if( buttonleft() )
	{
		cb_joy_w(0,0,0,0);
	}

	if( buttonup() )
	{
		cb_joy_n(0,0,0,0);
	}

	if( buttonright() )
	{
		cb_joy_e(0,0,0,0);
	}

	if( buttondown() )
	{
		cb_joy_s(0,0,0,0);
	}

	if( buttonA() || buttonB() || buttonC() || buttonD() )
	{
		cb_joy_fire(0,0,0,0);
	}

	if( !joystick_moved )
			joystick_reset_movement();

	if( !joystick_fired )
		joystick_reset_fire();

}

/* --- Callback Functions --- 
 *
 * Callback functions that are called by the check_keyboard
 * function when the user has pressed the "correct" area
 *
 * ---
 */
void cb_load_screen( int xpos, int ypos, int xchar, int ychar )
{
	dir_element_ptr dir_element;
	char snapshot_name[ MAX_PATH ];

	dir_element = get_sel_dir_element( top_dir_index + ychar );

	if( dir_element != NULL )
	{
		build_absolute_filename( snapshot_name, "Roms\\" );
		strcat( snapshot_name, dir_element->name );
	  read_snapshot( snapshot_name );
		all_is_dirty();
	}

	set_emulation_mode();
}

void cb_config_up( int xpos, int ypos, int xchar, int ychar )
{
	int i;
	dir_element_ptr dir_element;

	if( top_dir_index < 15 )
		top_dir_index = 0;
	else
	{
		top_dir_index -= 15;
	}

	bar( 10, 11, 217, 181, 0xffff );
	
	dir_element = get_sel_dir_element( top_dir_index );

	if( dir_element == NULL )
		return;

	for( i=0; i<15; i++ )
	{
		print( dir_element->name, 20, i * 10 + 24, BLACK );
		dir_element = get_next_dir_element();
		if( dir_element == NULL )
			break;
	}
}

void cb_config_down( int xpos, int ypos, int xchar, int ychar )
{
	int i;
	dir_element_ptr dir_element;

	if( top_dir_index < ( nr_dir_elements() - 15 ) )
		top_dir_index += 15;

	bar( 10, 11, 217, 181, 0xffff );
	
	dir_element = get_sel_dir_element( top_dir_index );

	if( dir_element == NULL )
		return;

	for( i=0; i<15; i++ )
	{
		print( dir_element->name, 20, i * 10 + 24, BLACK );
		dir_element = get_next_dir_element();
		if( dir_element == NULL )
			break;
	}
}

void cb_sub_config_screen( int xpos, int ypos, int xchar, int ychar )
{
	if( sub_screen == TARGET_SUB_SCREEN )
	{
		if( ychar == 0 )
			platform = SPECTRUM_48K;
		else if( ychar == 1 )
			platform = SPECTRUM_128K;

		setup_machine( platform );
		z80_reset();
	}
	else if( sub_screen == FRAME_SUB_SCREEN )
	{
		conf_frame_skip = ychar;	
	}

	/* uugly programing - 
	 * I just want to go back to the configuration screen 
	 */
	mode = EMULATION_MODE;
	cb_config_key_3(0,0,0,0);
}

void cb_config_screen( int xpos, int ypos, int xchar, int ychar )
{
	if( ychar == 0 )
	{
		sub_screen = TARGET_SUB_SCREEN;
		setup_sub_screen( target_platform, 2, 24 ); 
	}
	else if( ychar == 2 )
	{
		sub_screen = FRAME_SUB_SCREEN;
		setup_sub_screen( frameskip_text, 4, 40 ); 
	}
  else if( ychar == 6 ) 	
	{
		z80_reset();	
	}
	else if( ychar == 7 )
	{
		Terminated = TRUE;
	}
	else if( ychar == 8 )
	{
		show_dialog( "ABOUT POCKETCLIVE", "     RELEASE 1B          2001-10-15     (C) ANDERS HOLMBERG" );
	}
}

void cb_keyboard( int xpos, int ypos, int xchar, int ychar )
{
  keyboard_down( xpos, ypos );
}

void cb_joy_n( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 8 );
}
   
void cb_joy_ne( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 9 );
}
  
void cb_joy_e( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 1 );
}   

void cb_joy_se( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 5 );
}
  
void cb_joy_s( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 4 );
}
   
void cb_joy_sw( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 6 );
}  

void cb_joy_w( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 2 );
}
   
void cb_joy_nw( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_set( 10 );
}  

void cb_joy_n_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 24 );
}
   
void cb_joy_ne_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 25 );
}
  
void cb_joy_e_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 17 );
}   

void cb_joy_se_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 21 );
}
  
void cb_joy_s_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 20 );
}
   
void cb_joy_sw_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 22 );
}  

void cb_joy_w_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 18 );
}
   
void cb_joy_nw_fire( int xpos, int ypos, int xchar, int ychar )
{
	joystick_moved = 1;
  joystick_fired = 1;
  joystick_set( 26 );
}  

void cb_joy_fire( int xpos, int ypos, int xchar, int ychar )
{
  joystick_fired = 1;
  joystick_set( 16 );
}  

void cb_config_key_1( int xpos, int ypos, int xchar, int ychar )
{
	unsigned short *p;
	dir_element_ptr dir_element;
	int i;
	WORD filename[ MAX_PATH ];

	if( mode == EMULATION_MODE )	
	{
		mode     = LOAD_MODE;
		get_area = get_load_area;
		
		p = (unsigned short*)getbuffer();
		lbuild_absolute_filename( filename, LOAD );
		ATgaFile::load( filename, 240, 192, p );

		top_dir_index = 0;
		dir_snapshot();

		dir_element = get_first_dir_element();

		if( dir_element != NULL )
		{	
			for( i=0; i<15; i++ )
			{
				print( dir_element->name, 20, i * 10 + 24, BLACK );
				dir_element = get_next_dir_element();
				if( dir_element == NULL )
					break;
			}
		}
	}
	else if( mode == LOAD_MODE )
	{
		all_is_dirty();
		set_emulation_mode();
	}
}

void cb_config_key_2( int xpos, int ypos, int xchar, int ychar )
{
	unsigned short *p;
	WORD filename[ MAX_PATH ];

	if( input_type == JOYSTICK_EMULATION )
	{
		p = (unsigned short*)getbuffer();
		p += 192*240;
		lbuild_absolute_filename( filename, KEYBOARD );
		ATgaFile::load( filename, 240, 128, p );

		input_type = KEYBOARD_EMULATION;
		get_area   = get_emulation_keyboard_area;
	}
	else
	{
		p = (unsigned short*)getbuffer();
		p += 192 * 240;
		lbuild_absolute_filename( filename, JOYSTICK );
		ATgaFile::load( filename, 240, 128, p );

		input_type = JOYSTICK_EMULATION;
		get_area   = get_emulation_joystick_area;
	}
}

void cb_config_key_3( int xpos, int ypos, int xchar, int ychar )
{
	unsigned short *p;
	int i;
	WORD filename[ MAX_PATH ];


	if( mode == EMULATION_MODE )
	{
		mode = CONFIGURATION_MODE;
		get_area = get_configuration_area;

		p = (unsigned short*)getbuffer();
		lbuild_absolute_filename( filename, CONFIGURE );
		ATgaFile::load( filename, 240, 192, p );

		if( platform == SPECTRUM_48K )
			strcpy( &configure[0][11], "SPECTRUM 48K" );
		else if( platform == SPECTRUM_128K )
			strcpy( &configure[0][11], "SPECTRUM 128K" );
		
		if( conf_frame_skip == 0 )
			strcpy( &configure[2][11], "NONE" );
		else if( conf_frame_skip == 1 )
			strcpy( &configure[2][11], "1 FRAME" );
		else if( conf_frame_skip == 2 )
			strcpy( &configure[2][11], "2 FRAMES" );
		else if( conf_frame_skip == 3 )
			strcpy( &configure[2][11], "3 FRAMES" );

		for( i=0; i<9; i++ )
			print( configure[ i ], 20, i * 12 + 24, BLACK );
	}
	else if( CONFIGURATION_MODE )	
	{
		all_is_dirty();
		set_emulation_mode();
	}

}

void cb_config_key_4( int xpos, int ypos, int xchar, int ychar )
{
	if( screen_rot == SCREEN_PORTRAIT )
		screen_rot = SCREEN_LANDSCAPE_RIGHT;
	else if( screen_rot == SCREEN_LANDSCAPE_RIGHT )
		screen_rot = SCREEN_LANDSCAPE_LEFT;
	else
		screen_rot = SCREEN_PORTRAIT;
}

#if 0

/* --- sound support functions ---
 *
 * ---
 */
void sound_init( void )
{
  if( !FSOUND_Init( 40000, 32, FSOUND_INIT_GLOBALFOCUS ) )
	{
		show_error_dialog( FMOD_ErrorString(FSOUND_GetError()) );
		return;
	}

/*	sound_stream =  FSOUND_Stream_Create( sound_callback, 1000, 
									FSOUND_LOOP_NORMAL | FSOUND_8BITS | FSOUND_MONO | FSOUND_2D, 
									100000, 0 );

	if( ! sound_stream )
	{
		show_error_dialog( FMOD_ErrorString(FSOUND_GetError()) );
		return;
	}

	if( !FSOUND_Stream_Play( FSOUND_FREE, sound_stream ) )
	{
		show_error_dialog( FMOD_ErrorString(FSOUND_GetError()) );
		return;
	}

*/

	sound_sample = FSOUND_Sample_Alloc( FSOUND_FREE, 1000, FSOUND_LOOP_NORMAL | FSOUND_8BITS | FSOUND_MONO | FSOUND_2D, 
											                100000, 255, 128, 255 );

	if( !sound_sample )
	{
		show_error_dialog( FMOD_ErrorString(FSOUND_GetError()) );
		return;
	}

	signed char *ptr1, *ptr2;
	unsigned int  len1, len2;

	if( FSOUND_Sample_Lock( sound_sample, 0, 1000, (void **)&ptr1, (void **)&ptr2, &len1, &len2 ) )
	{
		
		build_sound( ptr1, len1, 50, 50 );
		FSOUND_Sample_Unlock( sound_sample, ptr1, ptr2, len1, len2 );
		FSOUND_PlaySound( FSOUND_FREE, sound_sample ); 
		FSOUND_SetFrequency( FSOUND_ALL, 44000 );
	}
	else
	{
		show_error_dialog( FMOD_ErrorString(FSOUND_GetError()) );
		return;
	}
}

signed char __cdecl sound_callback(FSOUND_STREAM *stream, void *buff, int len, int param)
{
	int buffer_index, samples;
	signed char *sound_buffer = (signed char *)buff;

	/* --- Spectrum 48k sound support ---
	 *
	 *  A Spectrum 48k executes 69888 tstates in 20 ms (that is 50Hz), so to
	 *  convert from [ tstates ] to [ nr of samples ] I have to calculate the
	 *  difference between the Spectrum frequency and the samplerate...
	 *
	 *    difference = 3500000 Hz / 100000 Hz = 35 
	 *
	 *  Converstion [ samples / tstates ] = x / 35
	 *
	 */

	buffer_index = 0;

	if( low_tstates >= 69888 )
	{
		memset( (signed char*)sound_buffer, 0, 1000 );
		return TRUE;
	}

	while( buffer_index < 1000 )
	{
    samples = high_tstates / 35;

		if( buffer_index + samples >= 1000 )
			samples = 1000 - buffer_index;

		memset( (signed char*)( sound_buffer + buffer_index ), 120, samples );

		buffer_index += samples;

		if( buffer_index >= 1000 )
			break;

    samples = low_tstates / 35;

		if( buffer_index + samples >= 1000 )
			samples = 1000 - buffer_index;

		memset( (signed char*)( sound_buffer + buffer_index ), -120, samples - 1 );

		buffer_index += samples;
	}
*/
	return TRUE;
}

/* --- build_sound ---
 *
 * Fill the sound_buffer with a square wave
 *
 * ---
 */
void build_sound( signed char  *sound_buffer, 
								  unsigned int size, 
									unsigned int high_interval, 
									unsigned int low_interval )
{
	unsigned int index = 0;

	/* Check for unreasonable size */
	if( ( high_interval + low_interval ) > size )	
	{
		memset( sound_buffer, 0, size );
		return;
	}

	while( TRUE )
	{
		/* --- Write "high" cycle --- */
		
		/* The high cycle fits in the buffer */
		if( index <= ( size - high_interval ) )
		{
			memset( ( sound_buffer + index ), 120, high_interval );
			index += high_interval;
		}
		/* I have to truncate the high cycle */
		else
		{
			memset( ( sound_buffer + index ), 120, size - index );
			break;
		}
		
		/* --- Write "low" cycle --- */
		
		/* The low cycle fits in the buffer */
		if( index <= ( size - low_interval ) )
		{
			memset( ( sound_buffer + index ), -120, low_interval );
			index += low_interval;
		}
		/* I have to truncate the low cycle */
		else
		{
			memset( ( sound_buffer + index ), -120, size - index );
			break;
		}
	}
}

/* --- update_sound_output ---
 * 
 *  Set the frequency of the output sound
 *
 * ---
 */
void update_sound_output( void )
{
/*
	if( sound_interval == 0 )
	{
		FSOUND_SetVolume( FSOUND_ALL, 0 );
		return;
	}

	FSOUND_SetVolume( FSOUND_ALL, 255 );

	FSOUND_SetFrequency( FSOUND_ALL, ( 3500000 / sound_interval ) * 100 );

	sound_interval = 0;
*/
}

#endif