/** PocketColeco by Stuart Russell  **************************/
/**                                                         **/
/**                          CESound.c                      **/
/**                                                         **/
/** This file contains CE Specific sound routines.          **/
/**                                                         **/
/** Modified by Stuart Russell, May 2001.                   **/
/**		Rights to this code are indicated below.    		**/
/**														    **/
/*************************************************************/
/***************************************************************************

    M.A.M.E.CE  -  Multiple Arcade Machine Emulator for WinCE
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
	WinCE Portions Copyright (C) 1999 Benjamin Cooley
    
    This file is part of MAMECE, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

#include "windef.h"
#include "winbase.h"
#include "mmSystem.h"
#include "CESound.h"

int nOptions_SoundSampleRate = 3;
int nOptions_SoundBufSize    = 1000;

/***************************************************************************
    Internal structures
 ***************************************************************************/

#define NUM_WAVEHDRS  4

struct tSound_private {
	WAVEOUTCAPS		m_Caps;
	HWAVEOUT		m_hWaveOut;
	WAVEHDR			m_WaveHdrs[NUM_WAVEHDRS];

	int				m_nSampleRate;
	int				m_nSampleBits;
	int				m_nChannels;
	int				m_nVolume;		// -32 to 0 attenuation value
};


/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tSound_private      This;

unsigned char *sounddata;

int CESound_init(void)
{
	MMRESULT res;
	WAVEFORMATEX wf;

	This.m_hWaveOut = NULL;

	if (nOptions_SoundSampleRate ==1) 
		This.m_nSampleRate = 11025;
	else if (nOptions_SoundSampleRate ==2) 
		This.m_nSampleRate = 22050;
	else if (nOptions_SoundSampleRate ==3)
		This.m_nSampleRate = 44100;
	else
		This.m_nSampleRate = 11025;

	This.m_nSampleBits = 8;
	This.m_nChannels = 1;

	sounddata = (unsigned char*) (malloc(5000));

	wf.wFormatTag = WAVE_FORMAT_PCM;
	wf.nChannels = This.m_nChannels; 
	wf.nSamplesPerSec = This.m_nSampleRate; 
	wf.nAvgBytesPerSec = This.m_nSampleRate * This.m_nSampleBits / 8; 
	wf.nBlockAlign = This.m_nSampleBits * This.m_nChannels / 8;
	wf.wBitsPerSample = This.m_nSampleBits; 
	wf.cbSize = 0;

	res = waveOutOpen(
		&This.m_hWaveOut,	// Handle
		0,					// ID (0 for wave mapper)
		&wf,				// Wave format
		0,					// Callback
		0,					// Instance data
		CALLBACK_NULL);

	if (res != MMSYSERR_NOERROR)
		return 1;

	memset(&This.m_WaveHdrs, 0, sizeof(WAVEHDR) * NUM_WAVEHDRS);

	This.m_nVolume = 0;

	return 0;
}

void CESound_exit(void)
{
	BOOL done;
	int i, ticks;

	ticks = GetTickCount();
	done = FALSE;
	while (!done && GetTickCount() - ticks < 500)
	{
		done = TRUE;
		for (i = 0; i < NUM_WAVEHDRS; i++)
		{
			if (This.m_WaveHdrs[i].dwFlags & WHDR_DONE)
			{
				waveOutUnprepareHeader(This.m_hWaveOut, &This.m_WaveHdrs[i], sizeof(WAVEHDR));
				This.m_WaveHdrs[i].lpData = NULL;
				This.m_WaveHdrs[i].dwBufferLength = 0;
				This.m_WaveHdrs[i].dwFlags = 0;
			}
			else if (This.m_WaveHdrs[i].dwFlags)
				done = FALSE;
		}
	}

	waveOutClose(This.m_hWaveOut);
}

void CESound_set_volume(unsigned short left, unsigned short right)
{
	DWORD nVolume;

	nVolume = (DWORD)left | (DWORD)right << 16;

	waveOutSetVolume(This.m_hWaveOut, nVolume);
}

int CESound_get_volume(void)
{
	DWORD retvol;
	DWORD *nVolume;
	int volume;

	nVolume = &retvol;
	waveOutGetVolume(This.m_hWaveOut, nVolume);
	volume = 100-(int)((retvol & 0xffff)/655.35);//Convert to Percentage
	return volume;
}

void CESound_play_sample(char *data, int len)
{
	int i;
	char *s;
	char *d;
	DWORD pos;

	for (i = 0; i < NUM_WAVEHDRS; i++)
	{
		if (This.m_WaveHdrs[i].dwFlags == 0 || This.m_WaveHdrs[i].dwFlags & WHDR_DONE)
		{
			if (This.m_WaveHdrs[i].lpData && This.m_WaveHdrs[i].dwBufferLength != len)
			{
				waveOutUnprepareHeader(This.m_hWaveOut, &This.m_WaveHdrs[i], sizeof(WAVEHDR));
				free(This.m_WaveHdrs[i].lpData);
				This.m_WaveHdrs[i].lpData = NULL;
			}
			if (!This.m_WaveHdrs[i].lpData)
			{
				This.m_WaveHdrs[i].lpData = (LPSTR)malloc(len);
				This.m_WaveHdrs[i].dwBufferLength = len;
				This.m_WaveHdrs[i].dwFlags = 0;
				waveOutPrepareHeader(This.m_hWaveOut, &This.m_WaveHdrs[i], sizeof(WAVEHDR));
			}
			s = data;
			d = (char *)This.m_WaveHdrs[i].lpData;
			
			for (pos = 0; pos < len; pos++)
				*d++ = *s++;

			waveOutWrite(This.m_hWaveOut, &This.m_WaveHdrs[i], sizeof(WAVEHDR));
			return;	
		}
	}
}

int CESound_update_audio_stream(WORD* buffer)
{

#if 0
	int i = 0;
	short *s;
	short *d;
	short *e;
	
	static char emptybuf[1000] = "";  // Empty buffer
	e = (short *)emptybuf;
	
	s = (short *)buffer;
	for (i = 0; i < NUM_WAVEHDRS; i++)
	{		
		d = (short *)This.m_WaveHdrs[i].lpData;
		memcpy(d,e,This.m_nBytesPerFrame + 1);   //copy empty buffer to ensure NULLs 
		memcpy(d,s,This.m_nBytesPerFrame);

		waveOutWrite(This.m_hWaveOut, &This.m_WaveHdrs[i], sizeof(WAVEHDR));
	}

	samples_left_over += This.m_nSamplesPerFrame;
	samples_this_frame = (UINT32)samples_left_over;
	samples_left_over -= (double)samples_this_frame;

	return samples_this_frame;
#endif

	return 0;
}
