/* keyboard.c: Routines for dealing with the Spectrum's keyboard
   Copyright (c) 1999-2000 Philip Kendall

   $Id: keyboard.c,v 1.7 2001/08/13 21:49:40 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/
#include "keyboard.h"
#include "..\pocket_clive_types.h"

/* What to return if no keys are pressed; depends on the last byte
   output to the ULA; see CSS FAQ | Technical Information | Port #FE
   for full details */
BYTE keyboard_default_value;

/* Bit masks for each of the eight keyboard half-rows; `AND' the selected
   ones of these with `keyboard_default_value' to get the value to return
*/
BYTE keyboard_return_values[8];

/* When each Spectrum key is pressed, twiddle this {port,bit} pair
   in `keyboard_return_values'. 
*/
typedef struct keyboard_key_info {
  keyboard_key_name key;
  int port; BYTE bit;
} keyboard_key_info;

static keyboard_key_info keyboard_data[] = {

  { KEYBOARD_1      , 3, 0x01 },
  { KEYBOARD_2      , 3, 0x02 },
  { KEYBOARD_3      , 3, 0x04 },
  { KEYBOARD_4      , 3, 0x08 },
  { KEYBOARD_5      , 3, 0x10 },
  { KEYBOARD_6      , 4, 0x10 },
  { KEYBOARD_7      , 4, 0x08 },
  { KEYBOARD_8      , 4, 0x04 },
  { KEYBOARD_9      , 4, 0x02 },
  { KEYBOARD_0      , 4, 0x01 },

  { KEYBOARD_q      , 2, 0x01 },
  { KEYBOARD_w      , 2, 0x02 },
  { KEYBOARD_e      , 2, 0x04 },
  { KEYBOARD_r      , 2, 0x08 },
  { KEYBOARD_t      , 2, 0x10 },
  { KEYBOARD_y      , 5, 0x10 },
  { KEYBOARD_u      , 5, 0x08 },
  { KEYBOARD_i      , 5, 0x04 },
  { KEYBOARD_o      , 5, 0x02 },
  { KEYBOARD_p      , 5, 0x01 },

  { KEYBOARD_a      , 1, 0x01 },
  { KEYBOARD_s      , 1, 0x02 },
  { KEYBOARD_d      , 1, 0x04 },
  { KEYBOARD_f      , 1, 0x08 },
  { KEYBOARD_g      , 1, 0x10 },
  { KEYBOARD_h      , 6, 0x10 },
  { KEYBOARD_j      , 6, 0x08 },
  { KEYBOARD_k      , 6, 0x04 },
  { KEYBOARD_l      , 6, 0x02 },
  { KEYBOARD_Enter  , 6, 0x01 },

  { KEYBOARD_Caps   , 0, 0x01 },
  { KEYBOARD_z      , 0, 0x02 },
  { KEYBOARD_x      , 0, 0x04 },
  { KEYBOARD_c      , 0, 0x08 },
  { KEYBOARD_v      , 0, 0x10 },
  { KEYBOARD_b      , 7, 0x10 },
  { KEYBOARD_n      , 7, 0x08 },
  { KEYBOARD_m      , 7, 0x04 },
  { KEYBOARD_Symbol , 7, 0x02 },
  { KEYBOARD_space  , 7, 0x01 },

  { KEYBOARD_Ext    , 0, 0x00 },

  { KEYBOARD_NONE   , 0, 0x00 } /* End marker: DO NOT MOVE! */

};


/* Keyboard layout added by Anders Holmberg */
keyboard_key_name keyboard_loc[ 5 ][ 10 ] = 
{
	{ KEYBOARD_1, KEYBOARD_2, KEYBOARD_3, KEYBOARD_4, KEYBOARD_5, 
	  KEYBOARD_6, KEYBOARD_7, KEYBOARD_8, KEYBOARD_9, KEYBOARD_0  }, 
	{ KEYBOARD_q, KEYBOARD_w, KEYBOARD_e, KEYBOARD_r, KEYBOARD_t, 
	  KEYBOARD_y, KEYBOARD_u, KEYBOARD_i, KEYBOARD_o, KEYBOARD_p  }, 
	{ KEYBOARD_a, KEYBOARD_s, KEYBOARD_d, KEYBOARD_f, KEYBOARD_g, 
	  KEYBOARD_h, KEYBOARD_j, KEYBOARD_k, KEYBOARD_l, KEYBOARD_Enter  }, 
	{ KEYBOARD_Caps, KEYBOARD_z, KEYBOARD_x, KEYBOARD_c, KEYBOARD_v, 
	  KEYBOARD_b, KEYBOARD_n, KEYBOARD_m, KEYBOARD_Symbol, KEYBOARD_Ext  }, 
	{ KEYBOARD_NONE, KEYBOARD_NONE, KEYBOARD_NONE, KEYBOARD_NONE, KEYBOARD_NONE, 
	  KEYBOARD_NONE, KEYBOARD_NONE, KEYBOARD_space, KEYBOARD_space, KEYBOARD_space }
};

#define SYMBOL_DOWN    0
#define SYMBOL_GO_UP   1
#define SYMBOL_GO_DOWN 2
#define SYMBOL_UP      3

#define CAPS_DOWN    0
#define CAPS_GO_UP   1
#define CAPS_GO_DOWN 2
#define CAPS_UP      3

int caps_state   = CAPS_UP;
int symbol_state = SYMBOL_UP;

keyboard_key_name last_key;
BYTE joystick;

void keyboard_init(void)
{
  int i;
  
  keyboard_default_value=0xff;
  for(i=0;i<8;i++) keyboard_return_values[i]=0xff;
}

BYTE keyboard_read(BYTE porth)
{
  BYTE data=keyboard_default_value; 
	int i;

  for( i=0; i<8; i++,porth>>=1 ) 
	{
    if(! (porth&0x01) ) 
		{
      data &= keyboard_return_values[i];
    }
  }

  return data;
}

void keyboard_press( keyboard_key_name key )
{
  keyboard_key_info *ptr;

  if( key == KEYBOARD_NONE ) 
		return;

  for( ptr=keyboard_data; ptr->key; ptr++ ) 
	{
    if( key == ptr->key ) 
		{
      keyboard_return_values[ ptr->port ] &= ~( ptr->bit );
      return;
    }
  }
  return;
}

void keyboard_release(keyboard_key_name key)
{
  keyboard_key_info *ptr;

  if( key == KEYBOARD_NONE ) 
		return;

  for( ptr=keyboard_data; ptr->key; ptr++ ) 
	{
    if( key == ptr->key ) 
		{
      keyboard_return_values[ ptr->port ] |= ptr->bit;
      return;
    }
  }
  return;
}

/* All functions below added by Anders Holmberg */

void keyboard_clicked( int xpos, int ypos )
{
	keyboard_key_name key;

	key = keyboard_loc[ ( ypos - 202 ) / 20 ][ ( xpos - 1 ) / 24 ];
	keyboard_press( key );
}

void keyboard_down( int xpos, int ypos )
{
	keyboard_key_name key;

	key = keyboard_loc[ ( ypos - 202 ) / 20 ][ ( xpos - 1 ) / 24 ];

	if( key == KEYBOARD_Ext )	
	{
		keyboard_press( KEYBOARD_Caps );
		keyboard_press( KEYBOARD_Symbol );
		last_key = KEYBOARD_Ext;
	}

	if( key != KEYBOARD_NONE )
	{
	  keyboard_press( key );
	}

	if( key == KEYBOARD_Symbol )
	{
		if( symbol_state == SYMBOL_UP )
			symbol_state = SYMBOL_GO_DOWN;
		else if( symbol_state == SYMBOL_DOWN )
			symbol_state = SYMBOL_GO_UP;
	}
	else if( key == KEYBOARD_Caps )	
	{
		if( caps_state == CAPS_UP )
			caps_state = CAPS_GO_DOWN;
		else if( caps_state == CAPS_DOWN )
			caps_state = CAPS_GO_UP;
	}
	else
	{
	  last_key = key;
	}
}

void keyboard_up( void )
{
	if( last_key == KEYBOARD_Ext )
	{
		keyboard_release( KEYBOARD_Symbol );
		keyboard_release( KEYBOARD_Caps );

		return;
	}

	if( last_key != KEYBOARD_NONE )
	{
  	keyboard_release( last_key );
		if( caps() )
		{
			caps_state = CAPS_GO_UP;
		}
		if( symbol() )
		{
			symbol_state = SYMBOL_GO_UP;
		}
	}

	last_key = KEYBOARD_NONE;

	if( symbol_state == SYMBOL_GO_UP )
	{
		symbol_state = SYMBOL_UP;
		keyboard_release( KEYBOARD_Symbol );
	}
	else if( symbol_state == SYMBOL_GO_DOWN )
		symbol_state = SYMBOL_DOWN;

	if( caps_state == CAPS_GO_UP )
	{
		caps_state = CAPS_UP;
		keyboard_release( KEYBOARD_Caps );
	}
	else if( caps_state == CAPS_GO_DOWN )
		caps_state = CAPS_DOWN;
}

void joystick_set( BYTE value )
{
  joystick |= value;
}

void joystick_reset_movement( void )
{
	joystick &= 0xf0;
}

void joystick_reset_fire( void )
{
	joystick &= 0xef;
}

int symbol( void )
{
	if( symbol_state == SYMBOL_UP )
		return 0;
	else
		return 1;
}

int caps( void )
{
	if( caps_state == CAPS_UP )
		return 0;
	else
		return 1;
}
