/*
    (C) 2001 Anders Holmberg

    This file is part of PocketClive.

    PocketClive is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    PocketClive is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with PocketClive; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Author contact information : aagholmberg@yahoo.se

    PocketClive uses code and inspiration from the following sources 
    (see AUTHORS.TXT for more details) :

    FUSE - The Free Unix Spectrum Emulator - (C) 1999-2001 Philip Kendall
		
    EasyCE - PocketPC game programming library - (C) 2001 Jacco Bikker

    MESS - Multi-Emulator Super System - (C) 1998-2001 MESS Team

    Note : Source code files present in the sub directory PocketClive\EasyCE
           does not comply with the GPL license. These files are (C) Jacco Bikker
           with the following attachment : 

           "If you want to use EasyCE for commercial applications
            you should let me know. I will ask you a small fee,
            since I spend a lot of time developing this software.
            For freeware products, EasyCE can be used freely."
*/

#include "pocket_clive_files.h"
#ifndef _WIN32_WCE
#include "windows.h"
#else
#include "EasyCE\EasyCE.h"
#include "aygshell.h"
#endif
#include "mmsystem.h"

int ATgaFile::s_IDLen = 0;
int ATgaFile::s_CMapType = 0;
int ATgaFile::s_ImgType = 0;
int ATgaFile::s_CMapOrig = 0;
int ATgaFile::s_CMapLen = 0;
int ATgaFile::s_CMapSize = 0;
int ATgaFile::s_XPos = 0;
int ATgaFile::s_YPos = 0;
int ATgaFile::s_Width = 0;
int ATgaFile::s_Height = 0;
int ATgaFile::s_PixSize = 0;

// ------------------------------------------------
// ATgaFile::ATgaFile
// Constructor
// ------------------------------------------------
ATgaFile::ATgaFile()
{
	// This class exists only for the load method
}

// ------------------------------------------------
// ATgaFile::~ATgaFile
// Destructor
// ------------------------------------------------
ATgaFile::~ATgaFile()
{
	// This class exists only for the load method
}

// ------------------------------------------------
// ATgaFile::load
// Load an indexed tga file from disk
// ------------------------------------------------
bool ATgaFile::load( ACHAR* file, int w, int h, IPIXEL* dest, PIXEL* pal )
{
	int size = w * h + 20 + 768;
	BYTE* tgabuff = new BYTE[size];
	HANDLE tga = CreateFile( file, GENERIC_READ, FILE_SHARE_READ,
	                         0, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0 );
	if (tga == INVALID_HANDLE_VALUE) return false;
	DWORD bytesread;
	ReadFile( tga, tgabuff, size, &bytesread, 0 );
	if (CloseHandle( tga ) == 0) return false;
	parseheader( tgabuff );
	// Check if the file is valid
	if ((s_Width != w) ||
	    (s_Height != h) ||
		(s_ImgType != 1) ||
		(s_CMapSize != 24)) return false;
	// Store the data at the specified target address
	if (pal)
	{
		for ( int i = 0; i < 256; i++ )
		{
			int b = *(tgabuff + 18 + i * 3);
			int g = *(tgabuff + 18 + i * 3 + 1);
			int r = *(tgabuff + 18 + i * 3 + 2);
			pal[i] = (PIXEL)(((r >> 3) << 11) + ((g >> 2) << 5) + (b >> 3));
		}
	}
	IPIXEL* src = tgabuff + 18 + 768 + (h - 1) * w;
	IPIXEL* dst = (IPIXEL*)dest;
	for ( int y = 0; y < h; y++ )
	{
		memcpy( dst, src, w );
		dst += w;
		src -= w;
	}
	// Release temporary storage
	delete tgabuff;
	return true;
}

// ------------------------------------------------
// ATgaFile::load
// Load a 16 bit tga file from disk
// ------------------------------------------------
bool ATgaFile::load( ACHAR* file, int w, int h, PIXEL* dest )
{
	int size = w * 2 * h + 20;
	BYTE* tgabuff = new BYTE[size];
	HANDLE tga = CreateFile( file, GENERIC_READ, FILE_SHARE_READ,
	                         0, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0 );
	if (tga == INVALID_HANDLE_VALUE) return false;
	DWORD bytesread;
	ReadFile( tga, tgabuff, size, &bytesread, 0 );
	if (CloseHandle( tga ) == 0) return false;
	parseheader( tgabuff );
	if (s_ImgType == 1)
	{
		// Palettized image
		PIXEL* pal = new PIXEL[256];
		for ( int i = 0; i < 256; i++ )
		{
			int b = *(tgabuff + 18 + i * 3);
			int g = *(tgabuff + 18 + i * 3 + 1);
			int r = *(tgabuff + 18 + i * 3 + 2);
			pal[i] = (PIXEL)(((r >> 3) << 11) + ((g >> 3) << 6) + (b >> 3));
		}
		IPIXEL* src = tgabuff + 18 + 768 + (h - 1) * w;
		PIXEL* dst = (PIXEL*)dest;
		for ( int y = 0; y < h; y++ )
		{
			for ( int x = 0; x < w; x++ )
			{
				int idx = *(src + x);
				*(dst + x) = pal[idx];
			}
			dst += w;
			src -= w;
		}
	}
	else
	{
		// Store the data at the specified target address
		PIXEL* src = (PIXEL*)tgabuff + 9 + (h - 1) * w;
		PIXEL* dst = (PIXEL*)dest;
		for ( int i = 0; i < h; i++ )
		{
			for ( int x = 0; x < w; x++ )
			{
				int red = (*(src + x) & (31 << 10)) >> 10;
				int green = (*(src + x) & (31 << 5)) >> 5;
				int blue = *(src + x) & 31;
				*(dst + x) = (red << 11) + (green << 6) + blue;
			}
			dst += w;
			src -= w;
		}
	}
	// Release temporary storage
	delete tgabuff;
	return true;
}

// ------------------------------------------------
// ATgaFile::parseheader
// Read the TGA file header to perform various tests
// ------------------------------------------------
void ATgaFile::parseheader( BYTE* a_Header )
{
	s_IDLen		= *a_Header;
	s_CMapType	= *(a_Header + 1);
	s_ImgType	= *(a_Header + 2);
	s_CMapOrig	= *(a_Header + 3) + 256 * *(a_Header + 4);
	s_CMapLen	= *(a_Header + 5) + 256 * *(a_Header + 6);
	s_CMapSize	= *(a_Header + 7);
	s_XPos		= *(a_Header + 8) + 256 * *(a_Header + 9);
	s_YPos		= *(a_Header + 10) + 256 * *(a_Header + 11);
	s_Width		= *(a_Header + 12) + 256 * *(a_Header + 13);
	s_Height	= *(a_Header + 14) + 256 * *(a_Header + 15);
	s_PixSize	= *(a_Header + 16);
}

// ------------------------------------------------
// EOF ORIGINAL FILE

SpectrumFile::SpectrumFile()
{
	// This class exists only for the load method
}

SpectrumFile::~SpectrumFile()
{
	// This class exists only for the load method
}


bool SpectrumFile::load_scr( ACHAR *file, unsigned char *dest )
{
	int size = 6912;

	HANDLE tga = CreateFile( file, GENERIC_READ, FILE_SHARE_READ,
	                         0, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0 );
	if (tga == INVALID_HANDLE_VALUE) return false;

	DWORD bytesread;
	ReadFile( tga, dest, size, &bytesread, 0 );

	if (CloseHandle( tga ) == 0) return false;
	return true;
}

bool SpectrumFile::load_rom( char *file, unsigned char *dest )
{
	int size = 16384;

	FILE *f;

	f = fopen( file, "rb" );
	
	if( f != NULL )
		fread( dest, 1, 16384, f );
	
	fclose( f );
 /*

	HANDLE tga = CreateFile( file, GENERIC_READ, FILE_SHARE_READ,
	                         0, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0 );
	if (tga == INVALID_HANDLE_VALUE) return false;

	DWORD bytesread;
	ReadFile( tga, dest, size, &bytesread, 0 );

	if (CloseHandle( tga ) == 0) return false;

	*/

	return true;
}

bool SpectrumFile::load_snap( ACHAR *file, unsigned char *dest )
{
	int size = 49179;

	HANDLE tga = CreateFile( file, GENERIC_READ, FILE_SHARE_READ,
	                         0, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0 );
	if (tga == INVALID_HANDLE_VALUE) return false;

	DWORD bytesread;
	ReadFile( tga, dest, size, &bytesread, 0 );

	if (CloseHandle( tga ) == 0) return false;
	return true;
}

bool load( char *file_name, int size, unsigned char *dest )
{
	FILE *f;
	int nr_read;

	f = fopen( file_name, "rb" );
	
	if( f != NULL )
	{
		nr_read = fread( dest, 1, size, f );
		fclose( f );

		if( nr_read != size )
			return FALSE;
		else
			return TRUE;
	}
	else
		return FALSE;

}

