// EasyCE.cpp : API implementation
// By Jacco Bikker, a.k.a. "The Phantom"
// Cool windows suggestions by Jaap Suter
// 2-bit dithering code by Jakob Nebeker
// --------------------------------------------------
 
/*  Modified by Anders Holmberg

    Supports the Pocket PC emulator with GAPI x86 emulator (see
    "http://frogengine.net-connect.net" for the GAPI x86 
	  emulator, Copyright (C) 2000 Thierry Tremblay)
	 
	  The GAPI x86 emulator do not run at all when the function
	  GXOpenDisplay is called with a windows handle equal to nil. 
	  It also has problems when the window is moved MENU_HEIGHT
	  pixels to accommodate for the loss of the menu taskbar. 

    The font has been changed from the "arcade-like" font (looks
	  very nice actually but not right for a Spectrum emulator) to
	  the old trustworthy Spectrum font. 
*/

#include "math.h"
#include "EasyCE.h"
#ifdef _WIN32_WCE
	#include "aygshell.h"
	#include "mmsystem.h"
#else
	#include "windows.h"
	#include "ptc.h"
	Surface* surface;
	Format* format;
	Console* console;
#endif

/* Added by Anders Holmberg */

#ifdef _X86_
#undef MENU_HEIGHT
#define MENU_HEIGHT 0
#endif

HINSTANCE hInst;
TCHAR szTitle[30] = TEXT("easyce");
TCHAR wincls[30] = TEXT("easyce");
HWND hWnd;
HBITMAP hBM;

// Framebuffer stuff
unsigned long* buffer, *temp;
char* m_chr[54][7];
int m_transl[256];
#ifdef USE_GAMEX
	GameX* gameX;
#endif
int cursorx, cursory;
int method = 1, BPP, xpitch, ypitch;
bool pendwn = false;
bool imm = true;
unsigned int basetime = 0;

// Hardware buttons
#ifdef _WIN32_WCE
	GXKeyList kl;
#endif
int b1, b2, b3, b4;
int bl, br, bu, bd;
bool button1up, button2up, button3up, button4up;

// Greyscale emulator timing table and variables
int tm[32] = { 0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 1, 3, 5, 7, 9, 11, 13, 15, 17, 19, 21, 23, 25, 27, 29, 31 };
int time, tmindx = 0;

// Dithering variables
int pvtime;
unsigned char pv7[2*8] = { 0,0, 1,1, 2,2, 3,3,
                           0,1, 1,2, 2,3, 3,3 };
unsigned char pv16[5*16] = { 0,0,0,0,0, 1,1,1,1,1, 2,2,2,2,2, 3,
                             0,0,1,1,1, 1,1,2,2,2, 2,2,3,3,3, 3,
                             0,0,0,0,1, 1,1,1,1,2, 2,2,2,2,3, 3,
                             0,0,0,1,1, 1,1,1,2,2, 2,2,2,3,3, 3,
                             0,1,1,1,1, 1,2,2,2,2, 2,3,3,3,3, 3 };
unsigned char pv31[10*32] = { 0,1,1,1,1,1,0,0,0,0, 1,2,2,2,2,1,1,1,1,1, 2,3,3,3,3,2,2,2,2,2, 3,3,
                              0,0,0,0,0,0,1,1,1,1, 1,1,1,1,1,1,2,2,2,2, 2,2,2,2,2,2,3,3,3,3, 3,3,
                              0,0,0,0,0,1,1,1,1,1, 1,1,1,1,1,2,2,2,2,2, 2,2,2,2,2,3,3,3,3,3, 3,3,
                              0,0,0,0,1,0,0,1,1,1, 1,1,1,1,2,1,1,2,2,2, 2,2,2,2,3,2,2,3,3,3, 3,3,
                              0,0,0,1,0,1,1,0,1,1, 1,1,1,2,1,2,2,1,2,2, 2,2,2,3,2,3,3,2,3,3, 3,3,
                              0,0,1,0,0,0,1,1,0,1, 1,1,2,1,1,1,2,2,1,2, 2,2,3,2,2,2,3,3,2,3, 3,3,
                              0,0,0,0,1,1,0,1,1,1, 1,1,1,1,2,2,1,2,2,2, 2,2,2,2,3,3,2,3,3,3, 3,3,
                              0,0,0,1,0,0,1,0,1,1, 1,1,1,2,1,1,2,1,2,2, 2,2,2,3,2,2,3,2,3,3, 3,3,
                              0,0,0,0,1,1,0,1,1,1, 1,1,1,1,2,2,1,2,2,2, 2,2,2,2,3,3,2,3,3,3, 3,3,
                              0,0,0,0,0,0,1,1,1,1, 1,1,1,1,1,1,2,2,2,2, 2,2,2,2,2,2,3,3,3,3, 3,3 };
// Stacks
int mxpos, mypos, mcxpos, mcypos;
bool mcprocessed, mbutton;

ATOM MyRegisterClass(HINSTANCE hInstance);
BOOL InitInstance(HINSTANCE, int);
LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK About(HWND, UINT, WPARAM, LPARAM);
extern void __cdecl main( void );
void initcharset( void );

void ttoa( int v, char* s )
{
	int sec = v % 60;
	int min = (v / 60) % 60;
	int hour = (v / 3600);
	if (hour > 99)
	{
		// overflow
		hour = 88;
		sec = 88;
		min = 88;
	}
	*s = hour / 10 + '0';
	*(s + 1) = hour % 10 + '0';
	*(s + 2) = ':';
	*(s + 3) = min / 10 + '0';
	*(s + 4) = min % 10 + '0';
	*(s + 5) = ':';
	*(s + 6) = sec / 10 + '0';
	*(s + 7) = sec % 10 + '0';
	*(s + 8) = 0;
}

void itoa( int v, char* s )
{
	if (v == 0)
	{
		*s = '0';
		*(s + 1) = 0;
		return;
	}
	int mul = 10000000;
	bool first = true;	
	char* cur = s;
	if ( v < 0)
	{
		*cur++ = '-';
		v = -v;
	}
	while (mul > 0)
	{
		int digit = (v / mul) % 10;
		if ((digit > 0) || (!first))
		{
			*cur++ = '0' + digit;
			first = false;
		}
		mul /= 10;
	}
	*cur = 0;
}

#ifdef _WIN32_WCE
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow)
#else
int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow )
#endif
{
	MSG msg;
	if( msg.hwnd = FindWindow(szTitle,NULL) )
	{
		ShowWindow(msg.hwnd,SW_SHOW);
		return FALSE;
	}     
	buffer = new unsigned long[160 * 240];
	mcprocessed = true;
	mbutton = false;
	b1 = b2 = b3 = b4 = 0;
	initcharset();
	cursorx = 2;
	cursory = 2;
#ifdef _WIN32_WCE
	taskbar( false );
	resettimer();
   	if (!MyRegisterClass(hInstance)) return FALSE;
	if (!InitInstance (hInstance, nCmdShow)) return FALSE;
#endif

#ifdef _WIN32_WCE
	#ifdef USE_GAMEX
		gameX = new GameX();
		if (!gameX->OpenGraphics()) return 0;
		BPP = gameX->GetFBBpp();
		ypitch = gameX->GetFBModulo();
		if (BPP == 16) xpitch = 2; else xpitch = 1;
	#else

#ifdef _X86_
		GXOpenDisplay( hWnd, GX_FULLSCREEN );
#else
		GXOpenDisplay( NULL, GX_FULLSCREEN );
#endif
		GXOpenInput();
		kl = GXGetDefaultKeys( GX_NORMALKEYS );
		GXDisplayProperties dp = GXGetDisplayProperties();
		BPP = dp.cBPP;
		xpitch = dp.cbxPitch;
		ypitch = dp.cbyPitch;
		if ((!GXIsDisplayDRAMBuffer()) && (ypitch == 480)) 
		{
			// No need to do our own double buffering
			imm = false;
		}
	#endif
#else
	format  = new Format( 16, 31 << 11, 63 << 5, 31 );
	console = new Console();
	console->option( "Windowed output" );
	console->open( "The Nutcracker", 240, 320, *format );
	surface = new Surface ( 240, 320, *format );
	BPP = 16;
	xpitch = 1;
	ypitch = 240;
	hWnd = console->window();
#endif

	
	/* memset( buffer, 0, 320 * 240 * 2 );
	update();
	while (msecs() < 500) {}; resettimer();
	centre( "powered by easyce", 150, 65535 );
	centre( "copyright 2001 the phantom", 159, 65535 );
	update();
	while (msecs() < 1200) {}; resettimer();
	memset( buffer, 0, 320 * 240 * 2 );
	update();
	while (msecs() < 300) {}; resettimer(); */
	main();
#ifdef _WIN32_WCE
	taskbar( true );
	#ifdef USE_GAMEX
		delete gameX;
	#else
		GXCloseInput();
		GXCloseDisplay();
	#endif
#endif
	delete buffer;
	for ( int x = 0; x < 54; x++ ) for ( int y = 0; y < 7; y++ ) delete m_chr[x][y];
	return 0;
}

ATOM MyRegisterClass(HINSTANCE hInstance)
{
	WNDCLASS wcls;
	wcls.style			= 0;
	wcls.lpfnWndProc	= (WNDPROC)WndProc;
	wcls.cbClsExtra		= 0;
	wcls.cbWndExtra		= 0;
	wcls.hInstance		= hInstance;
	wcls.hIcon			= LoadIcon(hInstance,MAKEINTRESOURCE(101));
	wcls.hCursor		= 0;
	wcls.hbrBackground	= (HBRUSH) GetStockObject(BLACK_BRUSH);
	wcls.lpszMenuName	= 0;
	wcls.lpszClassName	= wincls;
	return RegisterClass(&wcls);
}

BOOL InitInstance(HINSTANCE hi, int nCmdShow)
{
	hInst = hi;

	/* Changed by Anders Holmberg */

	//	hWnd = CreateWindow(wincls, szTitle, WS_POPUP, CW_USEDEFAULT, CW_USEDEFAULT, 240, 320, 0, 0, hi, 0);
	hWnd = CreateWindow(wincls, szTitle, WS_POPUP, 0, 0, 240, 320, NULL, NULL, hi, 0);
	
	if (!hWnd) return FALSE;
	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);
	return TRUE;
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	short key = 0;
#ifdef _WIN32_WCE
	static SHACTIVATEINFO sai;
#endif
	switch (message) 
	{
	#ifdef _WIN32_WCE
		case WM_ENABLE:
		case WM_ACTIVATE:
		case WM_CREATE:
		case WM_SETTINGCHANGE:
			taskbar( false );
			break;
	#endif
		case WM_LBUTTONDOWN:
			mcxpos = LOWORD( lParam );
			mcypos = HIWORD( lParam ) - MENU_HEIGHT;
			mcprocessed = false;
			mbutton = true;
 			break;
		case WM_LBUTTONUP:
			mbutton = false;
			break;
		case WM_MOUSEMOVE:
			mxpos = LOWORD( lParam );
			mypos = HIWORD( lParam ) - MENU_HEIGHT;
			// mbutton = (wParam == MK_LBUTTON);
			break;
		case WM_DESTROY:
			// ReleaseCapture();
			PostQuitMessage(0);
			break;
		case WM_KEYUP:
			key = (short)wParam;
		#ifdef _WIN32_WCE
			if (key == kl.vkLeft) bl = 0;
			if (key == kl.vkRight) br = 0;
			if (key == kl.vkUp) bu = 0;
			if (key == kl.vkDown) bd = 0;
		#endif
			break;
		case WM_KEYDOWN:
			key = (short)wParam;
		#ifdef _WIN32_WCE
			if (key == kl.vkA) b1 = 1;
			if (key == kl.vkB) b2 = 1;
			if (key == kl.vkC) b3 = 1;
			if (key == kl.vkStart) b4 = 1;
			if (key == kl.vkLeft) bl = 1;
			if (key == kl.vkRight) br = 1;
			if (key == kl.vkUp) bu = 1;
			if (key == kl.vkDown) bd = 1;
		#else
			if (key == VK_INSERT) b1 = 1;
			if (key == VK_HOME) b2 = 1;
			if (key == VK_DELETE) b3 = 1;
			if (key == VK_END) b4 = 1;
			if (key == VK_LEFT) bl = 1;
			if (key == VK_RIGHT) br = 1;
			if (key == VK_UP) bu = 1;
			if (key == VK_DOWN) bd = 1;
		#endif
			break;
		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
   }
   return 0;
}

void taskbar( bool show )
{
#ifdef _WIN32_WCE
	#ifdef USE_GAMEX
		// The winCE 2.x way
		#define TASKBAR_KEY _T("\\software\\microsoft\\shell") 
		HKEY hKey=0;
		RegOpenKeyEx( HKEY_LOCAL_MACHINE, TASKBAR_KEY, 0, KEY_ALL_ACCESS, &hKey );
		DWORD dwValue = 0;
		unsigned long lSize = sizeof( DWORD );
		DWORD dwType = REG_DWORD;
		RegQueryValueEx( hKey, _T("TBOpt"), 0, &dwType, (BYTE*)&dwValue, &lSize );
		if (show) dwValue &= 0xFFFFFFFF - 8;// reset bit to show taskbar
	 		 else dwValue |= 8;				// set bit to hide taskbar
		RegSetValueEx( hKey, _T("TBOpt"), 0, REG_DWORD, (BYTE*)&dwValue, lSize );
		MSG msg;
		msg.hwnd = FindWindow( _T("HHTaskBar"), NULL );
		::SendMessage( msg.hwnd, WM_COMMAND, 0x03EA, 0 );
		::SetForegroundWindow( hWnd );
	#else
		// The Pocket PC way
		RECT rc;
		GetWindowRect( hWnd, &rc );
		HWND hWndTB=FindWindow(TEXT("HHTaskbar"),NULL);
		if (show)
		{
			SHFullScreen( hWnd, SHFS_SHOWTASKBAR | SHFS_SHOWSIPBUTTON + SHFS_SHOWSTARTICON );
			ShowWindow( hWndTB, SW_SHOW );
			MoveWindow( hWnd, 
						rc.left, 
						rc.top + MENU_HEIGHT, 
						rc.right, 
						rc.bottom - 2 * MENU_HEIGHT,
						TRUE
					  );
		}
		else
		{
			SHFullScreen( hWnd, SHFS_HIDETASKBAR | SHFS_HIDESIPBUTTON + SHFS_HIDESTARTICON );
			ShowWindow( hWndTB, SW_HIDE );
			MoveWindow( hWnd, 
						rc.left, 
						rc.top - MENU_HEIGHT, 
						rc.right, 
						rc.bottom + MENU_HEIGHT,
						TRUE
					  );
		}
	#endif
#endif
}

void updatedisplay( void )
{
#ifdef _WIN32_WCE
	if (BPP == 2)
	{
		// Screen update routine for 2bit devices (Nino, E10)
	#ifdef USE_GAMEX
		if (!gameX->BeginDraw()) return;
		unsigned int* dest = (unsigned int*)getscrptr();
	#else
		unsigned int* dest = (unsigned int*)GXBeginDraw();
	#endif
		int gapsize = (ypitch - 60) >> 2;
		unsigned short* a = (unsigned short*)buffer;
		time = tm[tmindx] * 16;
		if (method == 1)
		{
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += (*(a   )>>14)<<6;
					c += (*(a+1 )>>14)<<4;
					c += (*(a+2 )>>14)<<2;
					c += (*(a+3 )>>14);
					c += (*(a+4 )>>14)<<14;
					c += (*(a+5 )>>14)<<12;
					c += (*(a+6 )>>14)<<10;
					c += (*(a+7 )>>14)<<8;
					c += (*(a+8 )>>14)<<22;
					c += (*(a+9 )>>14)<<20;
					c += (*(a+10)>>14)<<18;
					c += (*(a+11)>>14)<<16;
					c += (*(a+12)>>14)<<30;
					c += (*(a+13)>>14)<<28;
					c += (*(a+14)>>14)<<26;
					c += (*(a+15)>>14)<<24;
					a += 16;
					*dest++ = c;
				}
				dest += gapsize;
			}
		}
		else if (method == 2)
		{
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += -((*a >> 11) > time) & 192;
					c += -((*(a + 1) >> 11) > time) & 48;
					c += -((*(a + 2) >> 11) > time) & 12;
					c += -((*(a + 3) >> 11) > time) & 3;
					c += -((*(a + 4) >> 11) > time) & 49152;
					c += -((*(a + 5) >> 11) > time) & 12288;
					c += -((*(a + 6) >> 11) > time) & 3072;
					c += -((*(a + 7) >> 11) > time) & 768;
					c += -((*(a + 8) >> 11) > time) & 12582912;
					c += -((*(a + 9) >> 11) > time) & 3145728;
					c += -((*(a + 10) >> 11) > time) & 786432;
					c += -((*(a + 11) >> 11) > time) & 196608;
					c += -((*(a + 12) >> 11) > time) & 3221225472;
					c += -((*(a + 13) >> 11) > time) & 805306368;
					c += -((*(a + 14) >> 11) > time) & 201326592;
					c += -((*(a + 15) >> 11) > time) & 50331648;
					a += 16;
					*dest++ = c;
				}
				time = (time + 8) & 31;
				dest += gapsize;
			}
		}
		else if (method == 3)
		{
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += -((*a >> 11) > time) & 192;
					c += -((*(a + 1) >> 11) > time) & 48;
					c += -((*(a + 2) >> 11) > time) & 12;
					c += -((*(a + 3) >> 11) > time) & 3;
					c += -((*(a + 4) >> 11) > time) & 49152;
					c += -((*(a + 5) >> 11) > time) & 12288;
					c += -((*(a + 6) >> 11) > time) & 3072;
					c += -((*(a + 7) >> 11) > time) & 768;
					c += -((*(a + 8) >> 11) > time) & 12582912;
					c += -((*(a + 9) >> 11) > time) & 3145728;
					c += -((*(a + 10) >> 11) > time) & 786432;
					c += -((*(a + 11) >> 11) > time) & 196608;
					c += -((*(a + 12) >> 11) > time) & 3221225472;
					c += -((*(a + 13) >> 11) > time) & 805306368;
					c += -((*(a + 14) >> 11) > time) & 201326592;
					c += -((*(a + 15) >> 11) > time) & 50331648;
					a += 16;
					*dest++ = c;
				}
				time = (time + 4) & 31;
				dest += gapsize;
			}
		}
		if (method == 4)
		{
			unsigned char* dthptr = (unsigned char*)pv7 + pvtime*8;
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += dthptr[*(a     )>>5] << 6;
					c += dthptr[*(a + 1 )>>5] << 4;
					c += dthptr[*(a + 2 )>>5] << 2;
					c += dthptr[*(a + 3 )>>5];
					c += dthptr[*(a + 4 )>>5] << 14;
					c += dthptr[*(a + 5 )>>5] << 12;
					c += dthptr[*(a + 6 )>>5] << 10;
					c += dthptr[*(a + 7 )>>5] << 8;
					c += dthptr[*(a + 8 )>>5] << 22;
					c += dthptr[*(a + 9 )>>5] << 20;
					c += dthptr[*(a + 10)>>5] << 18;
					c += dthptr[*(a + 11)>>5] << 16;
					c += dthptr[*(a + 12)>>5] << 30;
					c += dthptr[*(a + 13)>>5] << 28;
					c += dthptr[*(a + 14)>>5] << 26;
					c += dthptr[*(a + 15)>>5] << 24;
					a += 16;
					*dest++ = c;
				}
				dest += gapsize;
			}
			pvtime++; if (pvtime > 2) pvtime = 0;
		}
		else if( method==5 )
		{
			unsigned char* dthptr = (unsigned char*)pv16 + pvtime*16;
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += dthptr[*(a     )>>4] << 6;
					c += dthptr[*(a + 1 )>>4] << 4;
					c += dthptr[*(a + 2 )>>4] << 2;
					c += dthptr[*(a + 3 )>>4];
					c += dthptr[*(a + 4 )>>4] << 14;
					c += dthptr[*(a + 5 )>>4] << 12;
					c += dthptr[*(a + 6 )>>4] << 10;
					c += dthptr[*(a + 7 )>>4] << 8;
					c += dthptr[*(a + 8 )>>4] << 22;
					c += dthptr[*(a + 9 )>>4] << 20;
					c += dthptr[*(a + 10)>>4] << 18;
					c += dthptr[*(a + 11)>>4] << 16;
					c += dthptr[*(a + 12)>>4] << 30;
					c += dthptr[*(a + 13)>>4] << 28;
					c += dthptr[*(a + 14)>>4] << 26;
					c += dthptr[*(a + 15)>>4] << 24;
					a += 16;
					*dest++ = c;
				}
				dest += gapsize;
			}
			pvtime++; if (pvtime > 4) pvtime = 0;
		}
		else if (method == 6)
		{
			unsigned char* dthptr = (unsigned char*)pv31 + pvtime*32;
			for ( int y = 0; y < 320; y++ )
			{
				for ( int x = 0; x < 15; x++ )
				{
					int c = 0;
					c += dthptr[*(a     )>>3] << 6;
					c += dthptr[*(a + 1 )>>3] << 4;
					c += dthptr[*(a + 2 )>>3] << 2;
					c += dthptr[*(a + 3 )>>3];
					c += dthptr[*(a + 4 )>>3] << 14;
					c += dthptr[*(a + 5 )>>3] << 12;
					c += dthptr[*(a + 6 )>>3] << 10;
					c += dthptr[*(a + 7 )>>3] << 8;
					c += dthptr[*(a + 8 )>>3] << 22;
					c += dthptr[*(a + 9 )>>3] << 20;
					c += dthptr[*(a + 10)>>3] << 18;
					c += dthptr[*(a + 11)>>3] << 16;
					c += dthptr[*(a + 12)>>3] << 30;
					c += dthptr[*(a + 13)>>3] << 28;
					c += dthptr[*(a + 14)>>3] << 26;
					c += dthptr[*(a + 15)>>3] << 24;
					a += 16;
					*dest++ = c;
				}
				dest += gapsize;
			}
			pvtime++; if (pvtime > 9) pvtime = 0;
		}
	#ifdef USE_GAMEX
		gameX->EndDraw();
	#else
		GXEndDraw();
	#endif
		tmindx++; if (tmindx > 15) tmindx = 0;
	}
	else if (BPP == 16)
	{
	#ifdef USE_GAMEX
		if (!gameX->BeginDraw()) return;
		unsigned short* dst = (unsigned short*)getscrptr();
		unsigned short* src = (unsigned short*)buffer;
		if (ypitch > 100)
		{
			for ( int y = 0; y < 320; y++ )
			{
				memcpy( dst, src, 480 );
				dst += ypitch >> 1;
				src += 240;
			}
		}
		else // Must be an aero (buffer rotated by 270 degrees)
		{
			dst += 320;
			for ( int y = 0; y < 320; y++ )
			{
				unsigned short* ldst = dst;
				for ( int x = 0; x < 240; x++ )
				{
					*(ldst) = *src++;
					ldst += (xpitch >> 1);
				}
				dst += ypitch >> 1;
			}
		}
		gameX->EndDraw();
	#else
		unsigned short* dst = (unsigned short*)GXBeginDraw();
		unsigned short* src = (unsigned short*)buffer;
		if (xpitch == 2)
		{
			if (ypitch == 480)
			{
				// This is a really nice display (casio's)
				memcpy( dst, src, 240 * 320 * 2 );
			}
			else
			{
				for ( int y = 0; y < 320; y++ )
				{
					memcpy( dst, src, 480 );
					dst += 256;
					src += 240;
				}
			}
		}
		else
		{
			// Let's do this really clean (and slow:)
			if (ypitch < 0) dst += 320;
			for ( int y = 0; y < 320; y++ )
			{
				unsigned short* ldst = dst;
				for ( int x = 0; x < 240; x++ )
				{
					*(ldst) = *src++;
					ldst += (xpitch >> 1);
				}
				dst += ypitch >> 1;
			}
		}
		GXEndDraw();
	#endif
	}
	else if (BPP == 4)
	{
		// Assume this is a monochrome iPaq
		unsigned char* dst = (unsigned char*)GXBeginDraw();
		unsigned short* src = (unsigned short*)buffer;
		src += 319 * 240; // start at lower left corner
		for ( int y = 0; y < 240; y++ )
		{
			unsigned char* ldst = dst;
			unsigned short* lsrc = src;
			for ( int x = 0; x < 160; x++ )
			{
				unsigned short c1 = *(lsrc - 240);
				unsigned short c2 = *lsrc;
				c1 = ((c1 & REDMASK) >> 11) + ((c1 & GREENMASK) >> 5) + (c1 & BLUEMASK);
				c2 = ((c2 & REDMASK) >> 11) + ((c2 & GREENMASK) >> 5) + (c2 & BLUEMASK);
				*(ldst) = (c1 >> 3) + ((c2 >> 3) << 4);
				ldst++;
				lsrc -= 480;
			}
			dst += 160;
			src++;
		}
		GXEndDraw();
	}
#else
	unsigned short* dest = (unsigned short*)surface->lock();
#ifdef GREYSCALETEST
	unsigned short* src = (unsigned short*)buffer;
	for ( int y = 0; y < 320; y++ )
	{
		for ( int x = 0; x < 240; x++ )
		{
			unsigned short c1 = *src++;
			c1 = ((c1 & REDMASK) >> 11) + ((c1 & GREENMASK) >> 5) + (c1 & BLUEMASK);
			*(dest++) = ((c1 >> 2) << 11) + ((c1 >> 1) << 5) + (c1 >> 2);
		}
	}
#else
	memcpy( dest, (unsigned short*)buffer, 240 * 320 * 2 );
#endif
	surface->unlock();
	surface->copy( *console );
	console->update();
#endif
}

void update( bool updscr )
{
	if (updscr) 
		updatedisplay();
#ifdef _WIN32_WCE
	MSG msg;
	while (PeekMessage( &msg, 0, 0, 0, PM_NOREMOVE ) == TRUE)
	{
		GetMessage( &msg, 0, 0, 0 );
		TranslateMessage( &msg );
		DispatchMessage( &msg );
	}
#else
	POINT p;
	RECT r;
	GetWindowRect( hWnd, &r );
	GetCursorPos( &p );
	mxpos = p.x - r.left;
	mypos = p.y - r.top - MENU_HEIGHT;
	if (GetAsyncKeyState( VK_LBUTTON ))
	{
		if (!mbutton)
		{
			mcxpos = p.x - r.left;
			mcypos = p.y - r.top - MENU_HEIGHT;
			mcprocessed = false;
			mbutton = true;
		}
	}
	else mbutton = false;
	short bt1 = GetAsyncKeyState( VK_INSERT );
	short bt2 = GetAsyncKeyState( VK_HOME );
	short bt3 = GetAsyncKeyState( VK_DELETE );
	short bt4 = GetAsyncKeyState( VK_END );
	if (bt1 && (button1up)) { b1 = 1; button1up = false; } else b1 = 0;
	if (bt2 && (button2up)) { b2 = 1; button2up = false; } else b2 = 0;
	if (bt3 && (button3up)) { b3 = 1; button3up = false; } else b3 = 0;
	if (bt4 && (button4up)) { b4 = 1; button4up = false; } else b4 = 0;
	if (!bt1) button1up = true;
	if (!bt2) button2up = true;
	if (!bt3) button3up = true;
	if (!bt4) button4up = true;
	if (GetAsyncKeyState( VK_LEFT )) bl = 1; else bl = 0;
	if (GetAsyncKeyState( VK_RIGHT )) br = 1; else br = 0;
	if (GetAsyncKeyState( VK_UP )) bu = 1; else bu = 0;
	if (GetAsyncKeyState( VK_DOWN )) bd = 1; else bd = 0;
#endif
}

bool buttonA()
{
	bool retval = (b1 != 0);
	b1 = 0;
	return retval;
}

bool buttonB()
{
	bool retval = (b2 != 0);
	b2 = 0;
	return retval;
}

bool buttonC()
{
	bool retval = (b3 != 0);
	b3 = 0;
	return retval;
}

bool buttonD()
{
	bool retval = (b4 != 0);
	b4 = 0;
	return retval;
}

bool buttonleft()
{
	bool retval = (bl != 0);
	return retval;
}

bool buttonright()
{
	bool retval = (br != 0);
	return retval;
}

bool buttonup()
{
	bool retval = (bu != 0);
	return retval;
}

bool buttondown()
{
	bool retval = (bd != 0);
	return retval;
}

bool clicked( void )
{
	bool retval = mcprocessed;
	mcprocessed = true;
	return !retval;
}

void cursorxpos( int x )
{
	cursorx = x;
}
void cursorypos( int y )
{
	cursory = y;
}

int cursorxpos( void )
{
	return cursorx;
}

int cursorypos( void )
{
	return cursory;
}

int clickxpos( void )
{
	return mcxpos;
}

int clickypos( void )
{
	return mcypos;
}

int penxpos( void )
{
	return mxpos;
}

int penypos( void )
{
	return mypos;
}

bool pendown( void )
{
	return mbutton;
}

void line( float x1, float y1, float x2, float y2, int color )
{
	if (y1 > y2)
	{
		float h = y1; y1 = y2; y2 = h;
		h = x1; x1 = x2; x2 = h;
	}
	int len = (int)fabs( x2 - x1);
	if ((int)fabs( y2 - y1 ) > len) len = (int)fabs( y2 - y1 );
	if (len == 0) return;
	float dx = (x2 - x1) / len;
	float dy = (y2 - y1) / len;
	for ( int i = 0; i < len; i++ )
	{
		plot( int(x1), int(y1), color );
		x1 += dx;
		y1 += dy;
	}
}

void hline( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot( x1 + i, y1, color );
}

void vline( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot( x1, y1 + i, color );
}

void box( int x1, int y1, int x2, int y2, int color )
{
	hline( x1, y1, x2 - x1, color );
	vline( x1, y1, y2 - y1, color );
	hline( x1, y2, x2 - x1, color );
	vline( x2, y1, y2 - y1, color );
}

void bar( int x1, int y1, int x2, int y2, int color )
{
	for ( int i = y1; i < y2; i++ ) hline( x1, i, x2 - x1, color );
}

void bar_a( int x1, int y1, int x2, int y2, int color )
{
	for ( int i = y1; i < y2; i++ ) hline_a( x1, i, x2 - x1, color );
}

void bar( int x1, int y1, int x2, int y2, int color, unsigned long* buff )
{
	unsigned long* temp = buffer;
	buffer = buff;
	for ( int i = y1; i < y2; i++ ) hline( x1, i, x2 - x1, color );
	buffer = temp;
}

void line_x( float x1, float y1, float x2, float y2, int color )
{
	if (y1 > y2)
	{
		float h = y1; y1 = y2; y2 = h;
		h = x1; x1 = x2; x2 = h;
	}
	int len = (int)fabs( x2 - x1);
	if ((int)fabs( y2 - y1 ) > len) len = (int)fabs( y2 - y1 );
	if (len == 0) return;
	float dx = (x2 - x1) / len;
	float dy = (y2 - y1) / len;
	for ( int i = 0; i < len; i++ )
	{
		plot_x( int(x1), int(y1), color );
		x1 += dx;
		y1 += dy;
	}
}

void line_a( float x1, float y1, float x2, float y2, int color )
{
	if (y1 > y2)
	{
		float h = y1; y1 = y2; y2 = h;
		h = x1; x1 = x2; x2 = h;
	}
	int len = (int)fabs( x2 - x1);
	if ((int)fabs( y2 - y1 ) > len) len = (int)fabs( y2 - y1 );
	if (len == 0) return;
	float dx = (x2 - x1) / len;
	float dy = (y2 - y1) / len;
	for ( int i = 0; i < len; i++ )
	{
		plot_a( int(x1), int(y1), color );
		x1 += dx;
		y1 += dy;
	}
}

void hline_x( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot_x( x1 + i, y1, color );
}

void vline_x( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot_x( x1, y1 + i, color );
}

void hline_a( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot_a( x1 + i, y1, color );
}

void vline_a( int x1, int y1, int len, int color )
{
	for ( int i = 0; i <= len; i++ ) plot_a( x1, y1 + i, color );
}

void box_x( int x1, int y1, int x2, int y2, int color )
{
	hline_x( x1, y1, x2 - x1, color );
	vline_x( x1, y1, y2 - y1, color );
	hline_x( x1, y2, x2 - x1, color );
	vline_x( x2, y1, y2 - y1, color );
}

void box_a( int x1, int y1, int x2, int y2, int color )
{
	hline_a( x1, y1, x2 - x1, color );
	vline_a( x1, y1, y2 - y1, color );
	hline_a( x1, y2, x2 - x1, color );
	vline_a( x2, y1, y2 - y1, color );
}

void bar_x( int x1, int y1, int x2, int y2, int color )
{
	for ( int i = y1; i < y2; i++ ) hline_x( x1, i, x2 - x1, color );
}

void plot_x( int x1, int y1, int color )
{
	unsigned char* dest = (unsigned char*)buffer;
	int pixel = x1 + y1 * 240;
	int shft = 6 - ((pixel & 3) << 1);
	*(dest + (pixel >> 2)) = *(dest + (pixel >> 2)) ^ (color << shft);
}

void plot_a( int x1, int y1, int color )
{
	unsigned short* dest = (unsigned short*)buffer;
	int pixel = x1 + y1 * 240;
	unsigned short c1 = *(dest + pixel);
	unsigned short c2 = color;
	int r = (c1 & (31 << 11)) + (c2 & (31 << 11));
	int g = (c1 & (63 << 5)) + (c2 & (63 << 5));
	int b = (c1 & 31) + (c2 & 31);
	if (r > (31 << 11)) r = 31 << 11;
	if (g > (63 << 5)) g = 63 << 5;
	if (b > 31) b = 31;
	*(dest + pixel) = (unsigned short)(r + g + b);
}

void plot( int x1, int y1, int color )
{
	unsigned short* dest = (unsigned short*)buffer;
	*(dest + x1 + y1 * 240) = color;
}

int getpixel( int x1, int y1 )
{
	unsigned char* dest = (unsigned char*)buffer;
	int pixel = x1 + y1 * 240;
	int shft = 6 - ((pixel & 3) << 1);
	return (*(dest + (pixel >> 2)) & (3 << shft)) >> shft;
}

void clear( int color )
{
	memset( buffer, color, 320 * 480 );
}

unsigned long* getbuffer( void )
{
	return buffer;
}

#ifdef USE_GAMEX
unsigned long* getscrptr( void )
{
	return (unsigned long*)gameX->GetFBAddress();
}
#endif

void resettimer( void )
{
	basetime = GetTickCount();
}

int clock( void )
{
	return GetTickCount();
}

int msecs( void )
{
	return GetTickCount() - basetime;
}

void setchar( int c, char* c1, char* c2, char* c3, char* c4, char* c5, char* c6, char* c7 )
{
	strcpy( m_chr[c][0], c1 );
	strcpy( m_chr[c][1], c2 );
	strcpy( m_chr[c][2], c3 );
	strcpy( m_chr[c][3], c4 );
	strcpy( m_chr[c][4], c5 );
	strcpy( m_chr[c][5], c6 );
	strcpy( m_chr[c][6], c7 );
}

void initcharset()
{
	for ( int x = 0; x < 54; x++ ) for ( int y = 0; y < 7; y++ ) m_chr[x][y] = new char[10]; 
	setchar(  0, ":oooo::","o::::o:","o::::o:","oooooo:","o::::o:","o::::o:",":::::::" );
	setchar(  1, "ooooo::","o::::o:","ooooo::","o::::o:","o::::o:","ooooo::",":::::::" );
	setchar(  2, ":oooo::","o::::o:","o::::::","o::::::","o::::o:",":oooo::",":::::::" );
	setchar(  3, "oooo:::","o:::o::","o::::o:","o::::o:","o:::o::","oooo:::",":::::::" );
	setchar(  4, "oooooo:","o::::::","ooooo::","o::::::","o::::::","oooooo:",":::::::" );
	setchar(  5, "oooooo:","o::::::","ooooo::","o::::::","o::::::","o::::::",":::::::" );
	setchar(  6, ":oooo::","o::::o:","o::::::","o::ooo:","o::::o:",":oooo::",":::::::" );
	setchar(  7, "o::::o:","o::::o:","oooooo:","o::::o:","o::::o:","o::::o:",":::::::" );
	setchar(  8, ":ooooo:",":::o:::",":::o:::",":::o:::",":::o:::",":ooooo:",":::::::" );
	setchar(  9, ":::::o:",":::::o:",":::::o:","o::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 10, "o:::o::","o::o:::","ooo::::","o::o:::","o:::o::","o::::o:",":::::::" );
	setchar( 11, "o::::::","o::::::","o::::::","o::::::","o::::::","oooooo:",":::::::" );
	setchar( 12, "o::::o:","oo::oo:","o:oo:o:","o::::o:","o::::o:","o::::o:",":::::::" );
	setchar( 13, "o::::o:","oo:::o:","o:o::o:","o::o:o:","o:::oo:","o::::o:",":::::::" );
	setchar( 14, ":oooo::","o::::o:","o::::o:","o::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 15, "ooooo::","o::::o:","o::::o:","ooooo::","o::::::","o::::::",":::::::" );
	setchar( 16, ":oooo::","o::::o:","o::::o:","o:o::o:","o::o:o:",":oooo::",":::::::" );
	setchar( 17, "ooooo::","o::::o:","o::::o:","ooooo::","o:::o::","o::::o:",":::::::" );
	setchar( 18, ":oooo::","o::::::",":oooo::",":::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 19, "oooooo:","::o::::","::o::::","::o::::","::o::::","::o::::",":::::::" );
	setchar( 20, "o::::o:","o::::o:","o::::o:","o::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 21, "o::::o:","o::::o:","o::::o:","o::::o:",":o::o::","::oo:::",":::::::" );
	setchar( 22, "o::::o:","o::::o:","o::::o:","o::::o:","o:oo:o:",":o::o::",":::::::" );
	setchar( 23, "o::::o:",":o::o::","::oo:::","::oo:::",":o::o::","o::::o:",":::::::" );
	setchar( 24, ":::::o:","o:::o::",":o:o:::","::o::::","::o::::","::o::::",":::::::" );
	setchar( 25, "oooooo:","::::o::",":::o:::","::o::::",":o:::::","oooooo:",":::::::" );
	setchar( 26, ":oooo::","o:::oo:","o::o:o:","o:o::o:","oo:::o:",":oooo::",":::::::" );
	setchar( 27, "::oo:::",":o:o:::",":::o:::",":::o:::",":::o:::",":ooooo:",":::::::" );
	setchar( 28, ":oooo::","o::::o:",":::::o:",":oooo::","o::::::","oooooo:",":::::::" );
	setchar( 29, ":oooo::","o::::o:",":::oo::",":::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 30, ":::o:::","::oo:::",":o:o:::","o::o:::","oooooo:",":::o:::",":::::::" );
	setchar( 31, "oooooo:","o::::::","ooooo::",":::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 32, ":oooo::","o::::::","ooooo::","o::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 33, "oooooo:",":::::o:","::::o::",":::o:::","::o::::","::o::::",":::::::" );
	setchar( 34, ":oooo::","o::::o:",":oooo::","o::::o:","o::::o:",":oooo::",":::::::" );
	setchar( 35, ":oooo::","o::::o:","o::::o:",":ooooo:",":::::o:",":oooo::",":::::::" );
	setchar( 36, ":::::::",":::::::",":::o:::",":::::::",":::::::",":::o:::",":::::::" );
	setchar( 37, ":::oo::",":::oo::",":::oo::",":::oo::",":::::::",":::oo::",":::oo::" );
	setchar( 38, ":oooo::","::::oo:","::::oo:",":::oo::",":::::::","::oo:::","::oo:::" );
	setchar( 39, ":::::::",":::::::",":::::::","::oooo:",":::::::",":::::::",":::::::" );
	setchar( 40, ":::::::",":::::::",":::::::",":ooooo:",":::::::",":ooooo:",":::::::" );
	setchar( 41, ":::::::",":::o:::",":::o:::",":ooooo:",":::o:::",":::o:::",":::::::" );
	setchar( 42, "::::o::",":::o:::",":::o:::",":::o:::",":::o:::","::::o::",":::::::" );
	setchar( 43, ":o:::::","::o::::","::o::::","::o::::","::o::::",":o:::::",":::::::" );
	setchar( 44, ":::::::",":::::::",":::::::",":::::::",":::::::",":::::::",":::::::" );
	setchar( 45, ":::::::",":::::::",":::::::",":::::::",":::::::",":::oo::",":::oo::" );
	setchar( 46, ":::::::",":::::::",":::::::",":::::::","::::o::","::::o::",":::o:::" );
	setchar( 47, ":o:::::","::o::::",":::o:::","::::o::",":::o:::","::o::::",":o:::::" );
	setchar( 48, "::::o::",":::o:::","::o::::",":o:::::","::o::::",":::o:::","::::o::" );
	setchar( 49, "ooooooo","ooooooo","ooooooo","ooooooo","ooooooo","ooooooo","ooooooo" );
	setchar( 50, ":::o:::",":::o:::","::o::::",":::::::",":::::::",":::::::",":::::::" );
	setchar( 51, "o::::o:","::oo::o",":o::::o",":o::::o","::oo::o","o::::o:",":oooo::" );
	setchar( 52, "::::::o",":::::o:","::::o::",":::o:::","::o::::",":o:::::","o::::::" );
	setchar( 53, "o::::::",":o:::::","::o::::",":::o:::","::::o::",":::::o:","::::::o" );
	// Redir table
	char c[] = "abcdefghijklmnopqrstuvwxyz0123456789:!?-=+() .,><#'@/S";
	for ( int i = 0; i < 256; i++ ) m_transl[i] = 44;
	for ( i = 0; i < 54; i++ ) m_transl[c[i]] = i;
}

void print( char* a_String, int x1, int y1, int color )
{
	int curx = x1;
	for ( int i = 0; i < (int)(strlen( a_String )); i++ )
	{	
		long pos = 0;
		if ((a_String[i] == 10) || (a_String[i] == 13)) curx = 400;
		else if (a_String[i] == 8)
		{
			if (curx > 2) curx -= 8;
		}
		else 
		{
			if ((a_String[i] >= 'A') && (a_String[i] <= 'Z')) pos = m_transl[a_String[i] - ('A' - 'a')];
														 else pos = m_transl[a_String[i]];
			for ( int h = 0; h < 7; h++ ) for ( int v = 0; v < 7; v++ )
				if (m_chr[pos][v][h] == 'o') plot( curx + h, y1 + v, color );
			curx += 8;
		}
		if (curx > 230)
		{
			curx = 2;
			if (y1 < 300) y1 += 9;
		}
	}
	cursorx = curx;
	cursory = y1;
}

void print( char* a_String, int color )
{
	print( a_String, cursorx, cursory, color );
}

void print( char* a_String, int color, unsigned long* buff )
{
	unsigned long* temp = buffer;
	buffer = buff;
	print( a_String, cursorx, cursory, color );
	buffer = temp;
}

void print( char* a_String, int x1, int y1, int color, unsigned long* buff )
{
	unsigned long* temp = buffer;
	buffer = buff;
	print( a_String, x1, y1, color );
	buffer = temp;
}

void centre( char* a_String, int y, int color )
{
	print( a_String, (240 - strlen( a_String) * 8) / 2, y, color );
}

// =========================================
// =========================================
// MISSING STUFF FROM WINDOWS FOR WINDOWS CE
// =========================================
// =========================================

double strtod( char* t, char** e )
{
	float mul = 1;
	float retval = 0;
	char* p = t;
	while (((*p >= '0') && (*p <= '9')) || (*p == '.'))
	{
		if (*p == '.')
		{
			if (mul < 1) 
			{
				p++;
				break;
			}
			mul = 0.1f;
		}
		else
		{
			if (mul > 0.1)
			{
				retval = retval * 10 + (*p - '0');
			}
			else
			{
				retval += mul * (float)(*p - '0');
				mul *= 0.1f;
			}
		}
		p++;
	}
	e = &p;
	return (double)retval;
}

int stricmp( char* s1, char* s2 )
{
	char* a = s1;
	char* b = s2;
	while ((*a) || (*b))
	{
		char c1 = *a;
		char c2 = *b;
		if ((c1 >= 'A') && (c1 <= 'Z')) c1 -= 'a' - 'A';
		if ((c2 >= 'A') && (c2 <= 'Z')) c2 -= 'a' - 'A';
		if (c1 == c2)
		{
			a++;
			b++;
		}
		else if (c1 < c2) return -1; else return 1; 
	}
	return 0;
}

// --------------------------------------------------
// EOF