#define _GNU_SOURCE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>

#include "options.h"

#define MID_SCREEN 30
#define TRUE 0==0
#define FALSE !TRUE

#define PERR(fmt, ...)	fprintf(stderr, fmt "\n", ## __VA_ARGS__)

/* -- function declarations -- */

/* The following functions do not take arguments, but the function
prototypes specify at least one arg */
int
give_help( char* unused );

/* -- */
int
give_version( char* unused );

/* -- */
int
set_forground( char* unused );

/* These take an optional arg, so the test "if(arg)" is required. */
/* -- */
int
set_debuglevel( char* num );

/* -- */
int
set_sync( char* server );

/* -- */
int
set_random_seed( char* file );

/* These require an argument */
int
set_pidfile( char* file );

/* -- */
int
set_configfile( char* file );

/* -- */
int
set_timeout( char* sec );

/* -- end func declarations */


/* These are the option definitions.  One of each combo
(long and short with no, optional, and required arg)
is present. */
static struct tbt_option option_list[] = {
	{ .type = 't', .help_str = "About Options" },
	{	&give_version,
		'V', "version",		'n', NULL,
		"Print version and exit."
	},
	{	&give_help,
		0, "help",		'n', NULL,
		"Print help."
	},
	{	&set_forground,
		'd', NULL,		'n', NULL,
		"Runs program in the forground."
	},

	{ .type = 't', .help_str = "Main Options" },
	{	&set_debuglevel,
		'D', NULL,		'o', "level",
		"Debug level.  Default is 1."
	},
	{	&set_sync,
		0, "sync",		'o', "server",
		"Synchronize with a server,"
	},
	{ .type = 'd', .help_str = "Default is 'localhost'" },
	{ .type = '\n' },

	{	&set_random_seed,
		'R', "random-seed",	'o', "file",
		"Seed random number generator with" },
	{ .type = 'd', .help_str = "a file.  Default is the current time." },

	{	&set_timeout,
		0, "idle-timeout",	'y', "sec",
		"Time to remain idle."
	},
	{	&set_configfile,
		'F', NULL,		'y', "file",
		"Use alternate config file."
	},
	{	&set_pidfile,
		'P', "pidfile",		'y', "file",
		"Use alternate pidfile, or use"
	},
	{ .type = 'd', .help_str = "\"/dev/null\" to not write a pidfile."},
	{ 0, 0, 0, 0, 0, 0 }
};


int
main(int argc, char** argv) {
	int retval;
	char* shortopts = get_short_optstring( option_list );
	if( ! shortopts ) {
		PERR("Could not initialize short option string");
		return -1;
	}
	struct option* longopts = get_long_opttable( option_list );
	if( ! longopts ) {
		PERR("Could not initialize long option table");
		return -1;
	}

	while(TRUE) {
		int loi = 0;		/* long option index */
		char c = getopt_long(argc, argv, shortopts, longopts, &loi);
		if( c == -1 )
			break;
		if( c == '?' )
			return -1;
		retval = process_option(option_list, c, longopts[loi].name);
		if(retval)
			return retval;
	}
	free(shortopts);
	free(longopts);

	if (optind < argc) {
		printf ("non-option ARGV-elements: ");
		while (optind < argc)
			printf ("%s ", argv[optind++]);
		printf ("\n");
	}
	return 0;
} /* -- end main() -- */

/* The following functions do not take arguments, but the function
prototypes specify at least one arg */
int
give_help( char* unused ) {
	printf("Help\n");
	cat_option_help( option_list );
	return 1;	/* indicates a break from the option proccessing loop in main() */
}

/* -- */
int
give_version( char* unused ) {
	printf("Version Undefined\n");
	return -1;
}

/* -- */
int
set_forground( char* unused ) {
	PERR("Runing in forground");
	return 0;
}

/* These take an optional arg, so the test "if(arg)" is required. */
/* -- */
int
set_debuglevel( char* num ) {
	if(num)
		PERR("Setting debug level %s", num);
	else
		PERR("No debugging info");
	return 0;
}

/* -- */
int
set_sync( char* server ) {
	if(server)
		PERR("Syncing with %s", server);
	else
		PERR("Syncing with localhost");
	return 0;
}

/* -- */
int
set_random_seed( char* file ) {
	if(file)
		PERR("Setting random seed with contents of %s", file);
	else
		PERR("Setting random seed with current time");
	return 0;
}

/* These require an argument */
int
set_pidfile( char* file ) {
	if(strcmp(file, "/dev/null") == 0)
		file = NULL;
	if(file)
		PERR("Using pidfile \"%s\"", file);
	else
		PERR("Not writing a pidfile");
	return 0;
}

/* -- */
int
set_configfile( char* file ) {
	PERR("Using configfile \"%s\"", file);
	return 0;
}

/* -- */
int
set_timeout( char* sec ) {
	PERR("Setting timeout to %s seconds", sec);
	return 0;
}

