/*
 * Auto Block plugin
 * Copyright (C) 2004, Trevor "beltorak" Torrez <beltorak@prheaker.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

/* To compile the file, you must link against libpcre by setting the
enviroment variable PLUGIN_LIBS to "pcre". */

#define _GNU_SOURCE
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <pcre.h>

#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>

/* internal.h does not check to see if GAIM_PLUGINS is defined, so it goes first */
#include "internal.h"
#ifndef GAIM_PLUGINS
#  define GAIM_PLUGINS
#endif

#include "plugin.h"

//#include "pluginpref.h"
//#include "prefs.h"
#include "debug.h"
#include "util.h"
#include "cmds.h"
//#include "blist.h"
//#include "gtkconv.h"
//#include "conversation.h"
//#include "gtkdialogs.h"
#include <gaim.h>

#define BLOCK_NEVER 1
#define BLOCK_ON_MATCH 2
#define BLOCK_ALWAYS 3

#define FILTER_DISPLAY_FULL 1
#define FILTER_DISPLAY_BRIEF 2
#define FILTER_DISPLAY_NONE 3


struct autoblock_info {
	/* A hash where the key is the username that is exempt
	from autoignores, and the value is any static string */
	GHashTable*		exempt;
	/* A hash where the key is a username for which a
	conversation is currently open.  An open conversation
	is define as one in which you have sent an IM */
	GHashTable*		open;
	/* A list of regular expressions to match a nick against
	for blocking */
	GList*			nick;
	/* A list of regular expressions to match a chat message
	for blocking */
	GList*			msg;
	/* The gaim command id */
	GaimCmdId		cmd;
};

/* These are the errors in pcre.h */
static char* PCRE_ERRSTR[] = {
        "No error",
        "Pattern did not match",
        "Invalid NULL parameter",
        "Unrecognized option",
        "Invalid quoted regex",
        "Corrupted quoted regex",
        "Out Of Memory for back references",
        "Substring does not exist",
        "Match limit reached",
        "Callout function failed",
        "Invalid UTF-8 sequence",
        "UTF-8 sequence split at startoffest",
        "Invalid PCRE_ERROR code"
};

/* Holds everything needed for a regex */
struct quoted_regex {
	/* Holds the name of this regex.  May be NULL. */
	char*			name;
	/* Holds the description for this regex.  May be NULL. */
	char*			description;
	/* A copy of the string, for reference */
	char*			qr_str;
	/* Options passed to the pcre compiler */
	int				qr_opts;
	int				qr_extra_opts;
	/* The compiled pattern */
	pcre*			qr_code;
	/* Result of studying the pattern */
	pcre_extra*		qr_extra;
	/* This is the number of possible string matches in a regex */
	int				qr_stringcount;
};

/* == function declarations == */

/* - */
static void
init_plugin(GaimPlugin* plugin);

/* - */
static gboolean
plugin_load(GaimPlugin* plugin);

/* - */
static gboolean
plugin_unload(GaimPlugin* plugin);

/* - */
void
plugin_destroy(GaimPlugin* plugin);

/* - */
static GaimPluginPrefFrame *
get_plugin_pref_frame(GaimPlugin *plugin);

/* Loads preferences */
void
load_ab_prefs( void );

/* Loads a file into a GList or GHashTable */
/* This function STRONGLY violates the principle of
"do one thing and do it well"; maybe i should break it up? */
void
load_file_into_abinfo(char* name);

/* Saves the list to a file by clobbering it */
gint
save_abinfo_list(char* name);

/* Writes a hash item (key) to the channel */
void
write_hash_item(char* key, char* unused, GIOChannel** ioc);

/* Drops comments from a line. */
// TODO: remove this, use quoted_regex.description
void
drop_comment(gchar* string);

/* Wrapper for gaim_debug */
void
autoblock_debug(GaimDebugLevel level, char* fmt, ...);

#define ab_debug_misc(fmt, ...) \
	autoblock_debug(GAIM_DEBUG_MISC, "%s() " fmt "\n", __FUNCTION__, ## __VA_ARGS__)
#define ab_debug_info(fmt, ...) \
	autoblock_debug(GAIM_DEBUG_INFO, fmt "\n", ## __VA_ARGS__)
#define ab_debug_warn(fmt, ...) \
	autoblock_debug(GAIM_DEBUG_WARNING, fmt "\n", ## __VA_ARGS__)
#define ab_debug_error(fmt, ...) \
	autoblock_debug(GAIM_DEBUG_ERROR, "%s() " fmt "\n", __FUNCTION__, ## __VA_ARGS__)

// This is here just to get kdevelop auto indentation to work right again
// if()

/* Frees the abinfo GLists */
void
free_abinfo_list(GList* list);

/* Frees one item in an abinfo list */
void
free_abinfo_list_1(GList** list, GList* item);

/* The callback for the autoblock command */
GaimCmdRet
autoblock_cmd_cb(GaimConversation *conv, const gchar *cmd, gchar **args, gchar **error);

/* Callback for receiving chat messages.  A return of TRUE means to drop the message. */
gboolean
ab_chat_recv_msg_cb(GaimAccount* account, char** who, char** msg, GaimConversation* conv);

/* Blocks a user upon joining a chat room */
void
ab_user_joined_chat_cb(GaimConversation* conv, char* who);

/* Blocks all users filtered users upon joining a chat room */
void
ab_chat_joined_cb(GaimConversation* conv);

/* Callback for sending Instant Messages -- add the nick to the
open list. */
void
ab_sent_im_cb(GaimAccount* account, char* who, char* msg);

/* Callback for receiving Instant Messages */
gboolean
ab_recv_im_cb(GaimAccount* account, char** who, char** text, guint flags__unused);

/* Callback for closing an Instant Message window */
void
ab_close_im_cb(GaimConversation* conv);

/* Returns TRUE if the nick is either a buddy or in the exempted list
or the open conversations list */
gboolean
is_nick_exempt(GaimAccount* ga, char* who);

/* Checks if a nick matches a blocked regex.  Returns the regex string
if found, or NULL if not found. */
char*
is_nick_blocked(char* who);

/* Checks if a message matches a blocked regex.  Returns the regex string
if found, or NULL if not found. */
char*
is_msg_blocked(char* msg);

/* Ignores and updates the user display for a blocked user */
void
ab_do_ignore(GaimConversation* conv, char* who);

/* Returns the room shared with a user or NULL if not found */
GaimConversation*
ab_get_common_room(GList* chat, char* who, GaimAccount* account);

/* Updates the ignored indicator in a chat user list */
//void
//ab_conv_chat_update_username(GaimConversation *conv, const char *name);

/* Adds a regex or nick to the appropriate list */
GaimCmdRet
autoblock_add(GaimConversation* conv, gchar** arg);

/* Removes a regex or nick from the appropriate list.
We search by REGEX, so be careful what you remove :-P */
GaimCmdRet
autoblock_remove(GaimConversation* conv, char** arg);

/* Dumps the specified list to the chat window */
GaimCmdRet
autoblock_dump(GaimConversation* conv, gchar** arg);

/* Dumps the specified list to the conv window */
void
ab_conv_dump_list(GaimConversation* conv, char* target);


/* Dumps one has key into the conv window */
void
ab_dump_hash_item(char* key, char* unused, GaimConversation* conv);

/* Appends data to one of the files */
void
ab_append(char* name, char* data);

/* Writes a message to the conversation window */
void
ab_conv_notice(GaimConversation* conv, char* msg, ...);

/* Returns a pointer to a pcre error string */
char*
pcre_strerror_np(int errnum);

/* Returns a ptr to a mallocated quoted_regex struct */
struct quoted_regex*
qre( const char* re, int opts, int extra_opts );

#define qr(re) qre(re, 0, 0)

/* Creates a fresh new quoted_regex */
struct quoted_regex*
new_qr( void );

/* frees a created quoted_regex struct */
void
free_quoted_regex( struct quoted_regex* qr );

#if 0
/* These are apperently broken */
/* Finds a regular expression in a list by compiling it first;
returns a GList ptr if found, NULL otherwise. */
GList*
find_re(GList* list, const char* re);

/* Returns 0 if the two quoted regex's are equal */
gboolean
is_qr_equal(struct quoted_regex* apple, struct quoted_regex* orange);
#endif

/* Returns 0 if the quoted regex matches the string */
gboolean
qr_match_string(struct quoted_regex* qr, char* str);


/* autoblock_H */
