/*
 * Auto Block plugin
 * Copyright (C) 2004, Trevor "beltorak" Torrez <beltorak@prheaker.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

/* To compile the file, you must link against libpcre by setting the
enviroment variable PLUGIN_LIBS to "pcre". */

#include "autoblock.h"

/* TODO:
	/autoblock edit <target> command that pops up an editor
	and reloads the target list.

	A better way to change stored regular expressions
*/

/* -- static data -- */
static GaimPluginUiInfo prefs_info = {
	get_plugin_pref_frame
};

static GaimPluginInfo info =  {
	GAIM_PLUGIN_API_VERSION,                          /**< api_version    */
	GAIM_PLUGIN_STANDARD,                             /**< type           */
	NULL,                                             /**< ui_requirement */
	0,                                                /**< flags          */
	NULL,                                             /**< dependencies   */
	GAIM_PRIORITY_DEFAULT,                            /**< priority       */

	"yahoo_autoblock",                                /**< id             */
	N_("Yahoo Chat Auto Block"),                      /**< name           */
	"0.1.pre1",                                       /**< version        */
	                                                  /**  summary        */
	N_("Automatically blocks nicks or messages in chat rooms"),
	                                                  /**  description    */
	N_("Automatically blocks nicks or messages in chat rooms "
	"based on regular expression matches."),
	"Trevor Torrez <beltorak@phreaker.net>",          /**< author         */
	NULL,                                             /**< homepage       */

	&plugin_load,                                     /**< load           */
	&plugin_unload,                                   /**< unload         */
	&plugin_destroy,                                  /**< destroy        */

	NULL,                                             /**< ui_info        */
	NULL,                                             /**< extra_info     */
	&prefs_info,                                      /**< prefs_info     */
	NULL
};

static struct autoblock_info abinfo;

/* For the hash tables, the key is the nickname, and
the value is this static string */
static const char* AB_EXTANT = "extant";

/* If we are inside an "update" routine, we want to filter out exterranous reports */
static gboolean ab_inside_update = FALSE;
/* and if this points to a regex, then that is the only one we want information for;
if it is null, we should just be silent.... */
static char* updated_re;

/* -- function definitions -- */
static void
init_plugin(GaimPlugin* plugin)
{
	gaim_prefs_add_none(  "/plugins/core/yahoo_autoblock");
	gaim_prefs_add_int(   "/plugins/core/yahoo_autoblock/filter_display", FILTER_DISPLAY_FULL);
	gaim_prefs_add_int(   "/plugins/core/yahoo_autoblock/filter_display_size", -1);
	gaim_prefs_add_none(  "/plugins/core/yahoo_autoblock/im");
	gaim_prefs_add_int(   "/plugins/core/yahoo_autoblock/im/user_in_room", BLOCK_ON_MATCH);
	gaim_prefs_add_int(   "/plugins/core/yahoo_autoblock/im/user_not_in_room", BLOCK_ON_MATCH);
	gaim_prefs_add_int(   "/plugins/core/yahoo_autoblock/im/not_chatting", BLOCK_ON_MATCH);
	gaim_prefs_add_bool(  "/plugins/core/yahoo_autoblock/im/block_on_ignore", TRUE);
	gaim_prefs_add_bool(  "/plugins/core/yahoo_autoblock/im/ignore_on_block", FALSE);
	gaim_prefs_add_string("/plugins/core/yahoo_autoblock/exemptions", "ab_exemptions.txt");
	gaim_prefs_add_string("/plugins/core/yahoo_autoblock/nicks", "ab_nicks.txt");
	gaim_prefs_add_string("/plugins/core/yahoo_autoblock/messages", "ab_messages.txt");
}

/* - */
static gboolean
plugin_load(GaimPlugin* plugin)
{
	abinfo.exempt = NULL;
	abinfo.open   = NULL;
	abinfo.nick   = NULL;
	abinfo.msg    = NULL;

	abinfo.cmd = 0;

	abinfo.exempt = g_hash_table_new_full(
		g_str_hash, g_str_equal, g_free, NULL
	);
	abinfo.open = g_hash_table_new_full(
		g_str_hash, g_str_equal, g_free, NULL
	);

	load_ab_prefs();

	abinfo.cmd = gaim_cmd_register(
		"autoblock", "s", GAIM_CMD_P_DEFAULT,
		GAIM_CMD_FLAG_CHAT | GAIM_CMD_FLAG_IM | GAIM_CMD_FLAG_PRPL_ONLY,
		"prpl-yahoo", autoblock_cmd_cb,
		"autoblock { add | remove | dump | help } {...}"
	);

	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"receiving-chat-msg", plugin,
		GAIM_CALLBACK(ab_chat_recv_msg_cb), NULL
	);
	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"chat-buddy-joined", plugin,
		GAIM_CALLBACK(ab_user_joined_chat_cb), NULL
	);
	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"chat-joined", plugin,
		GAIM_CALLBACK(ab_chat_joined_cb), NULL
	);
	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"deleting-conversation", plugin,
		GAIM_CALLBACK(ab_close_im_cb), NULL
	);
	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"sent-im-msg", plugin,
		GAIM_CALLBACK(ab_sent_im_cb), NULL
	);

	gaim_signal_connect(
		gaim_conversations_get_handle(),
		"receiving-im-msg", plugin,
		GAIM_CALLBACK(ab_recv_im_cb), NULL
	);

	ab_inside_update = FALSE;
	/* Block turds for open chat rooms */
	GList* conv = gaim_get_chats();
	for( ; conv; conv = g_list_next(conv) )
		ab_chat_joined_cb(conv->data);

	/* Add all Instant Message instances to open list */
	conv = gaim_get_ims();
	for( ; conv; conv = g_list_next(conv) ) {
		GaimConversation* t = conv->data;
		if( g_str_equal(t->account->protocol_id, "prpl-yahoo") ) {
			ab_debug_info("adding %s to open list", t->name);
			g_hash_table_insert(
				abinfo.open,
				g_strdup(t->name),
				(char*) AB_EXTANT
			);
		}
	}
	return TRUE;
}

/* - */
static gboolean
plugin_unload(GaimPlugin* plugin)
{
	free_abinfo_list(abinfo.nick);
	free_abinfo_list(abinfo.msg);
	g_hash_table_destroy(abinfo.exempt);
	g_hash_table_destroy(abinfo.open);
	gaim_cmd_unregister(abinfo.cmd);
	return TRUE;
}

/* - */
void
plugin_destroy(GaimPlugin* plugin)
{
	ab_debug_misc("called");
}

/* - */
static GaimPluginPrefFrame *
get_plugin_pref_frame(GaimPlugin *plugin)
{
	GaimPluginPrefFrame *frame;
	GaimPluginPref *ppref;

	frame = gaim_plugin_pref_frame_new();

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/filter_display",
		_("Display blocked nicks and messages in chat window")
	);
	gaim_plugin_pref_set_type(ppref, GAIM_PLUGIN_PREF_CHOICE);
	gaim_plugin_pref_add_choice(ppref, _("Verbose"),    GINT_TO_POINTER(FILTER_DISPLAY_FULL));
	gaim_plugin_pref_add_choice(ppref, _("Brief"),      GINT_TO_POINTER(FILTER_DISPLAY_BRIEF));
	gaim_plugin_pref_add_choice(ppref, _("No Display"), GINT_TO_POINTER(FILTER_DISPLAY_NONE));
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/filter_display_size",
		_("Font size modification for autoblock notices")
	);
	gaim_plugin_pref_set_bounds(ppref, -5, 5);
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/im/not_chatting",
		_("Block Instant Messages when not chatting in a room")
	);
	gaim_plugin_pref_set_type(ppref, GAIM_PLUGIN_PREF_CHOICE);
	gaim_plugin_pref_add_choice(ppref, _("On Match"), GINT_TO_POINTER(BLOCK_ON_MATCH));
	gaim_plugin_pref_add_choice(ppref, _("Never"),    GINT_TO_POINTER(BLOCK_NEVER));
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/im/user_in_room",
		_("Block Instant Messages if the user is in the room")
	);
	gaim_plugin_pref_set_type(ppref, GAIM_PLUGIN_PREF_CHOICE);
	gaim_plugin_pref_add_choice(ppref, _("Never"),    GINT_TO_POINTER(BLOCK_NEVER));
	gaim_plugin_pref_add_choice(ppref, _("On Match"), GINT_TO_POINTER(BLOCK_ON_MATCH));
	gaim_plugin_pref_add_choice(ppref, _("Always"),   GINT_TO_POINTER(BLOCK_ALWAYS));
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/im/user_not_in_room",
		_("Block Instant Messages if the user is not in the room")
	);
	gaim_plugin_pref_set_type(ppref, GAIM_PLUGIN_PREF_CHOICE);
	gaim_plugin_pref_add_choice(ppref, _("Never"),    GINT_TO_POINTER(BLOCK_NEVER));
	gaim_plugin_pref_add_choice(ppref, _("On Match"), GINT_TO_POINTER(BLOCK_ON_MATCH));
	gaim_plugin_pref_add_choice(ppref, _("Always"),   GINT_TO_POINTER(BLOCK_ALWAYS));
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/im/ignore_on_block",
		_("Ignore a user in chat if the user's Instant Messages were blocked")
	);
	gaim_plugin_pref_frame_add(frame, ppref);

	ppref = gaim_plugin_pref_new_with_name_and_label(
		"/plugins/core/yahoo_autoblock/im/block_on_ignore",
		_("Block a user's Instant Messages if a user was ignored in a chat room")
	);
	gaim_plugin_pref_frame_add(frame, ppref);

	return frame;
}

/* Loads preferences */
void
load_ab_prefs( void )
{
	load_file_into_abinfo("exemptions");
	load_file_into_abinfo("nicks");
	load_file_into_abinfo("messages");
}

/* Loads a file into a GList or GHashTable */
/* This function STRONGLY violates the principle of
"do one thing and do it well"; maybe i should break it up? */
void
load_file_into_abinfo(char* name)
{
	char* pref_name = g_strdup_printf("/plugins/core/yahoo_autoblock/%s", name);
	char* file = g_strdup_printf(
		"%s%s%s", gaim_user_dir(), G_DIR_SEPARATOR_S, gaim_prefs_get_string(pref_name)
	);
	g_free(pref_name);
	GIOChannel* ioc;
	GIOStatus status;
	GError* error = NULL;
	gsize len;

	if( g_file_test(file, G_FILE_TEST_IS_REGULAR) ) {
		ioc = g_io_channel_new_file(file, "r", &error);
		g_free(file);
		if( ! ioc ) {
			ab_debug_error("%s", error->message);
			g_error_free(error);
			return;
		}

		if( g_str_equal(name, "nicks") || g_str_equal(name, "messages") ) {
			GList** listp;
			if( g_str_equal(name, "nicks") )
				listp = &(abinfo.nick);
			else
				listp = &(abinfo.msg);
			gchar* line;
			status = g_io_channel_read_line(ioc, &line, &len, NULL, &error);
			if( len == 0 ) {
				/* File was touched by user, just quit */
				g_io_channel_shutdown(ioc, FALSE, NULL);
				g_io_channel_unref(ioc);
				return;
			}
			if( status != G_IO_STATUS_NORMAL ) {
				ab_debug_error("%s %s: %s",
					_("Could not read from file for"),
					name,
					error->message
				);
				g_io_channel_shutdown(ioc, FALSE, NULL);
				g_io_channel_unref(ioc);
				return;
			}

			while( status == G_IO_STATUS_NORMAL ) {
				drop_comment(line);
				g_strstrip(line);
				if( *line ) {
					struct quoted_regex* qr = qr(line);
					if( qr )
						*listp = g_list_append(*listp, qr);
					else {
						ab_debug_error("%s {%s}",
							_("Failed to compile regular expression"),
							line
						);
					}
				}
				g_free(line);
				status = g_io_channel_read_line(ioc, &line, &len, NULL, &error);
			}
			if( status != G_IO_STATUS_EOF ) {
				ab_debug_error("%s %s: %s",
					_("Could not read from file for"),
					name,
					error->message
				);
				g_error_free(error);
			}
			g_io_channel_shutdown(ioc, FALSE, NULL);
			g_io_channel_unref(ioc);
		} else if( g_str_equal(name, "exemptions") ) {
			gchar* line;
			status = g_io_channel_read_line(ioc, &line, &len, NULL, &error);
			if( len == 0 ) {
				/* File was touched by user, just quit */
				g_io_channel_shutdown(ioc, FALSE, NULL);
				g_io_channel_unref(ioc);
				return;
			}
			if( status != G_IO_STATUS_NORMAL ) {
				ab_debug_error("%s %s: %s",
					_("Could not read from file for"),
					name,
					error->message
				);
				g_io_channel_shutdown(ioc, FALSE, NULL);
				g_io_channel_unref(ioc);
				return;
			}

			while( status == G_IO_STATUS_NORMAL ) {
				drop_comment(line);
				g_strstrip(line);
				if( *line )
					g_hash_table_insert(
						abinfo.exempt, g_strdup(line), (char*) AB_EXTANT
					);
				g_free(line);
				status = g_io_channel_read_line(ioc, &line, &len, NULL, &error);
			}
			if( status != G_IO_STATUS_EOF ) {
				ab_debug_error("%s %s: %s",
					_("Could not read from file for"),
					name,
					error->message
				);
				g_error_free(error);
			}
			g_io_channel_shutdown(ioc, FALSE, NULL);
			g_io_channel_unref(ioc);
		}
	} else {
		gchar* header;
		/* File does not exist, create a stub file */
		if( g_str_equal(name, "nicks") || g_str_equal(name, "messages") ) {
			header = g_strdup_printf("# %s %s %s\n",
				_("A list of"), _(name), _("to automatically block")
			);
		} else {
			header = g_strdup_printf("# %s\n",
				_("A list of nicknames exempt from automatically blocking")
			);
		}

		ioc = g_io_channel_new_file(file, "w", &error);
		g_free(file);
		if( ! ioc ) {
			ab_debug_error("%s %s: %s",
				name,
				_("file could not be created"),
				error->message
			);
			g_error_free(error);
			return;
		}
		status = g_io_channel_write_chars(ioc, header, -1, NULL, &error);
		if( status != G_IO_STATUS_NORMAL ) {
			ab_debug_error("%s %s: %s",
				name,
				_("file could not be written to"),
				error->message
			);
			g_error_free(error);
		}
		g_io_channel_shutdown(ioc, TRUE, NULL);
		g_io_channel_unref(ioc);
	}

	if( TRUE )
		ab_debug_misc("This function is so ugly");
	return;
}

/* Saves the list to a file by clobbering it */
gint
save_abinfo_list(char* name)
{
	char* pref_name = g_strdup_printf(
		"/plugins/core/yahoo_autoblock/%s", name
	);
	char* file = g_strdup_printf(
		"%s%s%s", gaim_user_dir(), G_DIR_SEPARATOR_S,
		gaim_prefs_get_string(pref_name)
	);
	g_free(pref_name);

	GError* error = NULL;
	gsize len;
	GIOStatus status;
	GIOChannel* ioc = g_io_channel_new_file(file, "w", &error);
	g_free(file);
	if( ! ioc ) {
		ab_debug_error("%s", error->message);
		g_error_free(error);
		return -1;
	}

	if( g_str_equal(name, "nicks") || g_str_equal(name, "messages") ) {
		GList* curr;
		if( g_str_equal(name, "nicks") )
			curr = abinfo.nick;
		else
			curr = abinfo.msg;
		gchar* header = g_strdup_printf("A list of %s to automatically block", name);
		gchar* data = g_strdup_printf("# %s\n", _(header));
		g_free(header);
		status = g_io_channel_write_chars(ioc, data, -1, &len, &error);
		g_free(data);
		if( status != G_IO_STATUS_NORMAL ) {
			ab_debug_error("%s", error->message);
			g_io_channel_shutdown(ioc, TRUE, NULL);
			g_io_channel_unref(ioc);
			return -1;
		}
		for( ; curr; curr = g_list_next(curr) ) {
			struct quoted_regex* qr = curr->data;
			data = g_strdup_printf("%s\n", qr->qr_str);
			g_io_channel_write_chars(ioc, data, -1, &len, &error);
			if( status != G_IO_STATUS_NORMAL ) {
				ab_debug_error("%s", error->message);
				g_io_channel_shutdown(ioc, TRUE, NULL);
				g_io_channel_unref(ioc);
				return -1;
			}
			g_free(data);
		}
		g_io_channel_shutdown(ioc, TRUE, NULL);
		g_io_channel_unref(ioc);
		return 0;
	}

	if( g_str_equal(name, "exemptions") ) {
		gchar* data = g_strdup_printf("# %s\n",
			_("A list of nicknames exempt from autoblocking")
		);
		status = g_io_channel_write_chars(ioc, data, -1, &len, &error);
		g_free(data);
		if( status != G_IO_STATUS_NORMAL ) {
			ab_debug_error("%s", error->message);
			g_io_channel_shutdown(ioc, TRUE, NULL);
			g_io_channel_unref(ioc);
			return -1;
		}
		g_hash_table_foreach(abinfo.exempt, (GHFunc) write_hash_item, &ioc);
		if( ioc ) {
			g_io_channel_shutdown(ioc, TRUE, NULL);
			g_io_channel_unref(ioc);
			return 0;
		}
		return -1;
	}

	g_return_val_if_reached(-1);
}

/* Writes a hash item (key) to the file channel */
void
write_hash_item(char* key, char* unused, GIOChannel** ioc)
{
	g_return_if_fail(ioc && *ioc);
	gsize len;
	GError* error = NULL;
	gchar* data = g_strdup_printf("%s\n", key);
	GIOStatus status = g_io_channel_write_chars(*ioc, data, -1, &len, &error);
	g_free(data);
	if( status != G_IO_STATUS_NORMAL ) {
		/* This will get bloody since there is no way to break out of the foreach!!! */
		ab_debug_error("%s", error->message);
		g_error_free(error);
		g_io_channel_shutdown(*ioc, TRUE, NULL);
		g_io_channel_unref(*ioc);
		*ioc = NULL;
	}
}

/* We can only drop comment lines :-/ */
void
drop_comment(gchar* string)
{
	g_return_if_fail(string);

	if( *string == '#' )
		*string = 0;
	return;
}

/* Wrapper for gaim_debug */
void
autoblock_debug(GaimDebugLevel level, char* fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	gaim_debug_vargs(level, info.id, fmt, ap);
	va_end(ap);
}

#if 0
/* For reference, these are in the header file */
/* Miscellaneous debug statement; includes function name */
#define ab_debug_misc(fmt, ...)
/* Informational debug statement */
#define ab_debug_info(fmt, ...)
/* A Warning debug statement */
#define ab_debug_warn(fmt, ...)
/* An error debug statement: includes function name */
#define ab_debug_error(fmt, ...)
#endif

/* Frees the abinfo GLists */
void
free_abinfo_list(GList* list)
{
	if( ! list )
		return;
	list = g_list_first(list);
	GList* curr = list;
	for( ; curr; curr = g_list_next(curr) )
		free_quoted_regex(curr->data);
	g_list_free(list);
}

/* Frees one item in an abinfo list */
void
free_abinfo_list_1(GList** list, GList* item)
{
	g_return_if_fail((list && *list && item));
	*list = g_list_remove_link(*list, item);
	free_quoted_regex(item->data);
	g_list_free(item);
}

GaimCmdRet
todo(GaimConversation *conv, const gchar *cmd)
{
	ab_conv_notice(conv, "** \"%s\" %s", cmd, _("not implemented yet"));
	return GAIM_CMD_RET_OK;
}


/* The callback for the autoblock command */
GaimCmdRet
autoblock_cmd_cb(GaimConversation *conv, const gchar *cmd, gchar **args, gchar **error)
{
	if( g_str_equal(*args, "help") ) {
		char* help[] = {
			"autoblock add { nick | msg } RE:",
			_("    adds the regular expression RE "
			"to the list of nicks or messages to automatically block"),
			"autoblock add exemption NICK:",
			_("    adds the NICK to the list of nicknames "
			"exempt from automatically blocking"),
			"autoblock remove { nick | msg | exemption } T:",
			_("    removes T from the appropriate list.  For nicks and "
			" messages, T is a regular expression, so you may want to "
			"enclose it with \\Q and \\E."),
			"dump [ nicks | messages | exemptions ]",
			_("    displays the current list for the specified type, "
			"or all of them if no type is specified"),
			NULL
		};
		int k;
		for( k = 0; help[k]; ++k )
			ab_conv_notice(conv, "%s", help[k]);
		return GAIM_CMD_RET_OK;
	}


	gchar **arg = g_strsplit(*args, " ", 3);
	if( g_str_equal(arg[0], "add") )
		return autoblock_add(conv, arg);

	if( g_str_equal(arg[0], "remove") ) {
		return autoblock_remove(conv, arg);
	}

	if( g_str_equal(arg[0], "dump") ) {
		return autoblock_dump(conv, arg);
	}

	g_strfreev(arg);
	return GAIM_CMD_RET_FAILED;
}

/* Callback for receiving chat messages.  A return of TRUE means to drop the message. */
gboolean
ab_chat_recv_msg_cb(GaimAccount* account, char** who, char** msg, GaimConversation* conv)
{
	if( ! g_str_equal(gaim_account_get_protocol_id(account), "prpl-yahoo") )
		return FALSE;

	/* Ignore ignored users */
	if( gaim_conv_chat_is_user_ignored(GAIM_CONV_CHAT(conv), *who) )
		return TRUE;

	if ( is_nick_exempt(account, *who) )
		return FALSE;

	char* match = is_msg_blocked(*msg);
	if( match ) {
		ab_do_ignore(conv, *who);
		int v = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/filter_display");
		if( v == FILTER_DISPLAY_BRIEF ) {
			ab_conv_notice(conv, "** %s %s",
				*who, _("ignored due to message pattern")
			);
		} else if( v == FILTER_DISPLAY_FULL ) {
			ab_conv_notice(conv, "** %s %s %s %s",
				_("Message from"), *who, _("matched pattern"), match
			);
		}
		return TRUE;
	}

	return FALSE;
}
/*
if()
*/

/* Blocks a user upon joining a chat room */
void
ab_user_joined_chat_cb(GaimConversation* conv, char* who)
{
	GaimAccount* account = gaim_conversation_get_account(conv);
	if( ! g_str_equal(account->protocol_id, "prpl-yahoo") )
		return;
	if( is_nick_exempt(account, who) )
		return;

	char* match = is_nick_blocked(who);
	if( match ) {
		ab_do_ignore(conv, who);
		if( ab_inside_update && updated_re ) {
			if( ! g_str_equal(updated_re, match) )
				return;
		} else if( ab_inside_update )
			return;
		int v = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/filter_display");
		if( v == FILTER_DISPLAY_BRIEF )
			ab_conv_notice(conv, "** %s %s", who, _("blocked due to nick pattern"));
		else if( v == FILTER_DISPLAY_FULL )
			ab_conv_notice(conv, "** %s %s %s", who, _("matches nick pattern"), match);
	}
}

/* Blocks all filtered users upon joining a chat room */
void
ab_chat_joined_cb(GaimConversation* conv)
{
	GaimAccount* account = gaim_conversation_get_account(conv);
	if( ! g_str_equal(gaim_account_get_protocol_id(account), "prpl-yahoo") )
		return;
	ab_conv_notice(conv, "** %s", _("Filtering chat room"));
	GaimConvChat* chat = GAIM_CONV_CHAT(conv);
	GList* users = gaim_conv_chat_get_users(chat);
	for( ; users; users = g_list_next(users) )
		ab_user_joined_chat_cb(conv, users->data);
}

/* Callback for sending Instant Messages -- add the nick to the
open list. */
void
ab_sent_im_cb(GaimAccount* account, char* who, char* msg__unused)
{
	if( ! g_str_equal(gaim_account_get_protocol_id(account), "prpl-yahoo") )
		return;

	if( ! is_nick_exempt(account, who) ) {
		ab_debug_info("%s %s", who, _("added to open list"));
		g_hash_table_insert(abinfo.open, g_strdup(who), (char*) AB_EXTANT);
	}
}

/* Callback for receiving Instant Messages */
gboolean
ab_recv_im_cb(GaimAccount* account, char** who, char** text, guint flags__unused)
{
	const char* prpl = gaim_account_get_protocol_id(account);
	if( ! g_str_equal(prpl, "prpl-yahoo") )
		return FALSE;

	/* If the nick is exempt, do no harm */
	if( is_nick_exempt(account, *who) )
		return FALSE;

	/* Find the conversation for this IM if extant */
	GList* convlist = gaim_get_ims();
	GaimConversation* im = NULL;
	for( ; convlist; convlist = g_list_next(convlist) ) {
		im = convlist->data;
		if( (gaim_conversation_get_account(im) == account) && g_str_equal(im->name, *who) )
			break;
	}
	if( ! convlist )
		im = NULL;

	/* Determines when to block */
	GaimConversation* common_room = NULL;
	GList* chatlist = gaim_get_chats();
	int when;
	if( ! chatlist )
		when = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/im/not_chatting");
	else {
		common_room = ab_get_common_room(chatlist, *who, account);
		if(
			common_room &&
			gaim_prefs_get_bool("/plugins/core/yahoo_autoblock/im/block_on_ignore") &&
			gaim_conv_chat_is_user_ignored(GAIM_CONV_CHAT(common_room), *who)
		) {
			when = BLOCK_ALWAYS;
		} else if( common_room )
			when = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/im/user_in_room");
		else
			when = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/im/user_not_in_room");
	}

	int v = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/filter_display");
	/* handle the simple cases */
	if( when == BLOCK_NEVER )
		return FALSE;

	if( when == BLOCK_ALWAYS ) {
		if( im )
			gaim_conversation_destroy(im);
		ab_debug_info("%s %s %s",
			_("Blocked Instant Message from"), *who,
			_("due to preference setting")
		);
		if( common_room && v != FILTER_DISPLAY_NONE ) {
			/* We lost some information due to separating the decision above
			from the action to block.... */
			ab_conv_notice(common_room, "** %s %s %s",
				_("Blocked Instant Message from"), *who,
				_("due to a preference setting")
			);
		}
		return TRUE;
	}

	/* Otherwise see if the nick or message matches a pattern */
	int iob = gaim_prefs_get_bool("/plugins/core/yahoo_autoblock/im/ignore_on_block");
	char* match = is_nick_blocked(*who);
	if( match ) {
		if( im )
			gaim_conversation_destroy(im);
		ab_debug_info("%s %s %s %s",
			_("Blocked Instant Message from"), *who,
			_("due to nickname matching pattern"), match
		);
		if( common_room ) {
			if( v == FILTER_DISPLAY_BRIEF ) {
				ab_conv_notice(common_room, "** %s %s",
					_("Blocked Instant Message from"), *who
				);
			} else if( v == FILTER_DISPLAY_FULL ) {
				ab_conv_notice(common_room, "** %s %s %s %s",
					_("Blocked Instant Message from"), *who,
					_("due to nickname matching pattern"), match
				);
			}

			if( iob )
				ab_do_ignore(common_room, *who);
		}
		return TRUE;
	}

	match = is_msg_blocked(*text);
	if( match ) {
		if( im )
			gaim_conversation_destroy(im);
		ab_debug_info("%s %s %s %s",
			_("Blocked Instant Message from"), *who,
			_("due to message matching pattern"), match
		);
		if( common_room ) {
			if( v == FILTER_DISPLAY_BRIEF ) {
				ab_conv_notice(common_room, "** %s %s",
					*who, _("blocked due to Instant Message pattern")
				);
			} else if( v == FILTER_DISPLAY_FULL ) {
				ab_conv_notice(common_room, "** %s %s %s",
					*who,
					_("blocked due to Instant Message matching pattern"),
					match
				);
			}

			if( iob )
				ab_do_ignore(common_room, *who);
		}
		return TRUE;
	}
	return FALSE;
}

/* Callback for closing an Instant Message window -- remove nick from the open list */
void
ab_close_im_cb(GaimConversation* conv)
{
	GaimAccount* account = gaim_conversation_get_account(conv);
	if( ! g_str_equal(gaim_account_get_protocol_id(account), "prpl-yahoo") )
		return;
	if( gaim_conversation_get_type(conv) != GAIM_CONV_IM )
		return;

	const char* name = gaim_conversation_get_name(conv);
	if( g_hash_table_lookup(abinfo.open, name) ) {
		ab_debug_info("%s %s",
			name, _("removed from open list")
		);
		g_hash_table_remove(abinfo.open, name);
	}
}

/* Returns TRUE if the nick is either a buddy or in the exempted list
or the open conversations list */
gboolean
is_nick_exempt(GaimAccount* ga, char* who)
{
	if( g_str_equal(gaim_account_get_username(ga), who) )
		return TRUE;

	if( gaim_find_buddy(ga, who) ) {
		/* Add a pref to turn this noise off */
		ab_debug_info("%s %s", who, _("is a buddy"));
		return TRUE;
	}

	if( g_hash_table_lookup(abinfo.exempt, who) ) {
		ab_debug_info("%s %s", who, _("was found in the exempt list"));
		return TRUE;
	}

	if( g_hash_table_lookup(abinfo.open, who) ) {
		ab_debug_info("%s %s", _("Open conversation with"), who);
		return TRUE;
	}

	return FALSE;
}

/* Checks if a nick matches a blocked regex.  Returns the regex string
if found, or NULL if not found. */
char*
is_nick_blocked(char* who)
{
	GList* match = g_list_first(abinfo.nick);
	while( match ) {
		if( qr_match_string(match->data, who) )
			break;
		match = g_list_next(match);
	}
	if( match ) {
		struct quoted_regex* qr = (struct quoted_regex*) match->data;
		ab_debug_info("%s %s %s", who, _("matches pattern"), qr->qr_str);
		return qr->qr_str;
	}

	return NULL;
}

/* Checks if a message matches a blocked regex.  Returns the regex string
if found, or NULL if not found. */
char*
is_msg_blocked(char* msg)
{
	GList* match = g_list_first(abinfo.msg);
	while( match ) {
		if( qr_match_string(match->data, msg) )
			break;
		match = g_list_next(match);
	}

	if( match ) {
		struct quoted_regex* qr = match->data;
		ab_debug_info("%s %s", _("message matches pattern"), qr->qr_str);
		return qr->qr_str;
	}

	return NULL;
}

/* Ignores and updates the user display for a blocked user */
void
ab_do_ignore(GaimConversation* conv, char* turd)
{
	gaim_conv_chat_ignore(GAIM_CONV_CHAT(conv), turd);
	ab_conv_chat_update_username(conv, turd);
}

/* Returns the room shared with a user or NULL if not found */
GaimConversation*
ab_get_common_room(GList* chat, char* who, GaimAccount* account) {
	g_return_val_if_fail((chat && who && account), NULL);
	GaimConversation* room = NULL;
	for( ; chat; chat = g_list_next(chat) ) {
		room = chat->data;
		if( room->account != account )
			continue;
		GList* user = gaim_conv_chat_get_users(GAIM_CONV_CHAT(room));
		for( ; user; user = g_list_next(user) ) {
			if( g_str_equal(user->data, who) )
				break;
		}
		if( user )
			break;
	}
	return room;
}

/* Updates the ignored indicator in a chat user list; this should be done
by gaim_conv_chat_ignore() and ..._unignore()! */
void
ab_conv_chat_update_username(GaimConversation *conv, const char *name) {
	GaimGtkConversation *gtkconv;
	GaimGtkChatPane *gtkchat;
	GaimConvChat *chat;
	GtkTreeModel* model;
	GtkTreeIter iter;
	gboolean found = FALSE;

	chat    = GAIM_CONV_CHAT(conv);
	gtkconv = GAIM_GTK_CONVERSATION(conv);
	gtkchat = gtkconv->u.chat;

	/* Get the data representing the list */
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(gtkchat->list));
	/* Get first list entry */
	if( ! gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter) ) {
		ab_debug_warn("%s\n", _("no users in room"));
		return;
	}

	/* search thru list for name */
	do {
		gchar* user;
		gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 1, &user, -1);
		if( g_str_equal(user, name) ) {
			found = TRUE;
			g_free(user);
			break;
		}
		g_free(user);
	} while( gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &iter) );

	if( ! found ) {
		ab_debug_error("%s %s", name, _("is not in the room"));
		return;
	}

	/* Change the indicator */
	gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		0, (gaim_conv_chat_is_user_ignored(chat, name) ? "X" : " "), -1
	);
}
/*
if()
*/

/* Adds a regex or nick to the appropriate list */
GaimCmdRet
autoblock_add(GaimConversation* conv, gchar** cmd)
{
	if(! (cmd[1] && cmd[2]) ) {
		ab_debug_error("called with a null argument");
		g_strfreev(cmd);
		return GAIM_CMD_RET_FAILED;
	}
	char* target = cmd[1];

	if( g_str_equal(target, "exemption") ) {
		char* name = cmd[2];
		if( ! g_hash_table_lookup(abinfo.exempt, name) ) {
			g_hash_table_insert(abinfo.exempt, g_strdup(name), (char*) AB_EXTANT);
			ab_append("exemptions", name);
			gaim_conv_chat_unignore(GAIM_CONV_CHAT(conv), name);
			ab_conv_chat_update_username(conv, name);
			ab_conv_notice(conv, "** %s %s", name, _("added to exempt list"));
		}
		g_strfreev(cmd);
		return GAIM_CMD_RET_OK;
	}

	if( g_str_equal(target, "nick") ) {
		char* nick = cmd[2];
		struct quoted_regex* qr = qr(nick);
		if( qr ) {
			ab_append("nicks", nick);
			ab_conv_notice(conv, "** %s: %s",
				_("added pattern to nick list"), nick
			);
			abinfo.nick = g_list_append(abinfo.nick, qr);
			updated_re = qr->qr_str;
			ab_inside_update = TRUE;
			ab_chat_joined_cb(conv);
			ab_inside_update = FALSE;
		} else {
			ab_conv_notice(conv, "** %s: %s",
				_("Failed to compile regular expression"), nick
			);
		}
		g_strfreev(cmd);
		return GAIM_CMD_RET_OK;
	}

	if( g_str_equal(target, "msg") ) {
		char* msg = cmd[2];
		struct quoted_regex* qr = qr(msg);
		if( qr ) {
			ab_append("messages", msg);
			abinfo.msg = g_list_append(abinfo.msg, qr);
			ab_conv_notice(conv, "** %s: %s",
				_("added pattern to message list"), msg
			);
		} else {
			ab_conv_notice(conv, "** %s: %s",
				_("Failed to compile regular expression"), msg
			);
		}
		g_strfreev(cmd);
		return GAIM_CMD_RET_OK;
	}

	ab_conv_notice(conv, "** %s %s", target, _("Is not a valid autoblock target"));
	g_strfreev(cmd);
	return GAIM_CMD_RET_FAILED;
}

/*
if()
*/
/* Removes a regex or nick from the appropriate list.
We search by REGEX, so be careful what you remove :-P */
GaimCmdRet
autoblock_remove(GaimConversation* conv, char** arg)
{
	if( ! (arg[1] && arg[2]) ) {
		ab_debug_error("%s", _("called with a NULL argument"));
		g_strfreev(arg);
		return GAIM_CMD_RET_FAILED;
	}

	char* target = arg[1];
	char* junk = arg[2];


	GaimConvChat* chat = GAIM_CONV_CHAT(conv);
	if( g_str_equal(target, "nick") ) {
		struct quoted_regex* qr = qr(junk);
		if( ! qr ) {
			ab_conv_notice(conv, "** %s %s",
				_("Error removing regular expression: could not compile"),
				junk
			);
			g_strfreev(arg);
			return GAIM_CMD_RET_FAILED;
		}
		GList* curr = abinfo.nick;
		while( curr ) {
			struct quoted_regex* mark = curr->data;
			if( qr_match_string(qr, mark->qr_str) ) {
				/* This is a lil convoluted cause we want to unignore everyone
				that fell under this regex; */
				GList* oopsed = gaim_conv_chat_get_users(chat);
				while( oopsed ) {
					if( qr_match_string(mark, oopsed->data) ) {
						gaim_conv_chat_unignore(chat, oopsed->data);
						ab_conv_chat_update_username(conv, oopsed->data);
					}
					oopsed = g_list_next(oopsed);
				}
				ab_conv_notice(conv, "** %s %s",
					_("Removed nick pattern"),
					mark->qr_str
				);
				free_abinfo_list_1(&(abinfo.nick), curr);
				/* Unfortunately, this means we must start over in the list... */
				curr = abinfo.nick;
			} else
				curr = g_list_next(curr);
		}
		/* Now we must re-ignore all users if they match the remaining regexs */
		ab_chat_joined_cb(conv);
		/* Finally we must save our changes -- I'm just going to rewrite the file. */
		save_abinfo_list("nicks");
		return GAIM_CMD_RET_OK;
	} else if( g_str_equal(target, "msg") ) {
		struct quoted_regex* qr = qr(junk);
		if( ! qr ) {
			ab_conv_notice(conv, "** %s %s",
				_("Error removing regular expression: could not compile"),
				junk
			);
			g_strfreev(arg);
			return GAIM_CMD_RET_FAILED;
		}
		GList* curr = abinfo.msg;
		while( curr ) {
			struct quoted_regex* mark = curr->data;
			if( qr_match_string(qr, mark->qr_str) ) {
				ab_conv_notice(conv, "** %s %s",
					_("Removed message pattern"),
					mark->qr_str
				);
				free_abinfo_list_1(&(abinfo.msg), curr);
				/* Unfortunately, this means we must start over in the list... */
				curr = abinfo.msg;
			} else
				curr = g_list_next(curr);
		}
		save_abinfo_list("messages");
	} else if( g_str_equal(target, "exemption") ) {
		if( g_hash_table_remove(abinfo.exempt, junk) ) {
			ab_conv_notice(conv, "** %s %s", _("Removed exempted nickname"), junk);
			save_abinfo_list("exemptions");
		} else
			ab_conv_notice(conv, "** %s %s", _("Could not find"), junk);
	} else {
		ab_debug_error("%s %s", _("No such list"), target);
		ab_conv_notice(conv, "** %s %s", _("No suck list"), target);
	}

	g_strfreev(arg);
	return GAIM_CMD_RET_OK;
}

/* Dumps the specified list to the chat window */
GaimCmdRet
autoblock_dump(GaimConversation* conv, gchar** arg)
{
	char* target = arg[1];
	if( !  target ) {
		ab_dump_list(conv, "nicks");
		ab_dump_list(conv, "messages");
		if( g_hash_table_size(abinfo.exempt) ) {
			ab_conv_notice(conv, "**** %s", _("Exemptions from automatically blocking"));
			g_hash_table_foreach(abinfo.exempt, (GHFunc) ab_dump_hash_item, conv);
		} else
			ab_conv_notice(conv, _("**** There are no stored exemptions"));
	} else if( g_str_equal(target, "nicks") || g_str_equal(target, "messages") )
		ab_dump_list(conv, target);
	else if( g_str_equal(target, "exemptions") ) {
		if( g_hash_table_size(abinfo.exempt) ) {
			ab_conv_notice(conv, "**** %s", _("Exemptions from automatically blocking"));
			g_hash_table_foreach(abinfo.exempt, (GHFunc) ab_dump_hash_item, conv);
		} else
			ab_conv_notice(conv, _("**** There are no stored exemptions"));
	} else
		ab_conv_notice(conv, "** %s %s", target, _("is not a valid list to display"));
	g_strfreev(arg);
	return GAIM_CMD_RET_OK;
}

/* Dumps the specified list to the conv window */
void
ab_dump_list(GaimConversation* conv, char* target)
{
	GList* curr;
	if( g_str_equal(target, "nicks") )
		curr = g_list_first(abinfo.nick);
	else
		curr = g_list_first(abinfo.msg);
	if( curr ) {
		ab_conv_notice(conv, "**** %s %s",
			target, _("to automatically ignore")
		);
		while( curr ) {
			struct quoted_regex* qr = curr->data;
			ab_conv_notice(conv, "    %s", qr->qr_str );
			curr = g_list_next(curr);
		}
	} else
		ab_conv_notice(conv, "**** %s %s", _("There are no stored"), target);
}

/* Dumps one has key into the conv window */
void
ab_dump_hash_item(char* key, char* unused, GaimConversation* conv)
{
	ab_conv_notice(conv, "    %s", key);
}

/* Appends data to one of the files */
void
ab_append(char* name, char* data)
{
	g_return_if_fail(name && data);
	ab_debug_info("Appending %s to %s file", data, name);
	char* pref_name = g_strdup_printf("/plugins/core/yahoo_autoblock/%s", name);
	char* file = g_strdup_printf(
		"%s%s%s", gaim_user_dir(), G_DIR_SEPARATOR_S,
		gaim_prefs_get_string(pref_name)
	);
	g_free(pref_name);

	GError* error = NULL;
	GIOChannel* ioc = g_io_channel_new_file(file, "a+", &error);
	g_free(file);
	if( ! ioc ) {
		ab_debug_error("%s", error->message);
		g_error_free(error);
		return;
	}

	gsize  len = 0;
	GIOStatus status;
	gchar* line = g_strdup_printf("%s\n", data);
	status = g_io_channel_write_chars(ioc, line, -1, &len, &error);
	g_free(line);
	if( status != G_IO_STATUS_NORMAL ) {
		ab_debug_error("%s", error->message);
		g_error_free(error);
	}
	g_io_channel_shutdown(ioc, TRUE, NULL);
	g_io_channel_unref(ioc);
	return;
}

/* Writes a message to the conversation window */
void
ab_conv_notice(GaimConversation* conv, char* msg, ...)
{
	/* Build the notice */
	int size = gaim_prefs_get_int("/plugins/core/yahoo_autoblock/filter_display_size");
	va_list ap;
	va_start(ap, msg);
	char* vmsg = g_strdup_vprintf(msg, ap);
	va_end(ap);
	char* emsg = gaim_escape_html(vmsg);
	g_free(vmsg);
	char* notice;
	if( size > 0 )
		notice = g_strdup_printf("</b><font color=#808080 size=+%d>%s</font>", size, emsg);
	else if( size < 0 )
		notice = g_strdup_printf("</b><font color=#808080 size=%d>%s</font>", size, emsg);
	else
		notice = g_strdup_printf("</b><font color=#808080>%s</font>", emsg);
	g_free(emsg);

	/* Turn off smileys */
	const char* sv = g_strdup(gaim_prefs_get_string("/gaim/gtk/smileys/theme"));
	GSList* smiley = smiley_themes;
	struct smiley_theme* st = NULL;
	for( ; smiley; smiley = smiley->next ) {
		st = smiley->data;
		if( g_str_equal(st->name, "none") )
			break;
	}
	if( smiley )
		gaim_prefs_set_string("/gaim/gtk/smileys/theme", st->path);
	else
		ab_debug_warn("%s: 'none'", _("Could not find smiley theme"));

	/* write notice */
	gaim_conversation_write(
		conv, "*", notice,
		GAIM_MESSAGE_SYSTEM, time(NULL)
	);
	g_free(notice);

	/* Turn smiley's back on */
	if( smiley )
		gaim_prefs_set_string("/gaim/gtk/smileys/theme", sv);

	g_free((char*) sv);
	return;
}

/* Regular expression utilities */
char*
pcre_strerror_np(int errnum)
{
	if( errnum > 0 )
        	return *PCRE_ERRSTR;
        if( errnum < -11 )
        	return *PCRE_ERRSTR + 12;

        return *(PCRE_ERRSTR -errnum);
}

struct quoted_regex*
qre( const char* re, int opts, int extra_opts )
{
	struct quoted_regex* qr = new_qr();
        qr->qr_str = g_strdup(re);
        qr->qr_opts = opts;
	qr->qr_extra_opts = extra_opts;

	const char* errptr;
	int erroffset;
	qr->qr_code = pcre_compile(re, opts, &errptr, &erroffset, NULL);
	if( ! qr->qr_code ) {
		ab_debug_error("pcre compile of '%s' failed at char %d: %s",
				re, erroffset, errptr);
		free_quoted_regex(qr);
		return NULL;
	}

	qr->qr_extra = pcre_study(qr->qr_code, 0, &errptr);
	if(errptr)
		ab_debug_info("pcre study of %s failed: %s", re, errptr);

	pcre_fullinfo(qr->qr_code, qr->qr_extra,
        		PCRE_INFO_CAPTURECOUNT, &(qr->qr_stringcount));

        ++(qr->qr_stringcount);
        return qr;
}

struct quoted_regex*
new_qr( void )
{
	struct quoted_regex* qr = g_new0( struct quoted_regex, 1 );
	qr->qr_str = NULL;
        qr->qr_code = NULL;
        qr->qr_extra = NULL;
        return qr;
}

void
free_quoted_regex( struct quoted_regex* qr )
{
	g_return_if_fail(qr);
	/* g_free() checks if the pointer is null.  Isn't that nice?
	Why don't the standard libs do that by now?? */
	g_free(qr->qr_str);
	g_free(qr->qr_code);
	g_free(qr->qr_extra);
	g_free(qr);
	return;
}

/* Returns TRUE if the quoted regex matches the string */
gboolean
qr_match_string(struct quoted_regex* qr, char* str)
{
	int ovec[qr->qr_stringcount * 3];
	int rc = pcre_exec(
		qr->qr_code, qr->qr_extra, str, strlen(str),
		0, qr->qr_opts, ovec, qr->qr_stringcount * 3
	);
	if( rc < 0 ) {
		if( rc != PCRE_ERROR_NOMATCH )
			ab_debug_error("%s: %s", _("Error during match"), pcre_strerror_np(rc));
		return FALSE;
	}

	return ( rc ? TRUE : FALSE );
}


GAIM_INIT_PLUGIN(autoblock, init_plugin, info)
/* end autoblock.c */
