/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   networking.h, part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef networking_H
#define networking_H 1

#include <sys/types.h>

#include <strtotype.h>

/* Since anything larger than 512 bytes is considered hostile intent,
we read 513 bytes in a null termininated buffer and check that size < 513 */
#define MSG_SIZE 514

/* This provides all the necessary context for a query */
typedef struct connection_context ConnContext;

struct connection_context {
	/* Yet Another Linked List */
	struct connection_context*
				next;
	/* Socket of a query */
	int			qsock;
	/* Remote host, host order */
	ulong			rhost;
	/* Local host, host order. Currently unused. */
	ulong			lhost;
	/* Remote port of query, host order */
	uint			rport;
	/* Local port of query, host order */
	uint			lport;

	/* When we are done reading, lport and rport are updated. */
	uint			finished_read:1;

	/* Length read from socket */
	size_t			query_len;
	/* Length written to socket */
	size_t			written_len;
	/* Complete size of the response, sans '\0' */
	size_t			response_len;
	/* Query buffer */
	char			query[MSG_SIZE];
	/* outgoing buffer */
	char			response[MSG_SIZE];

	/* mallocated answer */
	char*			answer;
};


/* === function reference === */

/* Sets the authport.  0 on success, -1 on error. */
int
set_authport(char* port);

/* Prints the current value of authport.  Prefixes with
'*' if state.show_settings and it is the default value */
void
print_authport( void );

/* Sets the number of maximum connections... note that
there is a hard limit based on the number of open file
descriptors, which is 255.  This function rejects anything
over 250 */
int
set_max_queries( char* max_queries );

/* Displays the maximum number of queries */
void
print_max_queries( void );

/* binds and listens to the required port */
int
start_networking( void );

/* free's a connection context and its
associated resources */
void
free_connection_context( ConnContext* cc );

/* shutdown and close a socket */
void
kill_connection(ConnContext* cc);

/* Initializes a ConnContext */
void
clear_connection_context(ConnContext* cc);

/* Creates and returns a cleared ConnContext */
ConnContext*
new_connection_context( void );

/* Clears the list of sockets */
void
clean_cc_list( void );

/* Adds a new socket to the ConnContext list.
Returns 0 if successful, -1 on error. */
int
ref_socket(int fd, ulong rhost);

/* Retrives a ConnContext by filedescriptor.
Returns ConnContext* if found, NULL otherwise */
ConnContext*
get_connection_context(int fd);

/* Deletes an entry from the list of 'ConnContext's; free's
its resources.  Sets the passed ptr to the next ConnContext
in the list.  Returns 0 on success, -1 otherwise. */
int
unref_connection_context(ConnContext** cc_p);

/* Reads from a socket filling it's ConnContext.
On socket error, the socket is killed.  */
void
read_query(ConnContext* cc);

/* Writes to a socket from it's ConnContext.
On error, the socket is closed and set to -1.  */
void
write_response(ConnContext* cc);

/* Sets the response in a ConnContext, truncates to 512 bytes.
The answer must have room for the port pair, three spaces, a
comma, a colon, and a CR && LF, because the response must be
less than 512 bytes.  Returns 0 on success, -1 on error. */
int
set_response(ConnContext* cc, char* answer);

/* Gets the socket owner given the remote host and the
port pair.  Returns 0 on success, a c_error_t on error. */
c_error_t
get_sock_owner(uint lport, ulong rhost, uint rport, uid_t* owner);

/* Prepares a ConnContext for writing. Returns 0 on
success; -1 on error.  */
int
prepare_for_write(ConnContext* cc);

/******
 * The socket loop checks all referenced sockets
 * for data available for read, write, or exceptions.
 * Can you tell i've been reading glib?
 ******/

/* Initializes the socket loop */
int
sockloop_prepare( void );

/* Waits for a socket event.  Returns the number of socket
events (and errno will be EINTR if a signal was caught),
-1 on error. */
int
sockloop_select( void );

/* Performs socket actions on each socket.
0 on success, -1 on error.  If anything was interrupted by a signal,
we return with 0 and errno == EINTR */
int
sockloop_perform( void );

/* Performs one iteration thru the socket loop; returns 0

success (or if EINTR), -1 on error */
int
sockloop_iter( void );

/* Continuously loops thru the socket loop; returns

0 on success, -1 on error, or if an unhandled signal was caught. */
int
sockloop_do( void );


#endif /* networking_H */
