/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   xmalloc.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* Wrappers for various mallocating routines.  Exits on out of memory
conditions.  Checks for invalid and doubly-freed ptrs.  */

/* If this is to be used in pthread code, be sure to define THREADS */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <stdbool.h>

#if defined(THREADS)
#  include <pthread.h>
#  define XMALLOC_THREADS(x)     { x }
#  define XMALLOC_NO_THREADS(x)
#else
#  define XMALLOC_THREADS(x)
#  define XMALLOC_NO_THREADS(x)  { x }
#endif

#include "xmalloc.h"


/*
if()
*/

/* ** -- Pointer tracking stuff -- ** */
typedef struct {
	void**			start;
	size_t			endix;
	size_t			blk_size;
} xmptr_t;

static xmptr_t* gl_xmptrs = NULL;

XMALLOC_THREADS(
	pthread_mutex_t   gl_xmptrs_lock = PTHREAD_MUTEX_INITIALIZER;
)


/* ** -- Pointer tracking methods -- used internally -- ** */

/* Initializes the memory pointer list if not already done. */
void
xmptrs_init( void );

/* Checks the list for a pointer */
bool
xmptr_exists(void* p);

/* Grows the xmptrs list -- Requires the list be LOCKED */
void
xmptrs_grow( void );

/* ** -- end private xmptrs methods -- ** */


/* Exits the app or thread, whichever is relevent */
void
xmalloc_exit(int rc)
{
XMALLOC_THREADS(
	fprintf(stderr, "Thread exhuasted available memory\n");
	pthread_exit(NULL);
)
XMALLOC_NO_THREADS(
	fprintf(stderr, "Application exhuasted available memory\n");
	exit(rc);
)
}

void*
xmalloc(size_t bytes)
{
	xmptrs_init();
	if( ! bytes )
		return NULL;
	void* p = malloc(bytes);
	if( ! p )
		xmalloc_exit(-1);
	xmalloc_ptr_ref(p);
	return p;
}

void*
xcalloc(size_t size, size_t num)
{
	xmptrs_init();
	if( ! (size && num) )
		return NULL;
	void* p = calloc(size, num);
	if( ! p )
		xmalloc_exit(-1);
	xmalloc_ptr_ref(p);
	return p;
}

void*
xrealloc(void* p, size_t s)
{
	if( ! (p || s) )
		return NULL;

	xnullfree(p);
	return xmalloc(s);
}

void
xfree(void* p)
{
	if( ! p ) {
		fprintf(stderr, "Attempted to free NULL pointer!\n");
		return;
	}

	if( xmptr_exists(p) )
		xmalloc_ptr_unref(p);
	else {
		fprintf(stderr, "Attempted to free invalid pointer: %p!\n", p);
		return;
	}

	free(p);
	return;
}

/*
if()
*/

/* Initializes the memory pointer list if not already done. */
void
xmptrs_init( void )
{
	/* Initialize the global xmptr list */
	XMALLOC_THREADS(
		pthread_mutex_lock(&gl_xmptrs_lock);
	)
	if( ! gl_xmptrs ) {
		gl_xmptrs = malloc(sizeof(xmptr_t));
		if( ! gl_xmptrs )
			xmalloc_exit(-1);
		gl_xmptrs->start = malloc(100 * sizeof(void*));
		if( ! gl_xmptrs->start )
			xmalloc_exit(-1);

		gl_xmptrs->endix = 0;
		gl_xmptrs->blk_size = 100;
		size_t k;
		for( k = 0; k < 100; ++k )
			gl_xmptrs->start[k] = NULL;
	}
	XMALLOC_THREADS(
		pthread_mutex_unlock(&gl_xmptrs_lock);
	)
}

/*
if()
*/

/* Checks both global and thread specific lists for a pointer */
bool
xmptr_exists(void* p) {
	if( ! p ) {
		fprintf(stderr, "%s() Attempted to check for NULL ptr!\n");
		return false;
	}

	if( ! (gl_xmptrs && gl_xmptrs->endix) )
		return false;

	XMALLOC_THREADS(
		pthread_mutex_lock(&gl_xmptrs_lock);
	)
	size_t k;
	for( k = 0; k <= gl_xmptrs->endix; ++k ) {
		if( p == gl_xmptrs->start[k] )
			break;
	}

	if( p == gl_xmptrs->start[k] ) {
		XMALLOC_THREADS(
			pthread_mutex_unlock(&gl_xmptrs_lock);
		)
		return true;
	}
	XMALLOC_THREADS(
		pthread_mutex_unlock(&gl_xmptrs_lock);
	)

	return false;
}

/* Grows the xmptrs list -- Requires the list be LOCKED */
void
xmptrs_grow( void )
{
	gl_xmptrs->start = realloc(
		gl_xmptrs->start,
		(sizeof(void*) * gl_xmptrs->blk_size + 100)
	);
	if( ! gl_xmptrs->start )
		xmalloc_exit(-1);
	int k = 0;
	while(k < 100)
		gl_xmptrs->start[gl_xmptrs->blk_size + k++] = NULL;
	gl_xmptrs->blk_size += 100;
}

/* Adds a (non-duplicate) pointer to the lists */
void
xmalloc_ptr_ref(void* p)
{
	if( ! p )
		return;

	if( ! xmptr_exists(p) ) {
		XMALLOC_THREADS(
			pthread_mutex_lock(&gl_xmptrs_lock);
		)
		if( gl_xmptrs->endix >= gl_xmptrs->blk_size - 1 )
			xmptrs_grow();
		gl_xmptrs->start[gl_xmptrs->endix++] = p;
		XMALLOC_THREADS(
			pthread_mutex_unlock(&gl_xmptrs_lock);
		)
	}
}

/* Removes a ptr from the list. */
void
xmalloc_ptr_unref(void* p)
{
	if( ! p )
		return;
	XMALLOC_THREADS(
		pthread_mutex_lock(&gl_xmptrs_lock);
	)
	size_t k;
	for( k = 0; k < gl_xmptrs->endix; ++k ) {
		if( gl_xmptrs->start[k] == p )
			break;
	}

	if( gl_xmptrs->start[k] == p ) {
		if( k != gl_xmptrs->endix - 1 )
			gl_xmptrs->start[k] = gl_xmptrs->start[gl_xmptrs->endix - 1];
		gl_xmptrs->endix--;
		gl_xmptrs->start[gl_xmptrs->endix] = NULL;
	}
	XMALLOC_THREADS(
		pthread_mutex_unlock(&gl_xmptrs_lock);
	)
}


/* -- end xmalloc.c -- */
