/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   sfopen.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <sfopen.h>

#include <xstring.h>
#include <logutils.h>
#include <procprivs.h>

/* These routines open a file while satisfying
proc priv requirements. */

/* Opens a file for writing; the file must not exist.
Returns the file descriptor or -1 on error.  Does not report. */
int
sopen_write(char* file) {
        int fd;
        int fflags = O_WRONLY | O_CREAT | O_EXCL;
        mode_t fmode = S_IRUSR | S_IWUSR | S_IRGRP;

        fd = open(file, fflags, fmode);
        if( fd < 0 ) {
        	/* In order to continue,
                    the failure must be because of privs, and
                    we must be able to change privs
                */
        	if( ! ( (errno == EACCES) && proc_euid_mutable ) )
                	return -1;

                /* If the failure was due to privs, try higher privs */
                raise_proc_privs();
                fd = open(file, fflags, fmode);
                suspend_proc_privs();
        }

        return fd;
} /* -- end sopen() -- */

/* Opens a file for writing.  Returns FILE* on success,
NULL on error.  It is an error for the file to exist.  */
FILE*
sfopen_write(char* file) {
	FILE* FP = NULL;
        int fd = sopen_write(file);
        if( fd < 0 )
                return NULL;
	else
        	FP = fdopen(fd, "w");

        return FP;
} /* -- end sfopen_write() -- */

/* Opens a file for appending. */
FILE*
sfopen_append(char* file) {
	FILE* FP = NULL;
        FP = fopen(file, "a");
        if( ! FP ) {
        	if( proc_euid_mutable && (errno == EACCES) ) {
                        raise_proc_privs();
                        FP = fopen(file, "a");
                        suspend_proc_privs();
                }
        }

        return FP;
} /* -- end sfopen_append() -- */

/* Opens a file for reading.  Calls sfcheck_read().
Returns FILE* on success, -1 on error. */
FILE*
sfopen_read(char* file, uid_t uid) {
	FILE* FP = NULL;
        FP = fopen(file, "r");
        if( ! FP ) {
		if( errno == ENOENT )
			return NULL;
        	if( proc_euid_mutable && (errno == EACCES) ) {
                	raise_proc_privs();
                        FP = fopen(file, "r");
                        suspend_proc_privs();
                }

                if( ! FP )
                        return NULL;
        }

        if( sfcheck_read(FP, uid) ) {
        	carp("Security check failed for %s", file);
                fclose(FP);
                return NULL;
        }

	return FP;
} /* -- end sfopen_read() -- */

/* Checks that a file is safe to read.
Returns 0 on success, -1 on failure:
    if uid != 0
        FAIL if file owner != uid or root
	FAIL if the file is greater than
	    max_userfile_size (currently 5k)
*/
int
sfcheck_read(FILE* FP, uid_t uid) {
	if( uid == 0 )
        	return 0;

        if( ! FP ) {
        	carp("Cannot check NULL pointer");
                return -1;
	}
        struct stat st;
        if( fstat(fileno(FP), &st) ) {
        	choke("Could not stat file");
                return -1;
        }

        /* To continue, either:
            the file is owned by root,
            or the file is owned by uid
        */
        if( !(( st.st_uid == 0) || (st.st_uid == uid)) ) {
                carp("File is not owned by uid %lu or root!!",
                                uid, st.st_uid);
                return -1;
        }

	/* The file size must be less than 5k */
	if( st.st_size > 5*1024 ) {
		carp("User file too big to read");
		return -1;
	}
	return 0;
} /* -- end sfcheck_read() -- */

/* Checks that a file is not writable by a uid, any
of it's group memberships, or world.  Returns 0 on success,
-1 on failure.  */
int
sfcheck_readonly(FILE* FP, uid_t uid) {
	if( uid == 0 )
        	return 0;

        struct stat st;

        if( fstat(fileno(FP), &st) ) {
        	choke("Could not fstat file");
                return -1;
        }

        if( st.st_uid == uid ) {
        	carp("File is owned by uid %lu", uid);
                return -1;
        }

        if( st.st_mode & S_IWOTH ) {
        	carp("File is world writable");
                return -1;
        }

        /* If it is not group writable, then no groups
        need be checked */
        if( !(st.st_mode & S_IWGRP) )
        	return 0;

        struct passwd* pwd = getpwuid(uid);
        /* If there was no entry, then no groups need to be checked */
        if( ! pwd ) {
        	if(
                	(errno == ENOMEM)	||
                	(errno == EIO)		||
                	(errno == EINTR)	||
                	(errno == EMFILE)	||
                	(errno == ENFILE)
                ) {
                	choke("Could not get passwd entry for uid");
                        return -1;
                } else {
                	return 0;
                }
        }

        if( st.st_gid == pwd->pw_gid ) {
        	carp("File is writable by uid's primary gid");
                return -1;
        }

        struct group* grp = getgrgid(st.st_gid);
        if( ! grp ) {
        	if(
                	(errno == ENOMEM)	||
                	(errno == EIO)		||
                	(errno == EINTR)	||
                	(errno == EMFILE)	||
                	(errno == ENFILE)
                ) {
                	choke("Could not get group entry for file's gid");
                        return -1;
                } else {
                	/* The file's gid has no members... */
                	return 0;
                }
        }

        int i;
        for( i = 0; grp->gr_mem[i]; ++i ) {
        	if( xstr_is_equal(grp->gr_mem[i], pwd->pw_name) ) {
                	carp("File writable by uid %lu's group %lu",
                			pwd->pw_uid, st.st_gid);
                        return -1;
                }
        }

        return 0;

} /* -- end sfcheck_readonly() -- */


/* -- end sfopen.c -- */
