/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   procprivs.h, part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*----------------------
 * Privilage model:
 *      (1) -- privilaged
 *      (2) -- semi-privilaged
 *      (3) -- unprivilaged
 *
 *              [invoking user] | [executable's owner] | [global unpriv user]
 *                  root (1)    |       N / A          |      #65535 (3)
 *               joeBlow (2)    |       root (1)       |      #65533 (3)
 *               joeBlow (3)    |     daemon (1)       |       N / A
 ----------------------*/


#ifndef procprivs_H
#define procprivs_H 1

#include <unistd.h>
#include <sys/types.h>

/* Holds the process permission information */
struct process_uid {
	/* Real (invoking process') uid and gid */
	uid_t		ruid;
        gid_t		rgid;
        /* Corresponds to the 'set-userid' bit on the executable */
        uid_t		euid;
        gid_t		egid;
        /* The eventual uid and gid to run as */
        uid_t		uuid;
        gid_t		ugid;
        /* Whethor or not we can change privs. */
        uint		euid_mutable:1;
};
extern struct process_uid proc_uid;

extern uint server_runas_uid_set;
extern uint server_runas_gid_set;

/* The security model differences are giving me a headache.  I am so
tempted to drop suid support and abort if suid. */

/* If the invoking user is root we can bypass some checks */
#define proc_root_user		( (proc_uid.ruid == 0) ? true : false )
/* If we are root, we can change to arbitrary uids */
#define proc_root		( (proc_uid.euid == 0) || (proc_uid.ruid == 0) ? true : false )

/* Only set if the executable is suid and invoked by a user other than the executable's owner */
#define proc_suid		( (proc_uid.ruid != proc_uid.euid) ? true : false )
/* If the invoking user is not root and the executable is suid and owned by root */
#define proc_suid_root		( (proc_suid && (proc_uid.euid == 0)) ? true : false )

/* If we can change uids on the fly */
#define proc_euid_mutable	proc_uid.euid_mutable



#define		server_runas_uid	proc_uid.uuid
#define		server_runas_gid	proc_uid.ugid

/* === function reference === */

/* Stores process privilages.  */
void
store_proc_privs( void );

/* Temporarily changes to server_runas_ privilages if root,
invoking user privs otherwise. */
void
suspend_proc_privs( void );

/* Raises the process privs to:
    to the executable's owner if suid-non-root,
    or to the invoking user's privs otherwise
*/
void
raise_proc_privs( void );

/* Sets the privilages to the highes allowed */
void
proc_to_prime_privs();

/* Sets the process privs to root if possible */
void
proc_to_root_privs();

/* Regains the original privs allowed to the process */
void
regain_proc_privs( void );

/* Sets the process' privilages to that of
the invoking user's. */
void
proc_to_user_privs( void );

/* Drops process' privilages */
int
drop_proc_privs( void );


#endif /* procprivs_H */
