/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   procprivs.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>

#include "procprivs.h"

#include <defaults.h>
#include <logutils.h>
#include <state.h>

struct process_uid proc_uid = {0,0,0, 0,0,0, 0};

uint server_runas_uid_set = false;
uint server_runas_gid_set = false;

/* Stores process privilages.  */
void
store_proc_privs( void ) {
        proc_uid.euid = geteuid();
        proc_uid.ruid = getuid();
        proc_uid.egid = getegid();
        proc_uid.rgid = getgid();

        /* store unprivilaged uid and gid */
	strtouidt(&(proc_uid.uuid), default_runas_uid);
        strtogidt(&(proc_uid.ugid), default_runas_gid);

        proc_euid_mutable = (proc_root || proc_suid) ? true : false;
} /* -- end store_proc_privs() -- */

/* Temporarily changes to server_runas_ privilages if root,
invoking user privs otherwise. */
void
suspend_proc_privs( void ) {
        if( ! proc_euid_mutable )
        	return;

        if( proc_root ) {
        	proc_to_prime_privs();
                setegid(proc_uid.ugid);
                seteuid(proc_uid.uuid);
        } else {
        	/* proc_suid &! proc_root */
        	proc_to_user_privs();
        }
} /* -- end suspend_proc_privs() -- */

/* Raises the process privs to:
    to the executable's owner if suid but not root,
    or to the invoking user's privs otherwise
*/
void
raise_proc_privs( void ) {
	if( ! proc_euid_mutable )
        	return;
        if( proc_suid &! proc_root )
        	regain_proc_privs();
        else
        	proc_to_user_privs();
} /* -- end raise_proc_privs() -- */

/* Sets the privilages to the highest allowed */
void
proc_to_prime_privs() {
	if( ! proc_euid_mutable )
        	return;

        if( proc_suid &! proc_root )
        	regain_proc_privs();
        else
        	proc_to_root_privs();
} /* -- end proc_to_prime_privs() -- */

/* Sets the process privs to root if possible */
void
proc_to_root_privs() {
	if( ! proc_root )
        	return;

        if( geteuid() != 0 )
        	seteuid(0);
        if( getegid() != 0 )
        	setegid(0);
} /* -- end proc_to_root_privs() -- */

/* Regains the original privs allowed to the process */
void
regain_proc_privs( void ) {
        if( ! proc_euid_mutable )
        	return;

        if( proc_uid.euid != geteuid() )
                setegid(proc_uid.egid);
        if( proc_uid.egid != getegid() )
                seteuid(proc_uid.euid);
} /* -- end regain_proc_privs() --*/

/* Sets the process' privilages to that of the invoking user's.
If not suid, is essentially regain_proc_privs(). */
void
proc_to_user_privs( void ) {
        if( ! proc_euid_mutable )
        	return;

        if( proc_uid.ruid != geteuid() )
		seteuid(proc_uid.ruid);
        if( proc_uid.rgid != getegid() )
		setegid(proc_uid.rgid);

} /* end proc_to_user_privs() -- */

/* Drops process' privilages */
int
drop_proc_privs( void ) {
	if( ! proc_euid_mutable ) {
        	carp("Cannot set process uid when not suid or root");
                return 0;
        }

        regain_proc_privs();
        if( proc_root ) {
                setresgid(proc_uid.ugid, proc_uid.ugid, proc_uid.ugid);
                setresuid(proc_uid.uuid, proc_uid.uuid, proc_uid.uuid);
        } else {
        	carp("Cannot set process uid/gid to %lu/%lu; using %lu/%lu instead",
                                proc_uid.uuid, proc_uid.ugid, proc_uid.ruid, proc_uid.rgid);
                setresgid(proc_uid.rgid, proc_uid.rgid, proc_uid.rgid);
                setresuid(proc_uid.ruid, proc_uid.ruid, proc_uid.ruid);
        }

        /* Check to ensure that we cannot gain privs back */
        if( ! setuid(0) ) {
        	carp("Gained root privs!  Aborting!");
                return -1;
        }

        if( ! setuid(proc_uid.euid) ) {
        	carp("Regained suid privs!  Aborting!");
                return -1;
        }

        if( ! seteuid(proc_uid.euid) ) {
        	carp("Regained euid privs!  Aborting!");
                return -1;
        }

        proc_euid_mutable = false;
        return 0;
} /* -- end drop_proc_privs() -- */


/* -- end procprivs.c -- */
