/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   logutils.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <stdarg.h>
#include <unistd.h>
#include <time.h>

#include "logutils.h"

#include <state.h>
#include <xmalloc.h>
#include <xstring.h>

/* writes to stderr */
void
c_logf(char* str) {
        /* log string */
        fprintf(stderr, "%s\n", str);
        fflush(stderr);
        fsync(fileno(stderr));
	/* and return */
        return;
} /* -- end c_logf() -- */

/* Logs an error to stderr in an errno-specific way */
void
freporterr(char* file, uint line, char* func, int errnum, char* fmt, ...)
{
        char* str, * tmp;
	int retval;
        va_list ap;
        va_start(ap, fmt);
        retval = vasprintf( &str, fmt, ap );
        va_end(ap);
        if( retval < 0 )
		xmalloc_exit(-1);
	xmalloc_ptr_ref(str);
        /* Prepend function name */
        tmp = xsprintf("%s()  %s", func, str);
        if( tmp ) {
        	xfree(str);
                str = tmp;
        }

        /* If debugging requested, prepend file, line, and function */
        if( state.debug ) {
                tmp = xsprintf("%s (line %d):: %s", file, line, str);
                if(tmp) {
                        xfree(str);
                        str = tmp;
                }
        }

        /* prepend time */
        char* timestr = get_timestr();
        if( timestr ) {
        	tmp = xsprintf("%s %s", timestr, str);
                if(tmp) {
                	xfree(str);
                        str = tmp;
                }
                xfree(timestr);
        }

        /* Append error string */
        if( errnum ) {
        	char* err_str = get_strerror_r(errnum);
                if( err_str ) {
        		tmp = xsprintf("%s -- %s", str, err_str);
                        xfree(err_str);
                        if( tmp ) {
                                xfree(str);
                                str = tmp;
                        }
                }
        }

        /* Log it */
        if(str) {
            c_logf(str);
            xfree(str);
        }

        /* Reset errno -- note, for carp this clears errno */
        errno = errnum;
} /* -- end freporterr() -- */

/* Logs status to stderr */
void
freportstat(char* fmt, ...)
{
        char* str, * tmp;
        int retval;
        va_list ap;
        va_start(ap, fmt);
        retval = vasprintf( &str, fmt, ap );
        va_end(ap);
                /* If that worked, return str */
        if( retval < 0 )
		xmalloc_exit(-1);
	xmalloc_ptr_ref(str);

        char* timestr = get_timestr();
        if(timestr) {
        	tmp = xsprintf("%s %s", timestr, str);
                if(tmp) {
                	xfree(str);
                        str = tmp;
                }
                xfree(timestr);
        }

        if(str) {
                c_logf(str);
                xfree(str);
        }

} /* -- end freportstat() -- */

/* allocates and returns a time string for logfiles */
char*
get_timestr( void ) {
        struct tm currtm;
        time_t currtime;
        time(&currtime);
        localtime_r(&currtime, &currtm);
        return xsprintf("%.2d/%.2d/%.4d %.2d:%.2d:%.2d",
        		currtm.tm_mon+1, currtm.tm_mday, currtm.tm_year+1900,
                	currtm.tm_hour, currtm.tm_min, currtm.tm_sec);
} /* -- end get_timestr() -- */

/* Allocates and returns the string for an errno.  Masks the
incompatibility between glibc and other (standard) libraries. */
char*
get_strerror_r(int errnum) {
	char* errstr = NULL;
#ifdef STRERROR_R_CHAR_P
	errstr = xstrdup(strerror_r(errno, NULL, 49));
#else /* strerror_r() returns an int */
	errstr = xcalloc(50, sizeof(char) );
	if( strerror_r(errno, errstr, 49) )
        	xnullfree(errstr);
#endif
	return errstr;
} /* -- end get_strerror_r() -- */


/* -- end logutils.c -- */
