/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   response.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include "response.h"

#include <xmalloc.h>
#include <xstring.h>
#include <logutils.h>
#include <ansconf.h>
#include <anssettings.h>
#include <fileutils.h>
#include <reutils.h>
#include <strtotype.h>
#include <serverconf.h>

static AnswerConfig* ac_list = NULL;
struct timeval last_clean = {0,0};	/* To clean the cache periodically */

/*******
 * Answer caching utils
 *******/

/* Gets a cached answer for a user; creates one if necessary.
Returns 0 if no error, ET_UNKNOWN on error */
c_error_t
get_cached_answer(AnswerConfig** ac, uid_t owner, char* username) {
	/* Check for a cached answer */
	dreport("Checking cache for %d", owner);
	check_cache(ac, owner);
	if( *ac )
		return 0;

	/* create one if not extant */
	dreport("Creating AnswerConfig");
	*ac = new_answer_config();
	if( cp_answer_config(&default_ans_conf, *ac) ) {
		carp("Could not copy default_ans_conf for \"%s\"", username);
		free_answer_config(*ac);
		*ac = NULL;
		return ET_UNKNOWN;
	}

	(*ac)->username = xstrdup(username);
	(*ac)->uid = owner;
	if( server.userdir && parse_userfile(*ac) ) {
		free_answer_config(*ac);
		*ac = new_answer_config();
		if( cp_answer_config(&default_ans_conf, *ac) ) {
			carp("Could not copy default_ans_conf for \"%s\"", username);
			free_answer_config(*ac);
			*ac = NULL;
			return ET_UNKNOWN;
		}
		if( (*ac)->error_type == ET_ACTUAL )
			(*ac)->error_type = ET_UNKNOWN;
		(*ac)->username = xstrdup(username);
		(*ac)->uid = owner;
		(*ac)->file = xsprintf("%s/%s", server.userdir, username);
	}

	dreport("Generating answer string");
	if( gen_answer_string(&((*ac)->cached_answer), *ac) ) {
		carp("Could not store cached answer for \"%s\"", username);
		free_answer_config(*ac);
		*ac = NULL;
		return ET_UNKNOWN;
	}

	(*ac)->next = ac_list;
	ac_list = *ac;
	(*ac)->accessed = time(NULL);
	dreport("Added cached AnswerConfig %p to list", *ac);
	return 0;
} /* -- end get_cached_answer() -- */

/* This retrieves a cached answer for the passed uid.
If the cached answer exists, it is returned in the
passed ptr. */
void
check_cache(AnswerConfig** ac, uid_t owner) {
	if( ! ac_list ) {
		*ac = NULL;
		return;
	}

	AnswerConfig* curr = ac_list;
	AnswerConfig* last = NULL;
	AnswerConfig* tmp;
	while( curr ) {
		if( curr->uid == owner ) {
			if( is_outdated(curr) ) {
				tmp = curr->next;
				if(last)
					last->next = curr->next;
				else
					ac_list = curr->next;
				dreport("Deleting cached answer @ %p", curr);
				free_answer_config(curr);
				curr = tmp;
				*ac = NULL;
				continue;
			}
			*ac = curr;
			(*ac)->accessed = time(NULL);
			dreport("Found cached AnswerConfig %p", *ac);
			last = curr;
			curr = curr->next;
			continue;
		}
		last = curr;
		curr = curr->next;
	}
} /* -- end check_cache() -- */

/* Cleans out expired cached entries. */
void
clean_cache( void ) {
	dreport("Executing Purge...");
	AnswerConfig* curr = ac_list;
	AnswerConfig* last = NULL;
	AnswerConfig* tmp;
	while( curr ) {
		if( time(NULL) > (curr->accessed + server.cache_age) ) {
			dreport("deleting expired cache entry for %d (%p)",
					curr->uid, curr);
			tmp = curr->next;
			if(last)
				last->next = curr->next;
			else
				ac_list = curr->next;
			free_answer_config(curr);
			curr = tmp;
			continue;
		}
		last = curr;
		curr = curr->next;
	}
	last_clean.tv_sec = time(NULL);
} /* -- end clean_cache() -- */

/* Checks to see if a cached answer is outdated.
Returns true if the cached answer is outdated, false
otherwise.  */
uint
is_outdated(AnswerConfig* ac) {
	if( (! ac) || xstr_is_nil(ac->file) ) {
                carp("Invalid AnswerConfig reference!");
		exit(-1);
	}

        struct stat st;

        if( ! ac->modified ) {
        	if(
                    stat(ac->file, &st) &&
                    (
                        (errno == ENOENT) ||
                        (errno == ENOTDIR) ||
                        (errno == EACCES)
                    )
                ) {
                	dreport("Cached answer up to date: no file found");
                	return false;
                } else {
                	dreport("Cached answer outdated: file now extant");
                	return true;
                }
        }

        if( stat(ac->file, &st) ) {
        	dreport("cached answer is now outdated -- cannot stat file");
                return true;
        }
        if( (st.st_mtime > ac->modified) ) {
        	dreport("Cached answer is outdated: file modified");
                return true;
        }

        dreport("Cached answer up to date");
        return false;
} /* -- end is_outdated() -- */


/* -- end response.c -- */
