/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   anssettings.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "anssettings.h"

#include <xmalloc.h>
#include <xstring.h>
#include <defaults.h>
#include <logutils.h>
#include <state.h>
#include <strtotype.h>
#include <ansconf.h>
#include <fileutils.h>


/* This is used to create a new answer_config and overridden
by the user files */
AnswerConfig default_ans_conf;
/* The special case where a socket is owned by an unnamed uid */
AnswerConfig default_uuid_ans_conf;

/* These are in the server file and take two arguments:
answer_config* and a char*. */
struct key_func_pair server_answer_setting[] = {
        {  "unnamed-uid-answer-type",   &set_answer_type    },
        {  "unnamed-uid-userid",        &set_uuid_userid_type },
        {  "unnamed-uid-error",         &set_error_type     },
        {  "unnamed-uid-ostype",        &set_ostype         },
        {  "unnamed-uid-charset",       &set_charset        },
        {0,0}
};

/* These are both in the server file and the user files */
struct key_func_pair user_answer_setting[] = {
        {  "answer-type",               &set_answer_type    },
        {  "userid",                    &set_userid_type    },
        {  "charset",                   &set_charset        },
        {  "ostype",                    &set_ostype         },
        {  "error",                     &set_error_type     },
        {0,0}
};

/* The loop for setting an answer setting from a file_line struct.
Returns -1 on error (includes errors from the called function),
+1 on a successful call to a function, or 0 if the function was
not found */
int
answer_setting_loop(struct file_line* cfl, struct key_func_pair* kfp, AnswerConfig* ac) {
        uint found;
        uint filerr;
        for( ; kfp->keyword; ++kfp ){
                if( ! xstr_is_equal(cfl->key, kfp->keyword) )
                        continue;

                if( (kfp->func)(ac, cfl->value) )
                        return -1;
                else
                        return 1;
        }

        return 0;
} /* -- end answer_setting_loop() -- */

/* Sets the default answer settings */
int
set_default_answer_settings( void ) {
        clear_answer_config(&default_ans_conf);
	set_answer_type(&default_ans_conf, default_answer_type);
        set_userid_type(&default_ans_conf, default_userid);
        set_charset(&default_ans_conf, default_charset);
        set_ostype(&default_ans_conf, default_ostype);
        set_error_type(&default_ans_conf, default_error);
	memset(&(default_ans_conf.is_modified), false, sizeof(struct answer_config_modified));

        /* --------- unnamed uids --------- */
        clear_answer_config(&default_uuid_ans_conf);
        set_answer_type(&default_uuid_ans_conf, default_unnamed_uid_answer_type);
        set_uuid_userid_type(&default_uuid_ans_conf, default_unnamed_uid_userid);
        set_error_type(&default_uuid_ans_conf, default_unnamed_uid_error);
        set_charset(&default_uuid_ans_conf, default_unnamed_uid_charset);
        set_ostype(&default_uuid_ans_conf, default_unnamed_uid_ostype);
	memset(&(default_uuid_ans_conf.is_modified), false, sizeof(struct answer_config_modified));

	return 0;
} /* -- end set_default_answer_settings() -- */

/* Sets the answer type for an answer_config.  0 on success, -1 on error. */
int
set_answer_type(AnswerConfig* ac, char* answer_type) {
	c_answer_t at;
	if( strtoAnswerType(&at, answer_type) ) {
        	carp("Cannot set answer type");
                return -1;
        }

        if( (! state.initting) || (at != ac->answer_type) )
                ac->is_modified.answer_type = true;
        if( at != ac->answer_type )
        	ac->answer_type = at;

	return 0;
} /* -- end set_answer_type() -- */

/* Sets the userid.  0 on success, -1 on error */
int
set_userid_type(AnswerConfig* ac, char* userid) {
        c_userid_t ut;
        if( strtoUseridType( &ut, userid) ) {
        	carp("Could not set userid type");
                return -1;
        }

	/* This should be broken out */
	if( ut == UT_LITERAL ) {
		char* literal;
                userid = strchr(userid, '=');
                ++userid;
                if( *userid ) {
                        literal = xstrdup(userid);
                        dequote(&literal);
                } else
                	literal = NULL;

        	if( ! (
		    state.initting &&
		    xstr_is_equal(literal, ac->literal_userid)
		) ) {
                	ac->is_modified.userid_type = true;
                }

		if( ! xstr_is_equal(literal, ac->literal_userid) ) {
			xnullfree(ac->literal_userid);
			ac->literal_userid = literal;
                	ac->is_modified.literal_userid = true;
                }
        }

        if( (! state.initting) || (ut != ac->userid_type) )
                ac->is_modified.userid_type = true;
        if( ut != ac->userid_type ) {
		ac->userid_type = ut;
        	ac->is_modified.userid_type = true;
        }
        return 0;
} /* -- end set_userid_type() -- */

/* A wrapper for the above, rejecting "name" types */
int
set_uuid_userid_type(AnswerConfig* ac, char* userid) {
	if( set_userid_type(ac, userid) ) {
        	carp("Could not set userid type for unnamed uids");
                return -1;
        }
        if( ac->userid_type == UT_NAME ) {
        	carp("Cannot set unnamed userid type to \"name\"!!");
                return -1;
        }

        return 0;
}

/* Sets the charset.  NULL means not to specify a charset. */
int
set_charset(AnswerConfig* ac, char* charset) {
	char* cs;

        if( xstr_is_nil(charset) )
        	cs = NULL;
        else {
        	cs = xstrdup(charset);
        }

        if( ! state.initting )
        	ac->is_modified.userid_type = true;
        else if( ! xstr_is_equal(cs, ac->charset) ) {
		xnullfree(ac->charset);
		ac->charset = cs;
        	ac->is_modified.charset = true;
	}
        return 0;
} /* -- end set_charset() -- */

/* Sets the ostype to reply with.  NULL means not
to specify an OS (ie, report "OTHER"). */
int
set_ostype(AnswerConfig* ac, char* ostype) {
        char* ot;

        /* If the ostype is either NULL, NIL, or == "OTHER",
        clear ac->ostype */
        if( xstr_is_nil(ostype) || xstr_is_equal(ostype, "OTHER") )
        	ot = NULL;
        else {
                ot = xstrdup(ostype);
        }

        if( ! state.initting )
        	ac->is_modified.userid_type = true;
        else if( ! xstr_is_equal(ot, ac->ostype) ) {
		xnullfree(ac->ostype);
		ac->ostype = ot;
        	ac->is_modified.ostype = true;
        }
        return 0;
} /* -- end set_ostype() -- */

/* sets the error, rejects "actual" unless
state.initting. */
int
set_error_type(AnswerConfig* ac, char* error) {
	c_error_t sv = ac->error_type;
        if( strtoErrorType( &(ac->error_type), error ) ) {
        	carp("Cannot set error type");
                return -1;
        }

        if( (!state.initting) && (ac->error_type == ET_ACTUAL) ) {
        	carp("\"error: %s\" not allowed in userfiles", error);
        	return -1;
        }

        if( ac->error_type == ET_CUSTOM ) {
                char* tmp = ac->custom_error;
                error = strchr(error, '=');
                ++error;
                if( *error ) {
                        ac->custom_error = xstrdup(error);
                        dequote(&(ac->custom_error));
                } else
                	ac->custom_error = NULL;

        	if( ! state.initting ) {
                	ac->is_modified.error_type = true;
                } else if( ! xstr_is_equal(tmp, ac->custom_error) ) {
                	ac->is_modified.custom_error = true;
                }
                xnullfree(tmp);
        }

        if( ! state.initting ) {
                ac->is_modified.error_type = true;
        } else if( ac->error_type != sv ) {
                ac->is_modified.error_type = true;
        }

        return 0;
} /* -- end set_error_type() -- */


/* -- end anssettings.c -- */
