/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   ansconf.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>

#include "ansconf.h"

#include <xmalloc.h>
#include <xstring.h>
#include <anssettings.h>
#include <logutils.h>
#include <fileutils.h>
#include <sfopen.h>
#include <strtotype.h>
#include <state.h>
#include <serverconf.h>


/* Allocates an answer string based on the supplied answer_config.
Returns 0 on success, -1 on error.  Broken apart for simplicity. */
int
gen_answer_string(char** answer, AnswerConfig* ac) {
	if(ac->answer_type == AT_ERROR)
        	return gen_error_string(answer, ac, 0);
        else
		return gen_userid_string(answer, ac);
} /* -- end gen_answer_string() -- */

/* The error formatting protion of gen_answer_string(). */
int
gen_error_string(char** answer, AnswerConfig* ac, c_error_t err_type) {
        int retval = 0;
        char* tmp;

	if( ac->error_type )
		err_type = ac->error_type;
        switch( err_type ) {
        case ET_ACTUAL:
        	*answer = xstrdup("ERROR : <insert actual error type>");
                break;
        case ET_UNKNOWN:
                *answer = xstrdup("ERROR : UNKNOWN-ERROR");
                break;
        case ET_NOUSER:
                *answer = xstrdup("ERROR : NO-USER");
                break;
        case ET_HIDDEN:
                *answer = xstrdup("ERROR : HIDDEN-USER");
                break;
        case ET_INVAL:
                *answer = xstrdup("ERROR : INVALID-PORT");
                break;
        case ET_CUSTOM:
                if( ac->custom_error )
                	tmp = ac->custom_error;
                else
                	tmp = " ";

                if( xstr_is_nil(ac->username) )
                        *answer = xsprintf("ERROR : %s", tmp);
                else
                        *answer = xsprintf("ERROR : X%s", tmp);
                break;
        default:
                carp("Internal Logic Failure: Unknown ERROR_TYPE: uid == %d, type == %d",
                                ac->uid, ac->error_type);
                retval = -1;
                break;
        }

	return retval;
} /* -- end gen_error_string() -- */

/* The userid portion of gen_answer_string(). */
int
gen_userid_string(char** answer, AnswerConfig* ac) {
	int retval = 0;
        char* tmp;

        if( ! xstr_is_nil(ac->ostype) )
                tmp = xsprintf("USERID : %s", ac->ostype);
        else
                tmp = xsprintf("USERID : OTHER");
        *answer = tmp;

        if( ! xstr_is_nil(ac->charset) )
                tmp = xsprintf("%s , %s :", *answer, ac->charset);
        else
                tmp = xsprintf("%s :", *answer);
        xfree(*answer);

	char* lit_userid;
        switch( ac->userid_type ){
        case UT_UID:
        	if( ! state.initting )
                	*answer = xsprintf("%s%lu", tmp, ac->uid);
                else
                	*answer = xsprintf("%s<insert uid here>", tmp);
                break;
        case UT_NAME:
                if(! state.initting )
	                *answer = xsprintf("%s%s", tmp, ac->username);
                else
                	*answer = xsprintf("%s<insert name here>", tmp);
                break;
        case UT_LITERAL:
                if( ac->literal_userid )
	                lit_userid = ac->literal_userid;
                else
                	lit_userid = " ";
                *answer = xsprintf("%s%s", tmp, lit_userid);
                break;
        default:
                carp("Internal Logic Failure: Unknown USERID_TYPE: uid == %d, type == %d",
                		ac->uid, ac->userid_type );
            	retval = -1;
            	break;
        }
        xfree(tmp);

        return retval;
} /* -- end gen_userid_string() -- */

/* Frees memory associated with answer_config */
void
free_answer_config(AnswerConfig* ac) {
	if( ! ac )
        	return;
        xnullfree(ac->charset);
	xnullfree(ac->ostype);
	xnullfree(ac->custom_error);
        xnullfree(ac->literal_userid);
        xnullfree(ac->username);
        xnullfree(ac->file);
	xnullfree(ac->cached_answer);
        xfree(ac);
} /* -- end free_answer_config() -- */

/* Creates and returns a fresh answer_config* */
AnswerConfig*
new_answer_config( void ) {
	AnswerConfig* ac = xnew0(AnswerConfig, 1);
        clear_answer_config(ac);
        return ac;
}

/* NULLs the string pointers in an answer config and the
is_modified switches.  Translates ET_ACTUAL to ET_UNKNOWN. */
void
clear_answer_config(AnswerConfig* ac) {
        ac->username = NULL;
        ac->file = NULL;
        ac->charset = NULL;
        ac->ostype = NULL;
        ac->literal_userid = NULL;
        ac->custom_error = NULL;
	ac->cached_answer = NULL;
        memset(&(ac->is_modified), false, sizeof(ac->is_modified));
        if( ac->error_type == ET_ACTUAL)
                ac->error_type = ET_UNKNOWN;
} /* -- end clear_answer_config() -- */


/* performs a deep copy of an answer_config */
int
cp_answer_config(AnswerConfig* src, AnswerConfig* dst) {
        if( ! dst )
        	return -1;

        /* Copy the struct, but */
        *dst = *src;
        /* clear the pointers and .is_modified settings */
        clear_answer_config(dst);

        /* copy the strings */
        if( ! xstr_is_nil(src->ostype) ) {
                dst->ostype = xstrdup(src->ostype);
        }

        if( ! xstr_is_nil(src->charset) ) {
                dst->charset = xstrdup(src->charset);
        }

        if( ! xstr_is_nil(src->literal_userid) ) {
                dst->literal_userid = xstrdup(src->literal_userid);
        }

        if( ! xstr_is_nil(src->custom_error) ) {
                dst->custom_error = xstrdup(src->custom_error);
        }

        /* we do not duplicate the cached answer */
	return 0;
} /* -- end dup_answer_config() -- */

/* Reads a user's file and sets the passed AnswerConfig.
Returns 0 on success, -1 on error. */
int
parse_userfile(AnswerConfig* ac) {
	if( (!ac) || xstr_is_nil(ac->username) ) {
		carp("Invalid AnswerConfig!");
		if( ac )
			dreport("username blank");
		else
			dreport("ac NULL");
		return -1;
	}

	dreport("Parsing userfile for %s", ac->username);
	if( ! ac->file ) {
		ac->file = xsprintf("%s/%s", server.userdir, ac->username);
	}
        FILE* FP = sfopen_read(ac->file, ac->uid);
        if( ! FP ) {
		if( errno == ENOENT )
        		return -1;
		choke("Could not open \"%s\"s file", ac->username);
                ac->modified = 0;
                return -1;
        }

        struct stat st;
        if( fstat(fileno(FP), &st) ) {
        	choke("Could not stat \"%s\"s file", ac->username);
                ac->modified = 0;
                fclose(FP);
                return -1;
        }
        ac->modified = st.st_mtime;

        char* buf = fread_FILE(FP);
        if( ! buf ) {
        	carp("Failed to read \"%s\"s file", ac->username);
                ac->modified = 0;
                fclose(FP);
                return -1;
        }
        fclose(FP);

        struct file_line* fl = boil_contents(buf);
        xfree(buf);
        if( ! fl ) {
        	carp("Could not reduce contents of \"%s\"s file", ac->username);
                return -1;
        }

        struct file_line* fl_sv = fl;
        int retval;
        do {
        	retval = answer_setting_loop(fl++, user_answer_setting, ac);
        } while( (retval == 1) && fl->key );

        if( retval == 0 ) {
        	--fl;
        	carp("Unknown keyword {%s} in \"%s\"s file (line %d)",
        			fl->key, ac->username, fl->lineno);
                free_file_lines(&fl_sv);
                return -1;
        } else if( retval == -1 ) {
                free_file_lines(&fl_sv);
        	return -1;
        }
        free_file_lines(&fl_sv);

        /* Here we check for an implied answer type: if the
        userfile specifies nothing about an error type,
        then we set answer_type to userid -- and vice versa */
        if( ! (state.initting || ac->is_modified.answer_type) ) {
                if( ac->is_modified.error_type && (!ac->is_modified.userid_type) ) {
                	dreport("File implies error");
                        ac->answer_type = AT_ERROR;
                } else if( ac->is_modified.userid_type && (!ac->is_modified.error_type) ) {
                	dreport("File implies userid");
                        ac->answer_type = AT_USERID;
                }
	}

        return 0;
} /* -- end parse_userfile() -- */


/* -- end ansconf.c -- */
