/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   logfile.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>

#include "logfile.h"

#include <macros.h>
#include <procprivs.h>
#include <logutils.h>
#include <state.h>
#include <sfopen.h>

char*			server_logfile = NULL;
uint			server_logfile_set = FALSE;

char*			server_pidfile = NULL;
uint			server_pidfile_set = FALSE;
/* This is needed so that we can record the pid after the fork */
FILE*			server_pidFILE = NULL;

/* Starts logging -- everything going to stderr now goes to the logfile */
int
start_logging( void ) {
	if( ! server_logfile ) {
        	report("Logging to stderr");
        } else {
        	FILE* tmp = sfopen_append(server_logfile);
                if( ! tmp ) {
                	carp("Could not open server_logfile");
                        report("logging to stderr");
                        set_logfile(NULL);
                } else {
                	fclose(stderr);
                        stderr = tmp;
                }
        }
        report("cidentd initializing");
        return 0;
} /* -- end start_logging() -- */

/* Writes the pidfile.  Since this is a type of log,
it is defined here.  */
int
write_pidfile( void ) {
        if( ! server_pidFILE )
        	return 0;
        fprintf(server_pidFILE, "%lu\n", getpid());
        if( fclose(server_pidFILE) ) {
        	choke("Could not write pid to pidfile");
                server_pidfile = NULL;
                return 0;
        }
        return 0;

} /* -- end write_pidfile() -- */

/* Opens the pidfile.
Returns 0.
It is not an error if the pidfile cannot be written,
however, it is an error if the pidfile referes to an active
process. */
int
open_pidfile( void ) {
	/* Continue only the pidfile is to be written */
	if( !server_pidfile )
        	return 0;

        int retval;
        server_pidFILE = sfopen_write(server_pidfile);
        if( (! server_pidFILE) && (errno == EEXIST) ) {
        	if( check_pid() ) {
                	report("%s appears to be running.  "
                			"If this is not so, delete %s and try again",
                			PACKAGE, server_pidfile);
                	server_pidfile = NULL;
                        return -1;
                }
                proc_to_prime_privs();
                retval = unlink(server_pidfile);
                suspend_proc_privs();
                if( retval ) {
                	choke("Could not remove stale pidfile");
                        return 0;
                }
                server_pidFILE = sfopen_write(server_pidfile);
        }

        if( ! server_pidFILE ) {
        	report("Skipping the pidfile");
                server_pidfile = NULL;
                return 0;
        }

        uint pf_mode = S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH;
        proc_to_prime_privs();
        retval = fchmod(fileno(server_pidFILE), pf_mode);
	suspend_proc_privs();

        if( retval ) {
        	choke("could not chmod pidfile");
        }

        return 0;
} /* -- end open_pidfile() -- */

/* Checks the validity of a pidfile. Returns TRUE if a process
with that pid exists, FALSE otherwise.  On error, the
process is assumed to be valid. */
uint
check_pid( void ) {
        FILE* FP = sfopen_read(server_pidfile, 0);
        if( ! FP ) {
                choke("Could not read pidfile");
                return TRUE;
        }
        /* Twenty digits ought to be enough */
        char buf[21];
        memset(buf, 0, 21);
        fread(buf, 20, 1, FP);
        if( ferror(FP) ) {
                choke("Read of pidfile failed");
                fclose(FP);
                return TRUE;
        }
        fclose(FP);
        uint pid = strtoul(buf, NULL, 10);

        /* Check for a process */
        char* procfile = strprintf("/proc/%d", pid);
        if( ! procfile ) {
                choke("Could not make room for process dirname");
                return TRUE;
        }
        DIR* procdir = opendir(procfile);
        free(procfile);
        if( (! procdir) && (errno == ENOENT) ) {
        	closedir(procdir);
		return FALSE;
        }

        closedir(procdir);
        return TRUE;
} /* -- end check_pid -- */

/* Sets the logfile.  Returns 0 on success, -1 on error. */
int
set_logfile(char* file) {
	if( server_logfile_set && (!state.cmdline) )
        	return 0;

	char* sv;
        if( is_nil(server_logfile) )
        	sv = NULL;
        else
        	sv = server_logfile;

        if( is_nil(file) || strings_eq(file, "/dev/stderr") )
        	server_logfile = NULL;
        else {
        	server_logfile = strdup(file);
                if(!server_logfile) {
                	carp("Unable to store logfile name");
                        return -1;
                }
        }

        if( ! strings_eq(sv, server_logfile) )
        	server_logfile_set = TRUE;
	if_notnull_free(sv);
        return 0;
} /* -- end set_logfile() -- */

/* Prints the logfile.  Prefixes with '*' if needed. */
void
print_logfile( void ) {
        print_default_marker_if(!server_logfile_set);
        if( server_logfile )
	        println("logfile == %s", server_logfile);
        else
        	println("logging to stderr");
} /* -- end print_logfile() -- */

/* Sets the pidfile. Returns 0 on success, -1 on error. */
int
set_pidfile(char* file) {
	if( server_pidfile_set && (!state.cmdline) )
        	return 0;

        char* sv;
        if( is_nil(server_pidfile) )
        	sv = NULL;
        else
        	sv = server_pidfile;

        if( is_nil(file) || strings_eq(file, "/dev/null") )
        	server_pidfile = NULL;
        else {
        	server_pidfile = strdup(file);
                if( ! server_pidfile ) {
                	carp("Could not store pidfile name");
                        return -1;
                }
        }

	if( state.cmdline )
		server_pidfile_set = TRUE;
	else if( ! strings_eq(sv, server_pidfile) )
        	server_pidfile_set = TRUE;
        if_notnull_free(sv);
        return 0;
} /* -- end set_pidfile() -- */

/* Prints the pidfile.  Prefixes with '*' if needed. */
void
print_pidfile( void ) {
	print_default_marker_if(!server_pidfile_set);
        if( server_pidfile )
        	printf("pidfile == %s\n", server_pidfile);
        else
        	printf("(pidfile will not be written)\n");
} /* -- end print_pidfile() -- */


/* -- end logfile.c -- */
