/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   configfile.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "configfile.h"

#include <macros.h>
#include <state.h>
#include <logutils.h>
#include <fileutils.h>
#include <sfopen.h>
#include <anssettings.h>
#include <ansconf.h>
#include <serverconfig.h>
#include <procprivs.h>

char*	server_configfile = NULL;;
uint	server_configfile_set = FALSE;

/* Sets the config file */
int
set_configfile(char* file) {
	char* sv;
        if( is_nil(server_configfile) )
        	sv = NULL;
        else
        	sv = server_configfile;

	if( is_nil(file) )
        	server_configfile = NULL;
        else {
        	server_configfile = strdup(file);
                if( ! server_configfile ) {
                	choke("Could not store config file name");
                        return -1;
                }
        }

        if( ! strings_eq(sv, server_configfile) )
        	server_configfile_set = TRUE;
        if_notnull_free(sv);
        return 0;
} /* -- end set_configfile() -- */

/* prints the configfile */
void
print_configfile(void) {
	print_default_marker_if(!server_configfile_set);
        if( server_configfile )
        	println("config file == %s", server_configfile);
        else
        	println("No config file set.  Using compiled in defaults");
} /* -- end print_configfile() -- */

/* Processes the config file; 0 on success, -1 on error */
int
process_config_file( void ) {
	if( !(server_configfile && *server_configfile) )
        	return 0;

        /* open config file with necessary
        privs, but bypass sfcheck_read() since we
        will be using the stricter (and mutually
        incompatible) sfcheck_readonly() */
	FILE* FP = sfopen_read(server_configfile, 0 );
        if( ! FP ) {
        	choke("Could not read %s", server_configfile);
                return -1;
        }

        /* If proc_suid we need to check that the invoking user
        cannot write the config file, furthermore if suid_root
        we need to check that the runas_uid cannot write the
        config file.  The second check may be removed in the
        future. */
        if(
            ( proc_suid && sfcheck_readonly(FP, proc_uid.ruid) ) ||
            ( proc_suid_root && sfcheck_readonly(FP, proc_uid.uuid) )
        ) {
                carp("config file is not safe to use");
                fclose(FP);
                return -1;
        }

        char* buf = fread_FILE(FP);
        if( fclose(FP) ) {
        	choke("Closing file triggered error");
                return -1;
        }

        if( ! buf ) {
        	carp("Could not read config file into memory");
                return -1;
        }

        struct file_line* cfl = boil_contents(buf);
        free(buf);

        if( ! cfl ) {
        	carp("Could not reduce contents of config file");
                return -1;
        }

        struct file_line* cfl_sv = cfl;
        int retval;
        for( ; cfl->lineno; ++cfl ) {
                retval = deamon_setting_loop(cfl);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server_configfile, cfl->lineno, cfl->text);
                        return -1;
                }

                retval = answer_setting_loop(cfl, server_answer_setting, &default_uuid_ans_conf);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server_configfile, cfl->lineno, cfl->text);
                        return -1;
                }
                retval = answer_setting_loop(cfl, user_answer_setting, &default_ans_conf);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server_configfile, cfl->lineno, cfl->text);
                        return -1;
                }

                if( retval == 0 ) {
                	carp("Unknown keyword {%s} in %s (line %d) {%s}",
                			cfl->key, server_configfile, cfl->lineno, cfl->text);
                        return -1;
                }
	}

        free_file_lines(&cfl_sv);
        return 0;
} /* -- end process_config_file() -- */

/* The loop for processing a daemon setting.
Returns +1 on a successful call to a deamon
setting function, -1 on error (either in the
loop, or from the function called), or
-1 if nothing matched */
int
deamon_setting_loop(struct file_line* cfl) {
        uint found;
        uint filerr;
        struct key_func_pair* kfp;
        for( kfp = server_daemon_setting; kfp->keyword; ++kfp ){
                if( strcmp(cfl->key, kfp->keyword) != 0)
                        continue;

                if( (kfp->func)(cfl->value) )
                        return -1;
                else
                        return 1;
        }

        return 0;
} /* -- end deamon_setting_loop() -- */


/* -- end configfile.c -- */
