/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   init.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* This file will handle everything up to dropping root privs. */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>

#include "init.h"
#include "cmdline.h"
#include "defaults.h"

#include <serverconfig.h>
#include <strtotype.h>
#include <networking.h>
#include <logutils.h>
#include <state.h>
#include <procprivs.h>
#include <logfile.h>
#include <tbtopt.h>
#include <anssettings.h>

sigset_t sigs;


/* Begins the init process.  Calls all relevent
initialization routines.  Returns 0 on success,
-1 on error, or a +1 when the program must quit
but not due to an error (help requested). */
int
cidentd_init(int argc, char** argv) {
	store_proc_privs();
        suspend_proc_privs();
	state.initting = TRUE;

        if( init_regexps() )
        	return -1;

        if( set_defaults() )
        	return -1;

        int retval = process_cmdline(argc, argv);
        if( retval < 0 )
        	return -1;
        else if( retval > 0 )
        	return 1;

        if( process_config_file() )
        	return -1;

        /* if required display the current settings */
        if( state.show_settings ) {
        	show_settings();
	        return 1;
        }

        /* Set cleanup routine */
        atexit(&shutdown_cidentd);

        if( start_logging() )
        	return -1;

        if( init_signals() )
        	return -1;

        if( install_sig_handlers() )
        	return -1;

        if( open_pidfile() )
        	return -1;

        if( start_networking() )
        	return -1;

        if( drop_proc_privs() )
        	return -1;

        state.initting = FALSE;

	return 0;
} /* -- end cidentd_init() -- */

/* Calls every default setting function.
Returns o on success, -1 on error. */
int
set_defaults(void) {
	set_default_server_settings();
        set_default_answer_settings();
	return 0;
} /* -- end set_defaults() -- */

/* Sets the show_settings option */
int
set_show_settings( void ) {
	state.show_settings = TRUE;
        return 0;
} /* -- end set_show_settings() -- */

/* Calls every applicable show setting function */
void
show_settings( void ) {
	if( state.show_settings ) {
        	println("Command line and config file settings.");
        	print_default_marker_if(TRUE);
                println("marks a compiled in default setting.");
        }
        show_server_settings();
        show_answer_settings();
} /* -- end show_settings() -- */

/* Displays the default settings. */
int
display_defaults( void ) {
	println("Default compiled in settings:");
	set_defaults();
	state.show_settings = FALSE;
	show_settings();
	TBTOPT_EXIT_SUCCESS = TRUE;
	return 0;
} /* -- end display_defaults() -- */

/* Reports a shutdown to the logfile */
void
shutdown_cidentd( void ) {
        if( server_pidfile && unlink( server_pidfile ) )
                choke("Could not remove pidfile");

        if( close(server_auth_socket) )
        	choke("Could not close socket %d", server_auth_socket);

	report("shutting down\n-");
}

/* Initializes the handled signals */
int
init_signals( void ) {
        sigfillset(&sigs);

        /* Ignoring these signals can result in unpredictable
        behaviour */
        sigdelset(&sigs, SIGSEGV);
        sigdelset(&sigs, SIGFPE);
        sigdelset(&sigs, SIGILL);

        /* These cannot be caught, blocked, or ignored on Linux,
        so they are here merely for the sake of completeness  */
        sigdelset(&sigs, SIGSTOP);
        sigdelset(&sigs, SIGKILL);

        if( sigprocmask(SIG_BLOCK, &sigs, NULL) ) {
        	choke("Could not block exterraniuos signals");
                return -1;
        }

        return 0;
} /* -- end init_signals() -- */

/* Sets the signal handling routines */
int
install_sig_handlers( void ) {
        struct sigaction sact;
        sact.sa_mask = sigs;
        sact.sa_flags = 0;

        struct {
        	char* name;
                int   sig;
                void* handler;
        } list[] = {
        	{"SIGTERM", SIGTERM, &sa_cidentd_exit},
                {"SIGINT", SIGINT, &sa_cidentd_exit},
                {"SIGUSR2", SIGUSR2, &sa_toggle_debug},
                {0,0,0}
        };

        int k = 0;
        sigset_t noblock;
        sigemptyset(&noblock);

        while( list[k].name ) {
                sigaddset(&noblock, list[k].sig);
                sact.sa_handler = list[k].handler;
                if( sigaction(list[k].sig, &sact, NULL) ) {
                        choke("Could not install %s handler", list[k].name);
                        return -1;
                }
                ++k;
        }

        if( sigprocmask(SIG_UNBLOCK, &noblock, NULL) ) {
                choke("Could not unblock signals");
                return -1;
        }

        return 0;
} /* -- end install_sig_handlers() -- */

/* The TERM and INT sig handler */
void
sa_cidentd_exit(int sig) {
	report("SIG%s Caught", ((sig == SIGINT) ? "INT" : "TERM") );
        exit(EXIT_SUCCESS);
} /* -- end sa_cidentd_exit() -- */

/* The USR2 signal handler */
void
sa_toggle_debug(int sig) {
	state.debug ^= 1;
        report("Debugging turned %s", (state.debug ? "on" : "off") );
} /* -- end sa_toggle_debug() -- */


/* -- end init.c -- */
