/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   response.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "response.h"

#include <macros.h>
#include <logutils.h>
#include <ansconf.h>
#include <anssettings.h>
#include <fileutils.h>
#include <reutils.h>
#include <strtotype.h>
#include <reutils.h>
#include <serverconfig.h>

static AnswerConfig* ac_list = NULL;

/* Returns the default error string; used when an error occurs
before we can set up a user's AnswerConfig */
char*
get_default_error_string( c_error_t error_type ) {
	char* errorstr;
	if( default_ans_conf.error_type == ET_ACTUAL ) {
		default_ans_conf.error_type = error_type;
		gen_error_string(&errorstr, &default_ans_conf);
		default_ans_conf.error_type = ET_ACTUAL;
	} else
		gen_error_string(&errorstr, &default_ans_conf);

	if( ! errorstr ) {
		carp("Could not get error string");
		return NULL;
	}
	return errorstr;
} /* -- end get_default_error_string() -- */

/* Returns an answer given the remote host and the query.
Returns NULL on error.  The answer is mallocated and must be
free'd when no longer in use.  The passed uint*'s are set
to the decoded port pair values, if possible. */
char*
get_answer_string(char* query, ulong rhost, uint* rport, uint* lport) {
	char* answer;

	struct procnettcp_info tcpi;
	chomp(&query);
	if( is_nil(query) ) {
		dreport("Query was NIL");
		return NULL;
	}

	/* decode the query */
	if( extract_port_pair(&(tcpi.lport), &(tcpi.rport), query) )
		return NULL;
	tcpi.rhost = rhost;
	*rport = tcpi.rport;
	*lport = tcpi.lport;

	/* lookup sockowner */
	uid_t owner;
	c_error_t reterr = get_sock_owner(&tcpi, &owner);
	/* if no sockowner, return appropriate error string */
	if( reterr )
		return get_default_error_string(reterr);

	/* get username */
	char* username = NULL;
	if( get_usernam_r(&username, owner) )
		return get_default_error_string(ET_UNKNOWN);

	/* if no username, return uuid answer */
	if( ! username ) {
		if( gen_answer_string(&answer, &default_uuid_ans_conf) ) {
			carp("Failed to get uuid answer");
			return NULL;
		}
		return answer;
	}

	/* get cached AnswerConfig for user */
	AnswerConfig* ac = NULL;
	reterr = get_cached_answer(&ac, owner, username);
	free(username);
	if( reterr )
		return get_default_error_string(reterr);

	/* return strduped AnswerConfig string */
	return strdup(ac->cached_answer);
} /* -- end get_answer_string() -- */

/* Gets the socket owner given a struct procnettcp_info*.
Returns 0 on success, a c_error_t on error. */
c_error_t
get_sock_owner(struct procnettcp_info* tcpi, uid_t* owner) {
	char* line = NULL;
	size_t len = 0;
	FILE* FP = fopen("/proc/net/tcp", "r");
	if( ! FP ) {
		choke("Could not open /proc/net/tcp");
		return ET_UNKNOWN;
	}
	if( getline(&line, &len, FP) < 0 ) {
		choke("Could not read /proc/net/tcp");
		fclose(FP);
		if_notnull_free(line);
		return ET_UNKNOWN;
	}
	/* discard the header */
	if_notnull_free(line);

	struct procnettcp_info pntcpi;
	while(getline(&line, &len, FP) > 0) {
		chomp(&line);
		if( is_wspace_only(line) ) {
			if_notnull_free(line);
			continue;
		}
		if( get_sockinfo(&pntcpi, line) ) {
			carp("get_sockinfo() failed");
			if_notnull_free(line);
			fclose(FP);
			return ET_UNKNOWN;
		}
		if_notnull_free(line);
		if(
		    (pntcpi.lport == tcpi->lport) &&
		    (pntcpi.rport == tcpi->rport) &&
		    (pntcpi.rhost == tcpi->rhost)
		) {
			fclose(FP);
			*owner = pntcpi.owner;
			return 0;
		}
	}
	if_notnull_free(line);

	fclose(FP);
	return ET_NOUSER;
} /* -- end get_sock_owner() -- */

/*******
 * Answer caching utils
 *******/

/* Gets a cached answer for a user; creates one if necessary.
Returns 0 if no error, ET_UNKNOWN on error */
c_error_t
get_cached_answer(AnswerConfig** ac, uid_t owner, char* username) {
	/* Check for a cached answer */
	dreport("Checking cache for %d", owner);
	check_cache(ac, owner);
	if( *ac )
		return 0;

	/* create one if not extant */
	*ac = new_answer_config();
	if( ! *ac ) {
		carp("Could not make room for \"%s\" new AnswerConfig", username);
		return ET_UNKNOWN;
	}
	if( cp_answer_config(&default_ans_conf, *ac) ) {
		carp("Could not copy default_ans_conf for \"%s\"", username);
		free_answer_config(*ac);
		*ac = NULL;
		return ET_UNKNOWN;
	}

	(*ac)->username = strdup(username);
	if( ! (*ac)->username ) {
		choke("Could not store \"%s\"s username", username);
		free_answer_config(*ac);
		*ac = NULL;
		return ET_UNKNOWN;
	}

	(*ac)->uid = owner;
	if( parse_userfile(*ac) ) {
		free_answer_config(*ac);
		*ac = new_answer_config();
		if( ! *ac ) {
			carp("Could not create new AnswerConfig for \"%s\"", username);
			return ET_UNKNOWN;
		}
		if( cp_answer_config(&default_ans_conf, *ac) ) {
			carp("Could not copy default_ans_conf for \"%s\"", username);
			free_answer_config(*ac);
			*ac = NULL;
			return ET_UNKNOWN;
		}
		if( (*ac)->error_type == ET_ACTUAL )
			(*ac)->error_type = ET_UNKNOWN;
		(*ac)->answer_type = AT_ERROR;
	}
	if( gen_answer_string(&((*ac)->cached_answer), *ac) ) {
		carp("Could not store cached answer for \"%s\"", username);
		free_answer_config(*ac);
		*ac = NULL;
		return ET_UNKNOWN;
	}

	(*ac)->next = ac_list;
	ac_list = *ac;
	(*ac)->accessed = time(NULL);
	dreport("Added cached AnswerConfig %p to list", *ac);
	return 0;
} /* -- end get_cached_answer() -- */

/* This retrieves a cached answer for the passed uid.
If the cached answer exists, it is returned in the
passed ptr.  This also cleans out old cached answers. */
void
check_cache(AnswerConfig** ac, uid_t owner) {
	if( ! ac_list ) {
		*ac = NULL;
		return;
	}

	AnswerConfig* curr = ac_list;
	AnswerConfig* last = NULL;
	AnswerConfig* tmp;
	while( curr ) {
		if( curr->uid == owner ) {
			if( is_outdated(curr) ) {
				tmp = curr->next;
				if(last)
					last->next = curr->next;
				else
					ac_list = curr->next;
				free_answer_config(curr);
				curr = tmp;
				*ac = NULL;
				continue;
			}
			*ac = curr;
			(*ac)->accessed = time(NULL);
			dreport("Found cached AnswerConfig %p", *ac);
			last = curr;
			curr = curr->next;
			continue;
		}

		if( time(NULL) > (curr->accessed + server_cache_age) ) {
			dreport("deleting outdated cache entry for %d (%p) (last %p) (next %p)",
					curr->uid, curr, last, curr->next);
			tmp = curr->next;
			if(last)
				last->next = curr->next;
			else
				ac_list = curr->next;
			free_answer_config(curr);
			curr = tmp;
			continue;
		}
		last = curr;
		curr = curr->next;
	}
} /* -- end check_cache() -- */

/* Checks to see if a cached answer is outdated.
Returns TRUE if the cached answer is outdated, FALSE
otherwise.  */
uint
is_outdated(AnswerConfig* ac) {
	if( (! ac) || is_nil(ac->file) ) {
                carp("Invalid AnswerConfig reference!");
		exit(-1);
	}

        struct stat st;

        if( ! ac->modified ) {
        	dreport("File did not exist before");
        	if(
                    stat(ac->file, &st) &&
                    (
                        (errno == ENOENT) ||
                        (errno == ENOTDIR) ||
                        (errno == EACCES)
                    )
                ) {
                	dreport("and still does not exist");
                	return FALSE;
                } else {
                	dreport("but exists now");
                	return TRUE;
                }
        }

        dreport("File existed before");
        if( stat(ac->file, &st) ) {
        	dreport("cached answer is now outdated -- cannot stat file");
                return TRUE;
        }
        if( (st.st_mtime > ac->modified) ) {
        	dreport("and cached answer is now outdated");
                return TRUE;
        }

        dreport("cached answer is up to date");
        return FALSE;
} /* -- end is_outdated() -- */


/* -- end response.c -- */
