/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   response.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <signal.h>
#include <string.h>

#include "response.h"

#include <logutils.h>
#include <macros.h>
#include <anssettings.h>
#include <ansconf.h>
#include <anscache.h>
#include <reutils.h>
#include <strtotype.h>
#include <userdir.h>

#define DEBUGRESP

static pthread_key_t th_answer_config;
static pthread_key_t th_conn_context;

/* Initializes the response subsystem.  This includes
creating thread specific data spaces for thread global
variables. */
int
init_response( void ) {
        if( pthread_key_create(&th_answer_config, (void*) free_answer_config) ) {
        	choke("Could not initialize th_answer_config");
                return -1;
        }
        if( pthread_key_create(&th_conn_context, (void*) free_connection_context) ) {
        	choke("Could not initialize th_conn_context");
                return -1;
        }
        return 0;
} /* -- end init_response() -- */

/* Creates a new connection_context */
ConnContext*
new_connection_context( void ) {
	ConnContext* cc = malloc(sizeof(ConnContext));
        if( ! cc ) {
        	choke("Could not get new connection context");
                return NULL;
        }
        memset(cc, 0, sizeof(ConnContext));
        cc->query = NULL;
        return cc;
} /* -- end new_connection_context() -- */

/* Frees a connection_context */
void
free_connection_context(ConnContext* cc) {
	if(cc) {
        	dreport("Freeing ConnContext");
                if_notnull_free(cc->query);
                dreport("Closing socket");
                if( close(cc->sockfd) )
                	choke("Error on socket close");
                free(cc);
        }
} /* -- end free_connection_context() -- */


/* The initialization routine for a specific thread */
int
init_thread( ConnContext* cc ) {
        dreport("Initting new thread");
        pthread_detach(pthread_self());
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

        pthread_setspecific(th_conn_context, cc);

        AnswerConfig* ac = new_answer_config();
        if( ! ac ) {
        	carp("Could not create new AnswerConfig");
                return -1;
        }
        pthread_setspecific(th_answer_config, ac);

        return 0;
} /* -- end init_thread() -- */

/* This is the main entry point for responder threads */
int
r_main( ConnContext* cc ) {
        if(init_thread(cc)) {
        	carp("Could not initialize");
                send_error(ET_UNKNOWN);
                pthread_exit(NULL);
        }

        int retval;

        /* If we can't even get the query reliably,
        why bother sending a reply? */
	dreport("Getting query");
        if( get_query(cc) )
                pthread_exit(NULL);

        /* get_sock_owner */
	dreport("Getting socket owner");
        uint found = FALSE;
        uid_t owner;
        if( get_sock_owner(&found, &owner) ) {
        	send_error(ET_UNKNOWN);
                pthread_exit(NULL);
        }
        if( ! found ) {
        	send_error(ET_NOUSER);
                pthread_exit(NULL);
        }

        char* username = NULL;
        dreport("Getting username");
        if( get_usernam_r(&username, owner) ) {
                send_error(ET_UNKNOWN);
                pthread_exit(NULL);
        }

        /* Check the cache */
	dreport("Getting CachedAnswer");
        CachedAnswer* ca = NULL;
        get_cached_answer(&ca, owner);

        /* Generate a cached answer if neccessary */
        if( is_outdated(ca) ) {
        	if( ca ) {
                	dreport("Clearing outdated CachedAnswer");
                	free_cached_answer_data(ca);
                } else {
                	dreport("Creating new CachedAnswer");
                        ca = new_cached_answer();
                        if( ! ca ) {
                                choke("Could not create new CachedAnswer");
                                return -1;
                        }
                        ca->ca_uid = owner;
                        pthread_mutex_lock(&(ca->ca_lock));
                        dreport("Adding CachedAnswer (0x%.8x) to list", ca);
                        ca_list_add(ca);
                }

                dreport("Building CachedAnswer");
                if( build_cached_answer(&ca, owner, username) ) {
                        pthread_mutex_unlock(&(ca->ca_lock));
                        send_error(ET_UNKNOWN);
                        pthread_exit(NULL);
                }

                if( ! ca ) {
                	carp("Could not create CachedAnswer for %s", username);
                        send_error(ET_UNKNOWN);
                        pthread_exit(NULL);
                }
        }

        dreport("Sending output");
        send_output(ca->ca_string);

        pthread_mutex_unlock(&(ca->ca_lock));
        dreport("exiting");
        pthread_exit(NULL);
} /* -- end r_main() -- */

/* Retrives the query from the socket */
int
get_query(ConnContext* cc) {
	struct sockaddr_in rhost;
        /* Anything larger can be considered hostile intent */
        char buf[513];
        memset(buf, 0, 513);
        if( recv(cc->sockfd, buf, 513, 0) > 512 ) {
        	report("Hostile intent: query > 512 bytes");
                return -1;
        }
	cc->query = strdup(buf);
        chomp(&(cc->query));
        if( is_nil(cc->query) ) {
        	carp("Query was NIL");
                return -1;
        }
        if( decode_port_pair(cc) )
        	return -1;

        return 0;
} /* -- end get_query() -- */

/* Creates a CachedAnswer */
int
build_cached_answer(CachedAnswer** ca_p, uid_t uid, char* username) {
	AnswerConfig* ac = pthread_getspecific(th_answer_config);

        /* This needs to get broken up */
        if( ! username ) {
        	dreport("uid %d is unnamed");
        	/* Unnamed uids */
                dreport("Copying default_uuid_ans_conf");
                if( cp_answer_config(&default_uuid_ans_conf, ac) ) {
                	carp("Could not initialize baseline uuid AnswerConfig");
                        return -1;
                }
        } else {
        	dreport("uid %d is %s", uid, username);
        	/* Named uids */
                dreport("copying default_ans_conf");
                if( cp_answer_config(&default_ans_conf, ac) ) {
                        carp("Could not initialize baseline AnswerConfig");
                        return -1;
                }
                dreport("storing %s's file", username);
                char* file = strprintf("%s/%s", server_user_dir, username);
                if( ! file ) {
                        carp("Could not store \"%s\"s file", username);
                        return -1;
                }
                dreport("Transferring user spcific data to AnswerConfig");
                ac->ac_file = file;
                ac->ac_uid = uid;
                ac->ac_username = username;
                dreport("parsing userfile");
                if( parse_userfile(ac) ) {
                        /* Reset AnswerConfig since it may be corrupted
                        by a partial read of a user file */
                	dreport("saving user specific data from AnswerConfig");
                        ac->ac_file = NULL;
                        ac->ac_username = NULL;
                        time_t modified =  ac->ac_modified;
                        dreport("freeing AnswerConfig");
                        free_answer_config(ac);
                        dreport("Getting new AnswerConfig");
                        ac = new_answer_config();
                        if( ! ac ) {
                                carp("Could not recreate new AnswerConfig");
                                return -1;
                        }
                        dreport("Resetting thread specific AnswerConfig");
                        pthread_setspecific(th_answer_config, ac);

                        dreport("Copying default_ans_conf");
                        if( cp_answer_config(&default_ans_conf, ac) ) {
                                carp("Could not revert to default values");
                                return -1;
                        }

                        dreport("Resetting user specific data in AnswerConfig");
                        ac->ac_file = file;
                        ac->ac_username = username;
                        ac->ac_uid = uid;
                        ac->ac_modified = modified;
                }
        }

        dreport("setting CachedAnswer from AnswerConfig");
        if( set_cached_answer(*ca_p, ac) ) {
        	carp("Failed to set CachedAnswer");
                free_cached_answer(*ca_p);
                return -1;
        }

        return 0;
} /* end build_cached_answer() -- */

/* Decodes the port pair in the query.
Returns 0 on success, or -1 on error. */
int
decode_port_pair( ConnContext* cc ) {
        if( extract_port_pair(&(cc->localport), &(cc->remoteport), cc->query) ) {
        	report("Illegal port pair passed");
                return -1;
        }

        return 0;
} /* -- end decode_port_pair() -- */

/* Gets a socket owner by looking up the entry in
"/proc/net/tcp".  Returns 0 on success, -1 on error.
Sets the uint* to TRUE and the uid_t* to the owner
if found.  */
int
get_sock_owner(uint* found, uid_t* owner) {
	*found = FALSE;

        FILE* FP = fopen("/proc/net/tcp", "r");
        if( ! FP ) {
        	choke("Could not open \"/proc/net/tcp\"");
                return -1;
        }

        char* line = NULL;
        size_t len;
        struct procnettcp_info pntcpi;
        ConnContext* cc = pthread_getspecific(th_conn_context);

        /* Discard the header */
        if( getline(&line, &len, FP) < 0 ) {
        	choke("Could not read \"/proc/net/tcp\"");
                fclose(FP);
                return -1;
        }
        if_notnull_free(line);
        while( getline(&line, &len, FP) > 0 ) {
        	chomp(&line);
                if( is_wspace_only(line) ) {
                	if_notnull_free(line);
                        continue;
                }
                if( strip_whitespace(&line) ) {
                	carp("Failed to strip whitespace");
                        if_notnull_free(line);
                        return -1;
                }
                if( get_sockinfo(&pntcpi, line) < 0 )
                	return -1;
                if(
                    (cc->remotehost == pntcpi.rhost) &&
                    (cc->remoteport == pntcpi.rport) &&
                    (cc->localport  == pntcpi.lport)
                ) {
                	*found = TRUE;
                        break;
                }

                if_notnull_free(line);
                len = 0;
        }

        if_notnull_free(line);
        fclose(FP);

        if( *found )
                *owner = pntcpi.owner;

        return 0;
} /* -- end get_sock_owner() -- */

/* Checks to see if a cached answer is outdated.
Returns TRUE if the cached answer is outdated, FALSE
otherwise. If either the CachedAnswer or ca_file is
NULL then it is outdated by default.  */
uint
is_outdated(CachedAnswer* ca) {
	if( (! ca) || is_nil(ca->ca_file) )
                return TRUE;

        struct stat st;

        /* The ca_modified timestamp will be 0 if the
        file did not exist before */
        if( ! ca->ca_modified ) {
        	dreport("File did not exist before");
        	if(
                    stat(ca->ca_file, &st) &&
                    (
                        (errno == ENOENT) ||
                        (errno == ENOTDIR) ||
                        (errno == EACCES)
                    )
                ) {
                	dreport("and still does not exist");
                	return FALSE;
                } else {
                	dreport("but exists now");
                	return TRUE;
                }
        }

        dreport("File existed before");
        if( stat(ca->ca_file, &st) ) {
        	dreport("and is now outdated -- cannot stat file");
                return TRUE;
        }
        if( (st.st_mtime > ca->ca_modified) ) {
        	dreport("and is now outdated");
                return TRUE;
        }

        dreport("file is up to date");
        return FALSE;
} /* -- end is_outdated() -- */

/* Sends the error string out the socket.
Takes an ERROR_TYPE for an argument, but only
uses it if the AnswerConfig err_type ==
ET_ACTUAL. Also, we send the query right back to
the offender instead of sending (possibly
incorrectly) decoded port numbers.  */
void
send_error(uint err_type) {
        AnswerConfig* ac = pthread_getspecific(th_answer_config);
        ConnContext* cc = pthread_getspecific(th_conn_context);

        if( ! ac->ac_err_type )
        	ac->ac_err_type = err_type;
        ac->ac_ans_type = AT_ERROR;
        char* resp;
        if( gen_answer_string(&resp, ac) ) {
        	carp("Could not generate error string");
                if_notnull_free(resp);
                return;
        }
        send_output(resp);
        free(resp);
        return;
} /* -- end send_error() -- */

/* Sends the specified string out the socket, prepending
the port pairs from ConnContext */
int
send_output(char* str) {
	ConnContext* cc = pthread_getspecific(th_conn_context);
        str = strprintf("%d , %d : %s", cc->localport, cc->remoteport, str);
        if( ! str ) {
        	choke("Could not formulate response");
                return -1;
        }

        int retval = write(cc->sockfd, str, strlen(str));
        if( retval < 0 ) {
        	choke("Could not write to socket");
                return -1;
        }

        return 0;
} /* -- end send_output() -- */


/* -- end response.c -- */
