/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   fileutils.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "fileutils.h"

#include <macros.h>
#include <sfopen.h>
#include <logutils.h>
#include <reutils.h>



/* Counts the number of lines in a char*.
Returns the number of newlines found. */
uint
line_count(char* buf) {
	uint lines = 0;
	while( *buf ) {
		if( *buf++ == '\n' )
			++lines;
	}
        return lines;
} /* -- end line_count() -- */

/* Clears an array of file line constructs. */
void
clear_file_lines(struct file_line** flines, uint count) {
	struct file_line* fl = *flines;
        while(count--)
                clear_file_line(fl++);
} /* -- end clear_file_lines() -- */

/* Clears a file line construct */
void
clear_file_line(struct file_line* fl) {
        fl->lineno = 0;
        fl->text = NULL;
        fl->key = NULL;
        fl->value = NULL;
} /* -- end clear_file_line() -- */

/* Frees the memory associated with a file line */
void
free_file_line(struct file_line* fl) {
	if_notnull_free(fl->text);
        if_notnull_free(fl->key);
        if_notnull_free(fl->value);
} /* -- end free_file_line() -- */

/* Frees memory associated with a file line array,
including the array. */
void
free_file_lines(struct file_line** flines, uint count) {
	struct file_line* fl = *flines;
	while(count--)
        	free_file_line(fl++);
        free(*flines);
} /* -- end free_file_lines() -- */


/* Takes a raw file buffer and returns a ptr to an array
of file_line constructs.  leading and trailing whitespace
stripped (including teminating newlines), comment and
blank lines are removed. */
struct file_line*
boil_contents(char* buffer) {
        uint tot_lines = line_count(buffer);
        struct file_line* fline = malloc(++tot_lines * sizeof(struct file_line));
        if( ! fline ) {
        	choke("Could not make room for file line constructs");
                return NULL;
        }
        clear_file_lines(&fline, tot_lines);

        int curr_pos = 0;
	int lineno = 0;
	int k = 0;
        char* tmp;
	while(tmp = get_one_line(buffer, &curr_pos)) {
		fline[k].lineno = ++lineno;
                if_notnull_free(fline[k].text);

                chomp(&tmp);
                if( ! tmp )
                        continue;

                fline[k].text = strdup(tmp);

                if( strip_comment(&tmp) ) {
			carp("Failed at line %d", lineno);
			free_file_lines(&fline, k);
			if_notnull_free(tmp);
                        return NULL;
		}
		if( ! tmp )
			continue;

		if( strip_whitespace(&tmp) ){
			carp("Failed at line %d", lineno);
			free_file_lines(&fline, k);
			if_notnull_free(tmp);
                        return NULL;
		}
		if( ! tmp )
                	continue;

		if( split_key_value(&(fline[k].key), &(fline[k].value), tmp ) ) {
			carp("failed at line %d {%s}", lineno, fline[k].text);
			free_file_lines(&fline, k);
			if_notnull_free(tmp);
                        return NULL;
                }

                ++k;
                if_notnull_free(tmp);
        }

	clear_file_line( &(fline[k]) );
        return fline;
}

/* strndups a line starting at buf + *pos (includes newline);
updates *pos to point at the char after.  Returns char* on success,
NULL when no lines are left.  */
char*
get_one_line(char* buf, int* pos) {
	if( !( buf && buf[*pos] ) )
		return NULL;

	char* start = &buf[*pos];
	char* ret = strchr( start, '\n' );
	if( ! ret )
		return NULL;

        size_t len = ++ret - start;
	ret = strndup(start, len);
        if( ! ret ) {
        	choke("Could not extract line from string");
                return NULL;
        }
	*pos += len;
	return ret;
} /* -- end get_one_line() -- */

/* Loads a file into a mallocated buffer.
Newline and NULL terminates it.
Returns a char* on success, NULL on error.
Calls sfcheck_read(). */
char*
sfread_file(char* file, uid_t uid) {
        if( (! file ) || ( ! *file ) ) {
        	carp("Attempted to read a NULL file");
                return NULL;
        }

        FILE* FP = sfopen_read(file, uid);
        if( ! FP ) {
        	carp("Could not read %s", file);
                return NULL;
        }

	return fread_FILE(FP);
} /* -- end sfread_file() -- */

/* Reads a file stream into memory.
Ensures it is newline and nil terminated.
Returns a char* on success, NULL on error. */
char*
fread_FILE(FILE* FP) {
	if( ! FP ) {
        	carp("FILE stream is NULL");
                return NULL;
        }

        struct stat st;
        if( fstat(fileno(FP), &st) ) {
        	choke("Could not stat FILE");
                return NULL;
        }
        char* buf = calloc(1, st.st_size+2);
        if( ! buf ) {
        	choke("Could not make room for contents of file");
                return NULL;
        }

        if( fread( (void*)buf, 1, st.st_size, FP) < st.st_size ) {
        	if( feof(FP) )
                	carp("file truncated during read");
                else if( ferror(FP) )
                	choke("read failed for file");
                return NULL;
        }

        buf[st.st_size+1] = '\n';

        return buf;
} /* -- end fread_FILE() -- */


/* -- end fileutils.c -- */
