/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   anssettings.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include "anssettings.h"

#include <macros.h>
#include <defaults.h>
#include <logutils.h>
#include <state.h>
#include <strtotype.h>
#include <anscache.h>
#include <ansconf.h>
#include <fileutils.h>


/* This is used to create a new answer_config and overridden
by the user files */
AnswerConfig default_ans_conf;
/* The special case where a socket is owned by an unnamed uid */
AnswerConfig default_uuid_ans_conf;

/* These are in the server file and take two arguments:
answer_config* and a char* */
struct key_func_pair uuid_answer_setting[] = {
        {  "unnamed-uid-answer-type",   &set_ans_type       },
        {  "unnamed-uid-userid",        &set_uuid_userid_type },
        {  "unnamed-uid-error",         &set_error_type     },
        {  "unnamed-uid-ostype",        &set_ostype         },
        {  "unnamed-uid-charset",       &set_charset        },
        {0,0}
};

/* These are both in the server file and the user files */
struct key_func_pair user_answer_setting[] = {
        {  "answer-type",               &set_ans_type       },
        {  "userid",                    &set_userid_type    },
        {  "charset",                   &set_charset        },
        {  "ostype",                    &set_ostype         },
        {  "error",                     &set_error_type     },
        {0,0}
};


/* Sets the default answer settings */
int
set_default_answer_settings( void ) {
	/* This is so repetitave and badly coded it is likely
        to put one to sleep.  Do not read while operating heavy
        machinery */
        clear_answer_config(&default_ans_conf);

	set_ans_type(&default_ans_conf, default_answer_type);
	default_ans_conf.is_modified.ans_type = FALSE;

        set_userid_type(&default_ans_conf, default_userid);
	default_ans_conf.is_modified.uid_type = FALSE;
        default_ans_conf.is_modified.literal_userid = FALSE;

        set_charset(&default_ans_conf, default_charset);
	default_ans_conf.is_modified.charset = FALSE;

        set_ostype(&default_ans_conf, default_ostype);
        default_ans_conf.is_modified.ostype = FALSE;

        set_error_type(&default_ans_conf, default_error);
        default_ans_conf.is_modified.err_type = FALSE;
        default_ans_conf.is_modified.custom_error = FALSE;

        /* --------- unnamed uids --------- */
        clear_answer_config(&default_uuid_ans_conf);

        set_ans_type(&default_uuid_ans_conf, default_unnamed_uid_answer_type);
        default_uuid_ans_conf.is_modified.ans_type = FALSE;

        set_uuid_userid_type(&default_uuid_ans_conf, default_unnamed_uid_userid);
        default_uuid_ans_conf.is_modified.uid_type = FALSE;
        default_uuid_ans_conf.is_modified.literal_userid = FALSE;

        set_error_type(&default_uuid_ans_conf, default_unnamed_uid_error);
        default_uuid_ans_conf.is_modified.err_type = FALSE;
        default_uuid_ans_conf.is_modified.custom_error = FALSE;

        set_charset(&default_uuid_ans_conf, default_unnamed_uid_charset);
	default_uuid_ans_conf.is_modified.charset = FALSE;

        set_ostype(&default_uuid_ans_conf, default_unnamed_uid_ostype);
        default_uuid_ans_conf.is_modified.ostype = FALSE;

	return 0;
} /* -- end set_default_answer_settings() -- */

/* Sets the answer type for an answer_config.  0 on success, -1 on error. */
int
set_ans_type(AnswerConfig* ac, char* ans_type) {
	c_answer_t sv = ac->ac_ans_type;
	if( strtoAnswerType(&(ac->ac_ans_type), ans_type) ) {
        	carp("Cannot set answer type");
                return -1;
        }
        if( state.initting && (sv != ac->ac_ans_type) )
                ac->is_modified.ans_type = TRUE;
        else if( ! state.initting )
        	ac->is_modified.ans_type = TRUE;
	return 0;
} /* -- end set_ans_type() -- */

/* Sets the userid.  0 on success, -1 on error */
int
set_userid_type(AnswerConfig* ac, char* userid) {
        /* This is sloppy and needs to get refactored */
        c_userid_t sv = ac->ac_uid_type;

        if( strtoUseridType( &(ac->ac_uid_type), userid) ) {
        	carp("Could not set userid type");
                return -1;
        }

	if( ac->ac_uid_type == UT_LITERAL ) {
                char* tmp = ac->ac_literal_userid;
                userid = strchr(userid, '=');
                ++userid;
                if( *userid ) {
                        ac->ac_literal_userid = strdup(userid);
                        if( ! ac->ac_literal_userid ) {
                                carp("Could not strdup literal userid");
                                return -1;
                        }
                        dequote(&(ac->ac_literal_userid));
                } else
                	ac->ac_literal_userid = NULL;

        	if( state.initting && (! strings_eq(tmp, ac->ac_literal_userid)) )
                	ac->is_modified.literal_userid = TRUE;
                else if( ! state.initting )
                	ac->is_modified.uid_type = TRUE;
                if_notnull_free(tmp);
        }

        if( state.initting && (sv != ac->ac_uid_type) )
                ac->is_modified.uid_type = TRUE;
        else if( ! state.initting )
        	ac->is_modified.uid_type = TRUE;
        return 0;
} /* -- end set_userid_type() -- */

/* A wrapper for the above, rejecting "name" types */
int
set_uuid_userid_type(AnswerConfig* ac, char* userid) {
	if( set_userid_type(ac, userid) ) {
        	carp("Could not set userid type for unnamed uids");
                return -1;
        }
        if( ac->ac_uid_type == UT_NAME ) {
        	carp("Cannot set unnamed userid type to \"name\"!!");
                return -1;
        }

        return 0;
}

/* Sets the charset.  NULL means not to specify a charset. */
int
set_charset(AnswerConfig* ac, char* charset) {
	char* sv = ac->ac_charset;

        if( is_nil(charset) )
        	ac->ac_charset = NULL;
        else {
        	ac->ac_charset = strdup(charset);
                if( ! ac->ac_charset ) {
                	carp("Could not store charset");
                        return -1;
                }
        }

        if( state.initting && (!strings_eq(ac->ac_charset, sv)) )
        	ac->is_modified.charset = TRUE;
        else if( ! state.initting )
        	ac->is_modified.uid_type = TRUE;

        if_notnull_free(sv);
        return 0;
} /* -- end set_charset() -- */

/* Sets the ostype to reply with.  NULL means not
to specify an OS (ie, report "OTHER"). */
int
set_ostype(AnswerConfig* ac, char* ostype) {
        char* sv = ac->ac_ostype;

        /* If the ostype is either NULL, NIL, or == "OTHER",
        clear ac->ac_ostype */
        if( is_nil(ostype) || (strcmp(ostype, "OTHER") == 0) )
        	ac->ac_ostype = NULL;
        else {
                ac->ac_ostype = strdup(ostype);
                if( ! ac->ac_ostype ) {
                	carp("Could not store ostype");
                        return -1;
                }
        }

        /* Check to see if the ostype was modified */
        if( state.initting && (! strings_eq(ac->ac_ostype, sv)) )
        	ac->is_modified.ostype = TRUE;
        else if( ! state.initting )
        	ac->is_modified.uid_type = TRUE;
        if_notnull_free(sv);

        return 0;
} /* -- end set_ostype() -- */

/* sets the error. */
int
set_error_type(AnswerConfig* ac, char* error) {
	c_error_t sv = ac->ac_err_type;
        if( strtoErrorType( &(ac->ac_err_type), error ) ) {
        	carp("Cannot set error type");
                return -1;
        }

        if( ac->ac_err_type == ET_CUSTOM ) {
                char* tmp = ac->ac_custom_error;
                error = strchr(error, '=');
                ++error;
                if( *error ) {
                        ac->ac_custom_error = strdup(error);
                        if( ! ac->ac_custom_error ) {
                                carp("Could not strdup custom error");
                                return -1;
                        }
                        dequote(&(ac->ac_custom_error));
                } else
                	ac->ac_custom_error = NULL;

        	if( state.initting && (! strings_eq(tmp, ac->ac_custom_error)) )
                	ac->is_modified.custom_error = TRUE;
                else if( ! state.initting )
                	ac->is_modified.err_type = TRUE;
                if_notnull_free(tmp);
        }

        if( state.initting && (ac->ac_err_type != sv) )
                ac->is_modified.err_type = TRUE;
        else if( ! state.initting )
                ac->is_modified.err_type = TRUE;

        return 0;
} /* -- end set_error_type() -- */

/* The loop for setting an answer setting from a file_line struct.
Returns -1 on error (includes errors from the called function),
+1 on a successful call to a function, or 0 if the function was
not found */
int
answer_setting_loop(struct file_line* cfl, struct key_func_pair* kfp, AnswerConfig* ac) {
        uint found;
        uint filerr;
        for( ; kfp->keyword; ++kfp ){
                if( strcmp(cfl->key, kfp->keyword) != 0)
                        continue;

                if( (kfp->func)(ac, cfl->value) )
                        return -1;
                else
                        return 1;
        }

        return 0;
} /* -- end answer_setting_loop() -- */


/* -- end anssettings.c -- */
