/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   logutils.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <errno.h>
#include <stdarg.h>
#include <unistd.h>

#include "logutils.h"

#include <macros.h>
#include <state.h>

pthread_mutex_t		logfile_lock;
pthread_t		thread_main_id;

/* Initializes the log subsystem */
int
init_logutils( void ) {
	thread_main_id = pthread_self();
        pthread_mutex_init(&logfile_lock, NULL);
        return 0;
} /* -- end logutil_init() -- */

/* writes to stderr in a thread-safe way; */
void
c_logf(char* str) {
        /* lock the logfile */
        pthread_mutex_lock(&logfile_lock);

        /* log string */
        fprintf(stderr, "%s\n", str);
        fflush(stderr);
        fsync(fileno(stderr));

        /* unlock log file */
        pthread_mutex_unlock(&logfile_lock);

        return;
} /* -- end c_logf() -- */

/* Logs an error to stderr in a thread-aware, errno-specific way */
void
freporterr(char* file, uint line, char* func, pthread_t th_id, int errnum, char* fmt, ...) {
        char* str, * tmp;						// {
	int retval;
        va_list ap;
        va_start(ap, fmt);
        retval = vasprintf( &str, fmt, ap );
        va_end(ap);
                /* If that worked, return str */
        if( retval < 0 ) {
                c_logf("freporterr failed");
                return;
        }

        /* Prepend function name */
        tmp = strprintf("%s()  %s", func, str);
        if( tmp ) {
        	free(str);
                str = tmp;
        }

        /* If debugging requested, prepend file, line, and function */
        if( state.debug ) {
                tmp = strprintf("%s (line %d):: %s", file, line, str);
                if(tmp) {
                        free(str);
                        str = tmp;
                }
        }

        /* Prepend thread */
        if(th_id != thread_main_id) {
        	tmp = strprintf("th_%.8x %s", th_id, str);
                if(tmp) {
                	free(str);
                        str = tmp;
                }
        }

        /* prepend time */
        char* timestr = get_timestr();
        if( timestr ) {
        	tmp = strprintf("%s %s", timestr, str);
                if(tmp) {
                	free(str);
                        str = tmp;
                }
                free(timestr);
        }

        /* Append error string */
        if( errnum ) {
        	char* err_str = get_strerror_r(errnum);
                if( err_str ) {
        		tmp = strprintf("%s -- %s", str, err_str);
                        free(err_str);
                        if( tmp ) {
                                free(str);
                                str = tmp;
                        }
                }
        }

        /* Log it */
        if(str) {
            c_logf(str);
            free(str);
        }

        /* Reset errno -- not, for carp this clears errno */
        errno = errnum;
} /* -- end freporterr() -- */

/* Logs status to stderr in a thread-aware way */
void
freportstat(pthread_t th_id, char* fmt, ...) {
        char* str, * tmp;						// {
        int retval;
        va_list ap;
        va_start(ap, fmt);
        retval = vasprintf( &str, fmt, ap );
        va_end(ap);
                /* If that worked, return str */
        if( retval < 0 ) {
        	c_logf("freportstat failed");
                return;
        }

        if(th_id != thread_main_id) {
        	tmp = strprintf("th_%.8x %s", th_id, str);
                if(tmp) {
                	free(str);
                        str = tmp;
                }
        }

        char* timestr = get_timestr();
        if(timestr) {
        	tmp = strprintf("%s %s", timestr, str);
                if(tmp) {
                	free(str);
                        str = tmp;
                }
                free(timestr);
        }

        if(str) {
                c_logf(str);
                free(str);
        }

} /* -- end freportstat() -- */

/* Allocates and returns a formatted string
Returns char* on success, NULL on error.
This function will be removed in the future and
replaced with the GNU asprintf(). */
char*
strprintf( char* fmt, ... ) {
	char* str;							// {
        int retval;
        va_list ap;
        va_start(ap, fmt);
        retval = vasprintf( &str, fmt, ap );
        va_end(ap);
                /* If that worked, return str */
        if( retval < 0 ) {
                c_logf("strprintf failed");
                return NULL;
        }

	return str;
} /* -- end strprintf() -- */

/* allocates and returns a time string for logfiles */
char*
get_timestr( void ) {
        struct tm currtm;
        time_t currtime;
        time(&currtime);
        localtime_r(&currtime, &currtm);
        return strprintf("%.2d/%.2d/%.4d %.2d:%.2d:%.2d",
        		currtm.tm_mon+1, currtm.tm_mday, currtm.tm_year+1900,
                	currtm.tm_hour, currtm.tm_min, currtm.tm_sec);
} /* -- end get_timestr() -- */

/* Allocates and returns the string for an errno.  Masks the
incompatibility between glibc and other (standard) libraries. */
char*
get_strerror_r(int errnum) {
	char* errstr = NULL;
#ifdef STRERROR_R_CHAR_P
	errstr = strdup(strerror_r(errno, NULL, 49));
#else /* strerror_r() returns an int */
	errstr = calloc(50, sizeof(char) );
        if( ! errstr )
        	return NULL;
	if( strerror_r(errno, errstr, 49) )
        	null_free(errstr);
#endif
	return errstr;
} /* -- end get_strerror_r() -- */


/* -- end logutils.c -- */
