/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   anscache.h, part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef anscache_H
#define anscache_H 1

#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>

#include <ansconf.h>

/* This is kept in memory to prevent multiple reads of the
same file.  Since the file must be stat()'d before a read,
the only penalty is the lock needed to check the file's
modified time against ca_modified, and then strdup()'ing
ca_string before releasing the lock.  There will be two
of these globally accesible: one for the defaults, another
for the special case where a uid does not map to a username. */
typedef struct cached_answer {
        struct cached_answer*	ca_next;
        uid_t			ca_uid;

        /* Applies only to those members below */
        pthread_mutex_t		ca_lock;
        /* When the cached answer was last accessed */
        time_t			ca_accessed;
        /* The full path to the file used */
        char*			ca_file;
        /* Check against the file's modification time */
        time_t			ca_modified;
        /* The cached answer, verbatim */
        char*			ca_string;
} CachedAnswer;

struct cached_answer_list {
	uint			readers;
        uint			write_pending;
        pthread_mutex_t		lock;
        pthread_cond_t		is_unlocked;
        CachedAnswer*		ca_head;
};

/* Initialized the two default cached answers and the list. */
int
init_cached_answers( void );

/* Frees memory associated with a cached_answer */
void
free_cached_answer(CachedAnswer* ca);

/* Frees the data associated with a cached answer,
but does not free the struct itself, not destroy its
mutex */
void
free_cached_answer_data(CachedAnswer* ca);

/* Creates a new blank cached answer.  NULL on error. */
CachedAnswer*
new_cached_answer( void );

/* Clears all the entries in a cached_answer. */
void
clear_cached_answer(CachedAnswer* ca);

/* Dumps the values associated with a cached answer */
void
print_cached_answer(CachedAnswer* ca);

/* Sets a CachedAnswer based on an AnswerConfig */
int
set_cached_answer(CachedAnswer* ca, AnswerConfig* ac);

/* Locks the ca_list for reading.  Multiple readers are allowed,
provided there is no write pending */
void
ca_list_lock_read( void );

/* Unlocks a read lock */
void
ca_list_unlock_read( void );

/* Locks the ca_list for writing.  Superceeds all waiting reads. */
void
ca_list_lock_write( void );

/* Unlocks a write-lock */
void
ca_list_unlock_write( void );

/* Adds a cached answer to the list (at the beginning).
The cached answer must not be locked.  */
void
ca_list_add(CachedAnswer* ca);

/* Retrieves a CachedAnswer by uid, LOCKS it.
If there was no (usable) match, the ptr is null. */
void
get_cached_answer(CachedAnswer** ca, uid_t uid);

/* Performs the work of get_cached_answer, but DOES
NOT LOCK the list */
void
find_cached_answer(CachedAnswer** ca, uid_t uid);


#endif /* anscache_H */
