/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   ansconf.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>

#include "ansconf.h"

#include <anssettings.h>
#include <userdir.h>
#include <macros.h>
#include <logutils.h>
#include <fileutils.h>
#include <strtotype.h>
#include <state.h>

pthread_mutex_t pw_lock = PTHREAD_MUTEX_INITIALIZER;


/* Allocates an answer string based on the supplied answer_config.
Returns 0 on success, -1 on error.  Broken apart for simplicity. */
int
gen_answer_string(char** answer, AnswerConfig* ac) {
	if(ac->ac_ans_type == AT_ERROR)
        	return gen_error_string(answer, ac);
        else
		return gen_userid_string(answer, ac);
} /* -- end gen_answer_string() -- */

/* The error formatting protion of gen_answer_string(). */
int
gen_error_string(char** answer, AnswerConfig* ac) {
        int retval = 0;
        char* tmp;

        switch( ac->ac_err_type ) {
        case ET_UNKNOWN:
                *answer = strdup("ERROR : UNKNOWN-ERROR\r\n");
                if ( ! *answer ) {
                	carp("strdup() of ET_UNKNOWN string failed");
                        return -1;
                }
                break;
        case ET_NOUSER:
                *answer = strdup("ERROR : NO-USER\r\n");
                if ( ! *answer ) {
                	carp("strdup() of ET_NOUSER string failed");
                        return -1;
                }
                break;
        case ET_HIDDEN:
                *answer = strdup("ERROR : HIDDEN-USER\r\n");
                if ( ! *answer ) {
                	carp("strdup() of ET_HIDDEN string failed");
                        return -1;
                }
                break;
        case ET_INVAL:
                *answer = strdup("ERROR : INVALID-PORT\r\n");
                if ( ! *answer ) {
                	carp("strdup() of ET_INVAL string failed");
                        return -1;
                }
                break;
        case ET_CUSTOM:
                if( ac->ac_custom_error )
                	tmp = ac->ac_custom_error;
                else
                	tmp = " ";

                /* This is how the behavior is different between
                the user files and the main file */
                if(state.initting)
                        *answer = strprintf("ERROR : %s\r\n", tmp);
                else
                        *answer = strprintf("ERROR : X%s\r\n", tmp);

                if ( ! *answer ) {
                        carp("strprintf() of ET_CUSTOM string failed");
                        return -1;
                }
                break;
        default:
                carp("Internal Logic Failure: Unknown ERROR_TYPE: uid == %d, type == %d",
                                ac->ac_uid, ac->ac_err_type);
                retval = -1;
                break;
        }

	return retval;
} /* -- end gen_error_string() -- */

/* The userid portion of gen_answer_string(). */
int
gen_userid_string(char** answer, AnswerConfig* ac) {
	int retval = 0;
        char* tmp;

        if( ! is_nil(ac->ac_ostype) )
                tmp = strprintf("USER-ID : %s", ac->ac_ostype);
        else
                tmp = strprintf("USER-ID : OTHER");
        if ( ! tmp ) {
        	carp("strprintf() of OS-TYPE failed");
                return -1;
        }
        *answer = tmp;

        if( ! is_nil(ac->ac_charset) )
                tmp = strprintf("%s , %s :", *answer, ac->ac_charset);
        else
                tmp = strprintf("%s :", *answer);
        free(*answer);
        if ( ! tmp ) {
        	carp("strprintf() of CHARSET failed");
                *answer = NULL;
                return -1;
        }

        char* lit_userid;
        switch( ac->ac_uid_type ){
        case UT_UID:
        	if( ! state.initting )
                	*answer = strprintf("%s%lu\r\n", tmp, ac->ac_uid);
                else
                	*answer = strprintf("%s<insert uid here>\r\n", tmp);
                if ( ! *answer ) {
                	carp("strprintf() of ac->ac_uid failed");
                        retval = -1;
                }
                break;
        case UT_NAME:
                if(! state.initting )
	                *answer = strprintf("%s%s\r\n", tmp, ac->ac_username);
                else
                	*answer = strprintf("%s<insert name here>\r\n", tmp);

                if ( ! *answer ) {
                	carp("strprintf() of ac->ac_username failed");
                        retval = -1;
                }
                break;
        case UT_LITERAL:
                if( ac->ac_literal_userid )
	                lit_userid = ac->ac_literal_userid;
                else
                	lit_userid = " ";

                *answer = strprintf("%s%s\r\n", tmp, lit_userid);
                if ( ! *answer ) {
                	carp("strprintf() of ac->ac_literal_userid failed");
                        retval = -1;
                }
                break;
        default:
                carp("Internal Logic Failure: Unknown USERID_TYPE: uid == %d, type == %d",
                		ac->ac_uid, ac->ac_uid_type );
            	retval = -1;
            	break;
        }
        free(tmp);

        return retval;
} /* -- end gen_userid_string() -- */

/* Frees memory associated with answer_config */
void
free_answer_config(AnswerConfig* ac) {
	if( ! ac )
        	return;
        if_notnull_free(ac->ac_charset);
	if_notnull_free(ac->ac_ostype);
	if_notnull_free(ac->ac_custom_error);
        if_notnull_free(ac->ac_literal_userid);
        if_notnull_free(ac->ac_username);
        free(ac);
} /* -- end free_answer_config() -- */

/* Creates and returns a fresh answer_config* */
AnswerConfig*
new_answer_config( void ) {
	AnswerConfig* ac = malloc(sizeof(AnswerConfig));
        if( ! ac ) {
        	carp("Could not malloc new answer_config");
                return NULL;
        }
        memset(ac, 0, sizeof(AnswerConfig));

        clear_answer_config(ac);
        return ac;
}

/* NULLs the string pointers in an answer config and the
is_modified switches. */
void
clear_answer_config(AnswerConfig* ac) {
        ac->ac_username = NULL;
        ac->ac_file = NULL;
        ac->ac_charset = NULL;
        ac->ac_ostype = NULL;
        ac->ac_literal_userid = NULL;
        ac->ac_custom_error = NULL;
        memset(&(ac->is_modified), FALSE, sizeof(ac->is_modified));
} /* -- end clear_answer_config() -- */


/* performs a deep copy of an answer_config */
int
cp_answer_config(AnswerConfig* src, AnswerConfig* dst) {
        if( ! dst )
        	return -1;

        /* Copy the struct, but */
        *dst = *src;
        /* clear the pointers and .is_modified settings */
        clear_answer_config(dst);

        /* copy the default strings */
        if( ! is_nil(src->ac_ostype) ) {
                dst->ac_ostype = strdup(src->ac_ostype);
                if( ! dst->ac_ostype ) {
                        carp("could not strdup ac_ostype");
                        free_answer_config(dst);
                        return -1;
                }
        }

        if( ! is_nil(src->ac_charset) ) {
                dst->ac_charset = strdup(src->ac_charset);
                if( ! dst->ac_charset ) {
                        carp("could not strdup ac_charset");
                        free_answer_config(dst);
                        return -1;
                }
        }

        if( ! is_nil(src->ac_literal_userid) ) {
                dst->ac_literal_userid = strdup(src->ac_literal_userid);
                if( ! dst->ac_literal_userid ) {
                        carp("could not strdup ac_literal_userid");
                        free_answer_config(dst);
                        return -1;
                }
        }

        if( ! is_nil(src->ac_custom_error) ) {
                dst->ac_custom_error = strdup(src->ac_custom_error);
                if( ! dst->ac_custom_error ) {
                        carp("could not strdup ac_custom_error");
                        free_answer_config(dst);
                        return -1;
                }
        }

        return 0;
} /* -- end dup_answer_config() -- */

/* Reads a user's file and sets the passed AnswerConfig.
Returns 0 on success, -1 on error. */
int
parse_userfile(AnswerConfig* ac) {
        FILE* FP = sfopen_read(ac->ac_file, ac->ac_uid);
        if( ! FP ) {
        	choke("Could not open user file");
                ac->ac_modified = 0;
                return -1;
        }

        struct stat st;
        if( fstat(fileno(FP), &st) ) {
        	choke("Could not stat user file");
                fclose(FP);
                return -1;
        }
        ac->ac_modified = st.st_mtime;

        char* buf = fread_FILE(FP);
        if( ! buf ) {
        	carp("Failed to read user file");
                fclose(FP);
                return -1;
        }
        fclose(FP);

        uint tot_lines = line_count(buf);
        struct file_line* fl = boil_contents(buf);
        free(buf);
        if( ! fl ) {
        	carp("Could not reduce contents of user file");
                return -1;
        }

        struct file_line* fl_sv = fl;
        int retval;
        do {
        	retval = answer_setting_loop(fl++, user_answer_setting, ac);
        } while( (retval == 1) && fl->key );

        if( retval == 0 ) {
        	carp("Unknown keyword in user file {%s} (line %d)",
        			(--fl)->key, fl->lineno);
                free_file_lines(&fl_sv, tot_lines);
                return -1;
        } else if( retval == -1 ) {
                free_file_lines(&fl_sv, tot_lines);
        	return -1;
        }
        free_file_lines(&fl_sv, tot_lines);

        /* Here we check for an implied answer type: if the
        userfile specifies nothing about an error type,
        then we set answer_type to userid -- and vice versa */
        if( ! ac->is_modified.ans_type ) {
        	if( ac->is_modified.err_type && (!ac->is_modified.ans_type) )
                	ac->ac_ans_type = AT_ERROR;
                else if( ac->is_modified.ans_type && (!ac->is_modified.err_type) )
                	ac->ac_ans_type = AT_USERID;
        }

        return 0;
} /* -- end parse_userfile() -- */


/* -- end ansconf.c -- */
