/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   anscache.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include "anscache.h"

#include <macros.h>
#include <logutils.h>
#include <fileutils.h>
#include <anssettings.h>
#include <userdir.h>

/* Holds the default cached answer */
CachedAnswer default_cached_ans;

/* Information about the list of cached answers. */
struct cached_answer_list ca_list;

/* Initialized the list of cached answers. */
int
init_cached_answers( void ) {
        ca_list.ca_head = NULL;
        ca_list.readers = 0;
        ca_list.write_pending = FALSE;
        pthread_mutex_init(&(ca_list.lock), NULL);
        pthread_cond_init(&(ca_list.is_unlocked), NULL);
	return 0;
} /* -- end init_cached_answers() -- */

/* Frees memory associated with a cached_answer. Must
NOT be locked before hand. */
void
free_cached_answer(CachedAnswer* ca) {
        free_cached_answer_data(ca);
        pthread_mutex_destroy(&(ca->ca_lock));
        free(ca);
} /* -- end free_cached_answer() -- */

/* Frees the data associated with a cached answer,
but does not free the struct itself, not destroy its
mutex */
void
free_cached_answer_data(CachedAnswer* ca) {
	if(!ca)
        	return;
	if_notnull_free(ca->ca_string);
        if_notnull_free(ca->ca_file);
}
/* Creates a new blank cached answer.  NULL on error. */
CachedAnswer*
new_cached_answer( void ) {
	CachedAnswer* ca = malloc(sizeof(CachedAnswer));
        if( ! ca ) {
        	choke("Could not malloc a cached_answer");
                return NULL;
        }
        clear_cached_answer(ca);
        pthread_mutex_init(&(ca->ca_lock), NULL);
        return ca;
} /* -- end new_cached_answer() -- */

/* Clears all the entries in a cached_answer. */
void
clear_cached_answer(CachedAnswer* ca) {
	time(&(ca->ca_accessed));
        ca->ca_modified = 0;
        ca->ca_uid = 0;
        ca->ca_next = NULL;
        ca->ca_string = NULL;
        ca->ca_file = NULL;
} /* -- end clear_cached_answer() -- */

/* Dumps the values associated with a cached answer */
void
print_cached_answer(CachedAnswer* ca) {
	report("Next: 0x%.8x", ca->ca_next);
	report("Uid: %d", ca->ca_uid);
	report("Accessed: %lu", ca->ca_accessed);
	report("File name: %s", ca->ca_file);
	report("Modified: %lu", ca->ca_modified);
	report("Answer: %s", ca->ca_string);
} /* -- end print_cached_answer() -- */

/* Sets a CachedAnswer based on an AnswerConfig */
int
set_cached_answer(CachedAnswer* ca, AnswerConfig* ac) {
        if_notnull_free( ca->ca_string );
        if( gen_answer_string(&(ca->ca_string), ac) ) {
        	carp("Failed to set cached answer string");
                return -1;
        }
        ca->ca_file = ac->ac_file;
        ac->ac_file = NULL;
        ca->ca_uid = ac->ac_uid;
        ca->ca_modified = ac->ac_modified;
        time(&(ca->ca_accessed));
        return 0;
} /* -- end set_cached_answer() -- */

/* Locks the ca_list for reading.  Multiple readers are allowed,
provided there is no write pending */
void
ca_list_lock_read( void ) {
	pthread_mutex_lock(&(ca_list.lock));
        while(ca_list.write_pending)
        	pthread_cond_wait(&(ca_list.is_unlocked), &(ca_list.lock));
        ca_list.readers++;
        pthread_mutex_unlock(&(ca_list.lock));
} /* -- end ca_list_lock_read() -- */

/* Unlocks a read lock */
void
ca_list_unlock_read( void ) {
	pthread_mutex_lock(&(ca_list.lock));
        if( ! ca_list.readers ) {
        	pthread_mutex_unlock(&(ca_list.lock));
        	return;
        }
        ca_list.readers--;
        if( ! ca_list.readers )
        	pthread_cond_signal(&(ca_list.is_unlocked));
        pthread_mutex_unlock(&(ca_list.lock));
} /* -- end ca_list_unlock_read() -- */

/* Locks the ca_list for writing.  Superceeds all waiting reads. */
void
ca_list_lock_write( void ) {
	pthread_mutex_lock(&(ca_list.lock));
        while( ca_list.write_pending || ca_list.readers )
        	pthread_cond_wait(&(ca_list.is_unlocked), &(ca_list.lock));
        ca_list.write_pending = TRUE;
        pthread_mutex_unlock(&(ca_list.lock));
} /* -- end ca_list_lock_write() -- */

/* Unlocks a write-lock */
void
ca_list_unlock_write( void ) {
	pthread_mutex_lock(&(ca_list.lock));
        if( ! ca_list.write_pending ) {
        	pthread_mutex_unlock(&(ca_list.lock));
                return;
        }
        ca_list.write_pending = FALSE;
        pthread_cond_broadcast(&(ca_list.is_unlocked));
        pthread_mutex_unlock(&(ca_list.lock));
} /* -- end ca_list_unlock_write() -- */

/* Adds a cached answer to the list (at the beginning). */
void
ca_list_add(CachedAnswer* ca) {
	ca_list_lock_write();
        CachedAnswer* tmp;
        find_cached_answer(&tmp, ca->ca_uid);
        if( ! tmp ) {
                ca->ca_next = ca_list.ca_head;
                ca_list.ca_head = ca;
        }
        ca_list_unlock_write();
} /* -- end ca_list_add() -- */

/* Retrieves a CachedAnswer by uid, LOCKS it.
If there was no match, the ptr is null. */
void
get_cached_answer(CachedAnswer** ca, uid_t uid) {
        ca_list_lock_read();
        find_cached_answer(ca, uid);
        if(*ca) {
        	pthread_mutex_lock(&((*ca)->ca_lock));
        }
        ca_list_unlock_read();
} /* -- end get_cached_answer() -- */

/* Performs the work of get_cached_answer, but DOES
NOT LOCK the list */
void
find_cached_answer(CachedAnswer** ca, uid_t uid) {
        *ca = ca_list.ca_head;
        while( *ca && (uid !=(*ca)->ca_uid)) {
        	*ca = (*ca)->ca_next;
        }
} /* -- end find_cached_answer() -- */


/* -- end anscache.c -- */
