/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   serversettings.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>

#include "serversettings.h"

#include <serverconf.h>
#include <defaults.h>
#include <logutils.h>
#include <state.h>
#include <strtotype.h>
#include <networking.h>
#include <procprivs.h>
#include <fileutils.h>
#include <macros.h>
#include <sfopen.h>
#include <ansconf.h>
#include <anssettings.h>

/* These are only allowed in the server file, and take one "char*" argument */
struct key_func_pair server_daemon_setting[] = {
        {  "runas-uid",                 &set_runas_uid      },
        {  "runas-gid",                 &set_runas_gid      },
        {  "auth-port",                 &set_authport       },
        {  "pidfile",                   &set_pidfile        },
        {  "logfile",                   &set_logfile        },
        {  "user-dir",                  &set_user_dir       },
	{  "max-queries",               &set_max_queries    },
	{  "cache-age",                 &set_cache_age      },
        {0,0}
};

/* Sets the server settings to default values */
void
set_default_server_settings(void) {
	set_configfile(default_configfile);
	set_authport(default_authport);
        set_logfile(default_logfile);
	set_max_queries(default_max_queries);
	set_cache_age(default_cache_age);
        set_user_dir(default_user_dir);
        set_pidfile(default_pidfile);
        set_runas_uid(default_runas_uid);
        set_runas_gid(default_runas_gid);

	memset(&(server.is_modifed), FALSE, sizeof(server.is_modifed));
} /* -- end set_default_server_settings() -- */

/* The loop for processing a daemon setting.
Returns +1 on a successful call to a deamon
setting function, -1 on error (either in the
loop, or from the function called), or
-1 if nothing matched */
int
deamon_setting_loop(struct file_line* cfl) {
        uint found;
        uint filerr;
        struct key_func_pair* kfp;
        for( kfp = server_daemon_setting; kfp->keyword; ++kfp ){
                if( strcmp(cfl->key, kfp->keyword) != 0)
                        continue;

                if( (kfp->func)(cfl->value) )
                        return -1;
                else
                        return 1;
        }

        return 0;
} /* -- end deamon_setting_loop() -- */

/* Processes the config file; 0 on success, -1 on error */
int
process_config_file( void ) {
	if( is_nil(server.configfile) )
        	return 0;

        /* open config file with necessary
        privs, but bypass sfcheck_read() since we
        will be using the stricter (and mutually
        incompatible) sfcheck_readonly() */
	FILE* FP = sfopen_read(server.configfile, 0 );
        if( ! FP ) {
        	choke("Could not read %s", server.configfile);
                return -1;
        }

        /* If proc_suid we need to check that the invoking user
        cannot write the config file, furthermore if suid_root
        we need to check that the runas_uid cannot write the
        config file.  The second check may be removed in the
        future. */
        if(
            ( proc_suid && sfcheck_readonly(FP, proc_uid.ruid) ) ||
            ( proc_suid_root && sfcheck_readonly(FP, proc_uid.uuid) )
        ) {
                carp("config file is not safe to use");
                fclose(FP);
                return -1;
        }

        char* buf = fread_FILE(FP);
        if( fclose(FP) ) {
        	choke("Closing file triggered error");
                return -1;
        }

        if( ! buf ) {
        	carp("Could not read config file into memory");
                return -1;
        }

        struct file_line* cfl = boil_contents(buf);
        free(buf);

        if( ! cfl ) {
        	carp("Could not reduce contents of config file");
                return -1;
        }

        struct file_line* cfl_sv = cfl;
        int retval;
        for( ; cfl->lineno; ++cfl ) {
                retval = deamon_setting_loop(cfl);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server.configfile, cfl->lineno, cfl->text);
                        return -1;
                }

                retval = answer_setting_loop(cfl, server_answer_setting, &default_uuid_ans_conf);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server.configfile, cfl->lineno, cfl->text);
                        return -1;
                }
                retval = answer_setting_loop(cfl, user_answer_setting, &default_ans_conf);
                if( retval > 0 )
                	continue;
                if( retval < 0 ) {
                	carp("Error in %s (line %d) {%s}",
                			server.configfile, cfl->lineno, cfl->text);
                        return -1;
                }

                if( retval == 0 ) {
                	carp("Unknown keyword {%s} in %s (line %d) {%s}",
                			cfl->key, server.configfile, cfl->lineno, cfl->text);
                        return -1;
                }
	}

        free_file_lines(&cfl_sv);
        return 0;
} /* -- end process_config_file() -- */

/* Sets the uid to run as */
int
set_runas_uid(char* cp_uid) {
	uid_t uid;
        if( strtouidt(&(uid), cp_uid) ) {
        	carp("Unable to set runas_uid");
                return -1;
        }
        if( uid != proc_uid.uuid ) {
		proc_uid.uuid = uid;
		server.is_modifed.runas_uid = TRUE;
	}
        suspend_proc_privs();
        return 0;
} /* -- end set_runas_uid() -- */

/* Sets the gid to run as */
int
set_runas_gid(char* cp_gid) {
        gid_t gid;
        if( strtogidt(&(gid), cp_gid) ) {
        	carp("Unable to set the runas_gid");
                return -1;
        }

        if( gid != proc_uid.ugid ) {
		proc_uid.ugid = gid;
                server.is_modifed.runas_gid = TRUE;
	}
        suspend_proc_privs();
        return 0;
} /* -- end set_runas_gid() -- */

/* Turns on debugging info */
int
set_debug( void ) {
	state.debug = TRUE;
        return 0;
} /* -- end set_debug() -- */

/* Prevents running in the background */
int
set_nofork( void ) {
	state.nofork = TRUE;
	return 0;
} /* -- end set_nofork() -- */

/* Sets the age for the cache */
int
set_cache_age(char* age) {
	if( is_nil(age) ) {
		carp("age is null");
		return -1;
	}
	time_t expire;
	if( strtotime(&expire, age) ) {
		carp("Invalid time specification");
		return -1;
	}

	if( expire != server.cache_age ) {
		server.cache_age = expire;
		server.is_modifed.cache_age = TRUE;
	}

	return 0;
}/* -- end set_cache_age() -- */

/* Sets the user_dir.  Returns 0 on success, -1 on error. */
int
set_user_dir(char* dir) {
	char* userdir;

	if( is_nil(dir) )
        	userdir = NULL;
        else {
        	userdir = strdup(dir);
                if( ! userdir ) {
                	carp("Could not store user data dir");
                        return -1;
                }
        }

        if( ! strings_eq(userdir, server.userdir) ) {
		if_notnull_free(server.userdir);
		server.userdir = userdir;
        	server.is_modifed.userdir = TRUE;
	}
        return 0;
} /* -- end set_user_dir() -- */

/* Sets the logfile.  Returns 0 on success, -1 on error. */
int
set_logfile(char* file) {
	if( server.is_modifed.logfile && (!state.cmdline) )
        	return 0;

	char* logfile;

	if( is_nil(file) || strings_eq(file, "/dev/stderr") )
        	logfile = NULL;
        else {
        	logfile = strdup(file);
                if(!logfile) {
                	carp("Unable to store logfile name");
                        return -1;
                }
        }

        if( ! strings_eq(logfile, server.logfile) ) {
        	if_notnull_free(server.logfile);
		server.logfile = logfile;
		server.is_modifed.logfile = TRUE;
	}
        return 0;
} /* -- end set_logfile() -- */

/* Sets the pidfile. Returns 0 on success, -1 on error. */
int
set_pidfile(char* file) {
	if( server.is_modifed.pidfile && (!state.cmdline) )
        	return 0;

        char* pidfile;
        if( is_nil(file) || strings_eq(file, "/dev/null") )
        	pidfile = NULL;
        else {
        	pidfile = strdup(file);
                if( ! pidfile ) {
                	carp("Could not store pidfile name");
                        return -1;
                }
        }

	if( ! strings_eq(pidfile, server.pidfile) ) {
        	if_notnull_free(server.pidfile);
		server.pidfile = pidfile;
		server.is_modifed.pidfile = TRUE;
	}
        return 0;
} /* -- end set_pidfile() -- */

/* Sets the config file */
int
set_configfile(char* file) {
        char* configfile;
	if( is_nil(configfile) )
        	configfile = NULL;
        else {
        	configfile = strdup(file);
                if( ! configfile ) {
                	choke("Could not store config file name");
                        return -1;
                }
        }

        if( ! strings_eq(configfile, server.configfile) ) {
        	if_notnull_free(server.configfile);
		server.configfile = configfile;
		server.is_modifed.configfile = TRUE;
	}
        return 0;
} /* -- end set_configfile() -- */


/* -- end serversettings.c -- */
