/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   strtouidt.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <pwd.h>
#include <string.h>

#include "strtotype.h"

#include <macros.h>
#include <logutils.h>
#include <state.h>


typeof(uid_t) UID_T_MAX = -1;

/* Translates a string to a uid.  0 on success, -1 on error */
int
strtouidt(uid_t* uid, char* cp_uid) {
        if( is_nil(cp_uid) ) {
        	carp("uid is NULL");
                return -1;
        }
	/* First check to see if the string is a number */
        char* endptr;
        ulong ul_uid = strtoul(cp_uid, &endptr, 10);
        if( ! *endptr ) {
        	if( ul_uid > UID_T_MAX ) {
                	carp("runas_uid > max uid allowed for system");
                        return -1;
                }

                *uid = (uid_t) ul_uid;
                return 0;
        }

        /* Otherwise we try to find the entry in /etc/passwd */

        int retval = get_uid_r(uid, cp_uid);
        if( retval > 0 )
        	carp("Could not find username entry in /etc/passwd");
        if( retval )
        	return -1;

        return 0;
} /* -- end strtouidt() -- */

/* Mallocates sets a char* to a username for a given uid;
returns 0 on success (if the username could not be found
the char* is null), -1 on error. */
int
get_usernam_r(char** cp_uid, uid_t uid) {
        /*  Where exactly did the documentation on this go?
        getpwuid_r (
            uid_t uid, struct passwd *result_buf,
            char *buffer, size_t buflen, struct passwd **result
        );
        */
        size_t buflen = 512;
        char* buf = NULL;
        struct passwd pwd;
        struct passwd* pwd_p;
        *cp_uid = NULL;

        do {
                buflen *= 2;
                if_notnull_free(buf);
                buf = malloc(buflen);
                if( ! buf ) {
                	choke("Could not make room for pwd data");
                        return -1;
                }
        } while( getpwuid_r(uid, &pwd, buf, buflen, &pwd_p) == ERANGE );

        if( pwd_p != &pwd ) {
        	if_notnull_free(buf);
        	if(
                	(errno == ENOMEM)	||
                	(errno == EIO)		||
                	(errno == EINTR)	||
                	(errno == EMFILE)	||
                	(errno == ENFILE)
                ) {
                        choke("Unknown error occured.  Please " \
                                        "sacrifice two chickens and a goat to continue");
                        return -1;
                } else {
                	/* Entry does not exist */
			*cp_uid = NULL;
                	return 0;
                }
        }

        *cp_uid = strdup(pwd.pw_name);
        free(buf);
        if( ! *cp_uid ) {
        	choke("Could not copy username");
                return -1;
        }

	return 0;
} /* -- end get_usernam_r() -- */

/* Gets a uid given a usernam.  Threadsafe and all that.
Returns 0 on success, -1 on error, +1 if not found */
int
get_uid_r(uid_t* uid, char* cp_uid) {
        size_t buflen = 512;
        char* buf = NULL;
        struct passwd pwd;
        struct passwd* pwd_p;

        do {
                buflen *= 2;
                if_notnull_free(buf);
                buf = malloc(buflen);
                if( ! buf ) {
                	choke("Could not make room for pwd data");
                        return -1;
                }
        } while( getpwnam_r(cp_uid, &pwd, buf, buflen, &pwd_p) == ERANGE );

        if( pwd_p != &pwd ) {
        	if_notnull_free(buf);
        	if(
                	(errno == ENOMEM)	||
                	(errno == EIO)		||
                	(errno == EINTR)	||
                	(errno == EMFILE)	||
                	(errno == ENFILE)
                ) {
                        choke("Unknown error occured.  Please "
                                        "sacrifice two hogs and a hen to continue");
                        return -1;
                } else {
                	/* Entry does not exist */
                	return +1;
                }
        }

        *uid = pwd.pw_uid;

        free(buf);

	return 0;
} /* -- end get_uid_r() -- */


/* -- end strtouidt.c -- */
