/***************************************************************************
 *   Copyright (C) 2003 by beltorak                                        *
 *   beltorak@ananzi.co.za                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcre.h>
#include <netinet/in.h>

#include "reutils.h"

#include <macros.h>
#include <logutils.h>

/* These are the errors in pcre.h */
static char* PCRE_ERRSTR[] = {
        "No error",
        "Pattern did not match",
        "Invalid NULL parameter",
        "Unrecognized option",
        "Invalid quoted regex",
        "Corrupted quoted regex",
        "Out Of Memory for back references",
        "Substring does not exist",
        "Match limit reached",
        "Callout function failed",
        "Invalid UTF-8 sequence",
        "UTF-8 sequence split at startoffest",
        "Invalid PCRE_ERROR code"
};

static const char* re_stripspace = "^[\\t ]*(.*?)[\\t ]*$";
static const char* re_key_value = "^(.*?)\\s*:\\s*(.*)";
static const char* re_wspace_only = "^[\\t \\n]*$";
static const char* re_portpair = "^(\\d{1,5})\\s*,\\s*(\\d{1,5})$";
static const char* re_procnettcp = \
"(?x) \n\
# sl \n\
^\\s*\\d+:\\s* \n\
([[:xdigit:]]{8}):	# localhost \n\
([[:xdigit:]]{4})\\s+	# localport \n\
([[:xdigit:]]{8}):	# remotehost \n\
([[:xdigit:]]{4})\\s+	# remoteport \n\
([[:xdigit:]]{2})\\s+	# st \n\
# tx/rx-queue, tr & tm->when,  retrnsmt \n\
\\S+\\s+     \\S+\\s+      \\S+\\s+ \n\
([[:digit:]]+)		# socket owner \n\
.*?$";

struct quoted_regex* qr_stripspace;
struct quoted_regex* qr_key_value;
struct quoted_regex* qr_wspace_only;
struct quoted_regex* qr_portpair;
struct quoted_regex* qr_procnettcp;

/* Initializes the regular expression info; returns 0 on succes, -1 on error */
int
init_regexps( void ) {
	struct re_qr_pairs {
        	char*			name;
                char*			re;
                struct quoted_regex**	qr;
        };

        struct re_qr_pairs list[] = {
        	{ "re_stripspace", re_stripspace, &qr_stripspace },
                { "re_key_value", re_key_value, &qr_key_value },
                { "re_wspace_only", re_wspace_only, &qr_wspace_only},
                { "re_portpair", re_portpair, &qr_portpair},
                { "re_procnettcp", re_procnettcp, &qr_procnettcp},
                {0,0,0}
        };

        int k;
        for( k = 0; (list[k]).name; ++k ) {
        	*(list[k]).qr = qr(list[k].re);
                if( ! *(list[k].qr) ) {
                	carp("qr(%s) Failed", list->name);
                        return -1;
                }
        }

        return 0;
} /* -- end init_regexps() -- */

/* Removes a trailing CRs and NLs if present. */
void
chomp(char* string) {
	if( is_nil(string) ) {
                return;
        }
	size_t len = strlen(string);
        while( len-- && ((string[len] == '\n') || (string[len] == '\r')) )
        	string[len] = 0;

        return;
} /* -- end chomp() -- */

/* Simplest case for getting a compiled regex -- no options.
Returns ptr to a mallocated quoted_regex struct, or
NULL on error. IAW,
qr("string") == qre("string", 0, 0)
*/
/* __attribute__ ((__always_inline__)) */
struct quoted_regex*
qr( const char* re ) {
	return qre(re, 0, 0);
} /* -- end qr() -- */

/* Returns a ptr to a malocated quoted_regex struct,
or NULL on error. */
struct quoted_regex*
qre( const char* re, int opts, int extra_opts ) {
	struct quoted_regex* qr = new_qr();
        if( ! qr ) {
        	carp("Could not create quoted_regex");
                return NULL;
        }

        qr->qr_str = strdup(re);
	if( ! qr->qr_str ) {
		choke("strdup of re failed");
		return NULL;
	}

        qr->qr_opts = opts;
	qr->qr_extra_opts = extra_opts;
	const char* errptr;
	int erroffset;
	qr->qr_code = pcre_compile(re, opts, &errptr, &erroffset, NULL);
	if( ! qr->qr_code ) {
		carp("pcre_compile() of '%s' failed at char %d: %s",
				re, erroffset, errptr);
		free_quoted_regex(qr);
		return NULL;
	}

	qr->qr_extra = pcre_study(qr->qr_code, 0, &errptr);
	if(errptr) {
		carp("pcre_study() of %s failed: %s", re, errptr);
        	qr->qr_extra = NULL;
	}

	pcre_fullinfo(qr->qr_code, qr->qr_extra,
        		PCRE_INFO_CAPTURECOUNT, &(qr->qr_stringcount));

        ++(qr->qr_stringcount);
        return qr;
} /* -- end qre() -- */

/* frees a created quoted_regex struct */
void
free_quoted_regex( struct quoted_regex* qr ) {
	free(qr->qr_str);
	pcre_free(qr->qr_code);
	pcre_free(qr->qr_extra);
	free(qr);
	return;
} /* -- end free_quoted_regex() -- */

/* Creates a fresh new quoted_regex */
struct quoted_regex*
new_qr( void ) {
	struct quoted_regex* qr = malloc( sizeof( struct quoted_regex ) );
	if( ! qr ) {
		choke("malloc of 'qr' failed");
		return NULL;
	}

	memset(qr, 0, sizeof(struct quoted_regex));

        qr->qr_str = NULL;
        qr->qr_code = NULL;
        qr->qr_extra = NULL;

        return qr;
} /* end new_qr() -- */

/* These functions take a malloc()'d string ptr as an argument,
and upon success replace the string with the malloc()'d result.
Returns 0 on success, -1 on error */

/* Strips comments (but not any space before the comment).
If the line is a comment line, *line == NULL.  */
int
strip_comment( char** line ) {
	if( ! (*line && **line) )
		return 0;
	if( **line == '#' ) {
		free(*line);
		*line = NULL;
		return 0;
	}

	char* tmp = strchr(*line, '#');
	if( tmp ) {
		*tmp = 0;
	}

	return 0;
} /* -- end strip_comment() -- */

/* Checks to see if a string is entirely whitespace.
** Returns TRUE if true, false if non-whitespace.
A NULL or NIL string is true. */
int
is_wspace_only(char* str) {
	if( is_nil(str) )
        	return TRUE;
	struct quoted_regex* qr = qr_wspace_only;
        int ovector[qr->qr_stringcount * 3];
	int retval = pcre_exec(qr->qr_code, qr->qr_extra, str, strlen(str),
        		0, 0, ovector, qr->qr_stringcount * 3);
        if( retval == PCRE_ERROR_NOMATCH )
        	return FALSE;
        return TRUE;
} /* -- end is_wspace_only() -- */

/* Strips leading and trailing whitespace.  If the line is
either blank or entirely whitespace, *line == NULL.  This
function also strips the trailing newlines. */
int
strip_whitespace( char** line ) {
	if( is_nil(*line) ) {
        	if_notnull_free(*line);
		return -1;
        }

	struct quoted_regex* qr = qr_stripspace;
        int ovector[qr->qr_stringcount * 3];
	size_t len = strlen( *line );
	int retval = pcre_exec(qr->qr_code, qr->qr_extra, *line, len,
        		0, 0, ovector, qr->qr_stringcount*3 );
	if( retval == PCRE_ERROR_NOMATCH ) {
		null_free(*line);
                return 0;
	} else if( retval < 0 ) {
		choke("pcre_exec() Failed -- %s", pcre_strerror_np(retval));
		null_free(*line);
		return -1;
	} else if( retval == 0 ) {
		carp("Ran out of room to capture subpattern");
                return -1;
        } else if(retval != qr->qr_stringcount) {
		carp("I suck at pcre! %d matching string", retval);
		null_free(*line);
		return -1;
	}

	char* tmp;
	retval = pcre_get_substring(*line, ovector, retval,
        		1, (const char**)&tmp);
	if( retval < 0 ) {
		choke("pcre_get_substring() Failed : %s",
        			pcre_strerror_np(retval));
                tmp = NULL;
	}
	free(*line);
	*line = tmp;
        chomp(*line);
	return 0;
} /* -- end strip_whitespace() -- */

/* Removes encapsulating quotes from a string */
void
dequote(char** string) {
	if( ! ( *string && **string) )
        	return;

        char* str = *string;
        int len = strlen(str);

        if(
            /* If balanced simple quotes, remove the quotes */
            ((*str == '\'') && (str[len-1] == '\'')) ||
            ((*str == '"' ) && (str[len-1] == '"' ))
        ) {
        	str[len-1] = 0;
        	memmove(str, str + 1, len-1);
        } else if(
            /* If balanced escaped quotes, remove the escapes */
            (
                (strncmp(str, "\\\"", 2) == 0) &&
                (strncmp(&(str[len-2]), "\\\"", 2) == 0)
            ) || (
                (strncmp(str, "\\'",  2) == 0) &&
                (strncmp(&(str[len-2]), "\\'",  2) == 0)
            )
        ) {
        	str[len-2] = str[len-1];
                str[len-1] = 0;
                memmove(str, str + 1, len-1);
        }

        return;
} /* -- end dequote() -- */

/* Splits a string to its "key: value" equivalent, dropping
leading or trailing whitespace.  Also removes encapsulating
quotes from value.  0 on success, -1 on error.
NOTE: key and value must be null to prevent memory leaks. */
int
split_key_value(char** key, char** value, char* string) {
	if( !(string || *string) )
                return -1;

        struct quoted_regex* qr = qr_key_value;
        int ovector[qr->qr_stringcount * 3];
        int retval = pcre_exec(qr->qr_code, qr->qr_extra, string, strlen(string),
        		0, 0, ovector, qr->qr_stringcount * 3);

        if( retval == PCRE_ERROR_NOMATCH ) {
        	carp("Invalid key:value input -- %s", string);
        	return -1;
        } else if( retval < 0 ) {
        	carp("Failed to match key and value: %s",
        			pcre_strerror_np(retval));
		return -1;
	} else if( retval == 0 ) {
		carp("Ran out of room to capture subpattern");
                return -1;
        } else if(retval != qr->qr_stringcount) {
		carp("I suck at pcre! %d matching strings", retval);
		return -1;
        }

        /* Get the key and check that it is not NULL */
        retval = pcre_get_substring(string, ovector, qr->qr_stringcount,
        		1, (const char**) key);
        if( retval < 0 ) {
        	carp("Could not retrieve key: %s", pcre_strerror_np(retval));
        	return -1;
        }
	if( !( *key && **key ) ) {
        	carp("Key is NULL!");
                return -1;
        }

	/* Get the value and dequote */
        retval = pcre_get_substring(string, ovector, qr->qr_stringcount,
        		2, (const char**) value);
        if( retval < 0 ) {
        	carp("Could not retrieve value: %s", pcre_strerror_np(retval));
        	null_free(*key);
        	return -1;
        }
        if( ! is_nil(*value) )
        	dequote(value);

        if( is_nil(*value) )
                if_notnull_free(*value);

        return 0;
} /* -- end split_key_value() -- */

/* Extracts a port pair into the uint pointers provided. */
int
extract_port_pair(uint* local, uint* remote, char* str) {
	struct quoted_regex* qr = qr_portpair;
        int retval;
        uint ovector[qr->qr_stringcount * 3];

        retval = pcre_exec(qr->qr_code, qr->qr_extra, str, strlen(str),
        		0, 0, ovector, qr->qr_stringcount * 3);
        if( retval == PCRE_ERROR_NOMATCH ) {
        	dreport("Invalid port pair passed");
        	return -1;
        } else if( retval < 0 ) {
        	dreport("Failed to match port pair pattern: %s",
        			pcre_strerror_np(retval));
		return -1;
	} else if( retval == 0 ) {
		carp("Ran out of room to capture subpatterns");
                return -1;
        } else if(retval != qr->qr_stringcount) {
		carp("I suck at pcre! %d matching strings", retval);
		return -1;
        }

        char* tmp;
        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		1, (const char**) &tmp);
        if( retval < 0 ) {
        	carp("Could not retrieve local port number -- %s",
        			pcre_strerror_np(retval));
        	return -1;
        }

        if( strtoport(local, tmp) )
        	return -1;
	if_notnull_free(tmp);

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		2, (const char**) &tmp);
        if( retval < 0 ) {
        	carp("Could not retrieve remote port number -- %s",
        			pcre_strerror_np(retval));
                return -1;
        }

        if( strtoport(remote, tmp) )
		return -1;
	if_notnull_free(tmp);


        return 0;
} /* -- end extract_port_pair() -- */

/* Examines a "/proc/net/tcp" line, returns the following
info in the supplied procnettcp_info:
    localhost, remotehost, localport, remoteport, socket owner
All info is in host order.
Returns 0 on success, -1 on error. */
int
get_sockinfo( struct procnettcp_info* pntcp_info, char* str) {
	struct quoted_regex* qr = qr_procnettcp;
        int ovector[qr->qr_stringcount * 3];
        int retval;
        retval = pcre_exec(qr->qr_code, qr->qr_extra, str, strlen(str),
        		0, 0, ovector, qr->qr_stringcount * 3);
        if( retval == PCRE_ERROR_NOMATCH ) {
        	carp("Invalid \"/proc/net/tcp\" line");
        	return -1;
        } else if( retval < 0 ) {
        	carp("Failed to match \"/proc/net/tcp\" pattern: %s",
        			pcre_strerror_np(retval));
		return -1;
	} else if( retval == 0 ) {
		carp("Ran out of room to capture subpatterns");
                return -1;
        } else if(retval != qr->qr_stringcount) {
		carp("I suck at pcre! %d matching strings", retval);
		return -1;
        }

        char* tmp_cp;
        char* endp;

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		1, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve localhost: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }
	/* Why is the host in network order, but the ports are in host order?? */
        pntcp_info->lhost = ntohl(strtoul(tmp_cp, &endp, 16));
	if_notnull_free(tmp_cp);

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		2, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve localport: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }
        pntcp_info->lport = (uint) strtoul(tmp_cp, &endp, 16);
	if_notnull_free(tmp_cp);

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		3, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve remotehost: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }
	/* Why is the host in network order, but the ports are in host order?? */
        pntcp_info->rhost = ntohl(strtoul(tmp_cp, &endp, 16));
	if_notnull_free(tmp_cp);

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		4, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve remoteport: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }
        pntcp_info->rport = (uint) strtoul(tmp_cp, &endp, 16);
	if_notnull_free(tmp_cp);

        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		5, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve state: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }
        pntcp_info->state = (uint) strtoul(tmp_cp, &endp, 16);
	if_notnull_free(tmp_cp);


        retval = pcre_get_substring(str, ovector, qr->qr_stringcount,
        		6, (const char**) &tmp_cp);
        if( retval < 0 ) {
        	carp("Could not retrieve owner: %s",
        			pcre_strerror_np(retval));
        	return -1;
        }

        pntcp_info->owner = (uid_t) strtoul(tmp_cp, &endp, 10);
	if_notnull_free(tmp_cp);

        return 0;
} /* -- end get_sockinfo() -- */

/* Returns a pointer to a pcre error string */
char*
pcre_strerror_np(int errnum) {
	if( errnum > 0 )
        	return *PCRE_ERRSTR;
        if( errnum < -11 )
        	return *PCRE_ERRSTR + 12;

        return *(PCRE_ERRSTR -errnum);
} /* -- end pcre_strerror_np() -- */


/* -- end regexp.c -- */
