/***************************************************************************
 *   Copyright (C) 2004 by Trevor "beltorak" Torrez                        *
 *   beltorak@phreaker.net                                                 *
 *   init.c part of cidentd version 0.2                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* This file will handle everything up to dropping root privs. */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <dirent.h>

#include "init.h"

#include <defaults.h>
#include <serverconf.h>
#include <anssettings.h>
#include <serversettings.h>
#include <networking.h>
#include <logutils.h>
#include <state.h>
#include <procprivs.h>
#include <tbtopt.h>
#include <meta.h>
#include <sfopen.h>

sigset_t sigs;

struct tbt_option cmdline_options[] = {
	{	.type = 't',
        	.help_str = "Server Options"
        },
        {	&set_configfile,
        	'f', "configfile",	'y', "file",
                "Use an alternate config file"
        },
        {	&set_logfile,
        	'L', "logfile",		'y', "file",
                "Use an alternate logfile."
        },
        {	&set_authport,
        	'p', "port",		'y', "portspec",
                "Set the port to listen on.  Any valid port number"
        },
	{	.type = 'd',
		.help_str = "or name in '/etc/services' is allowed."
	},
        {	&set_pidfile,
        	'P', "pidfile",		'y', "file",
                "Write the PID in this file;"
        },
        {	.type = 'd',
        	.help_str = "Use '/dev/null' to not write a pidfile"
        },
        {	&set_debug,
        	'd', "debug",		'n', NULL,
                "Reports file and line info in error reports"
        },
        {	.type = 'd',
        	.help_str = "This may be toggled at runtime using SIGUSR2"
        },
        {	.type = 'd',
        	.help_str = "Note: This is *very* noisy atm"
        },
	{	&set_nofork,
		0, "forground",		'n', NULL,
		"Do not run in the background"
	},
        {	.type = 't',
        	.help_str = "About Options"
        },
        {	&cat_help,
        	'h', "help",		'n', NULL,
                "Prints this help and exits"
        },
        {	&cat_version,
        	'V', "version",		'n', NULL,
                "Prints version and exits"
        },
        {	&set_show_settings,
        	0,   "show-settings",	'n', NULL,
                "Shows the settings after parsing the"
        },
        {	.type = 'd',
        	.help_str = "command line and config file."
        },
        {	&display_defaults,
        	0,   "display-defaults", 'n', NULL,
                .help_str = "Displays the compiled in settings."
        },
        { 0, 0, 0, 0, 0, 0}
};

/* Begins the init process.  Calls all relevent
initialization routines.  Returns 0 on success,
-1 on error, or a +1 when the program must quit
but not due to an error (help requested). */
int
cidentd_init(int argc, char** argv) {
	store_proc_privs();
        suspend_proc_privs();
	state.initting = TRUE;

        if( init_regexps() )
        	return -1;

        if( set_defaults() )
        	return -1;

        int retval = process_cmdline(argc, argv);
        if( retval < 0 )
        	return -1;
        else if( retval > 0 )
        	return 1;

        if( process_config_file() )
        	return -1;

        /* if required display the current settings */
        if( state.show_settings ) {
        	show_settings();
	        return 1;
        }

        /* Set cleanup routine */
        atexit(&shutdown_cidentd);

        if( start_logging() )
        	return -1;

        if( init_signals() )
        	return -1;

        if( install_sig_handlers() )
        	return -1;

        if( open_pidfile() )
        	return -1;

        if( start_networking() )
        	return -1;

        if( drop_proc_privs() )
        	return -1;

        state.initting = FALSE;

	return 0;
} /* -- end cidentd_init() -- */

/* Calls every default setting function.
Returns o on success, -1 on error. */
int
set_defaults(void) {
	set_default_server_settings();
        set_default_answer_settings();
	return 0;
} /* -- end set_defaults() -- */

/* processes the command line.  Returns 0 on success,
-1 on error, +1 on success to exit the program without
error */
int
process_cmdline(int argc, char** argv) {
	state.cmdline = TRUE;
	int retval = process_argv(cmdline_options, argc, argv);
        if( retval < 0 )
        	return -1;
        else if( retval > 0 ) {
        	report("Extra arguments at end of command");
                return -1;
        } else if( TBTOPT_EXIT_SUCCESS )
        	return 1;

        state.cmdline = FALSE;
        return 0;
} /* -- end process_cmdline() -- */

/* Sets the show_settings option */
int
set_show_settings( void ) {
	state.show_settings = TRUE;
        return 0;
} /* -- end set_show_settings() -- */

/* Calls every applicable show setting function */
void
show_settings( void ) {
	if( state.show_settings ) {
        	println("Command line and config file settings.");
        	print_default_marker_if(TRUE);
                println("marks a compiled in default setting.");
        }
        show_server_settings();
        show_answer_settings();
} /* -- end show_settings() -- */

/* Displays the default settings. */
int
display_defaults( void ) {
	println("Default compiled in settings:");
	set_defaults();
	state.show_settings = FALSE;
	show_settings();
	TBTOPT_EXIT_SUCCESS = TRUE;
	return 0;
} /* -- end display_defaults() -- */

/* Reports a shutdown to the logfile */
void
shutdown_cidentd( void ) {
        if( server.pidfile && unlink( server.pidfile ) )
                choke("Could not remove pidfile");

        if( close(server.auth_socket) )
        	choke("Could not close auth socket %d", server.auth_socket);

	report("shutting down\n-");
}

/* Starts logging -- everything going to stderr now goes to the logfile */
int
start_logging( void ) {
	if( ! server.logfile ) {
        	report("Logging to stderr");
        } else {
        	FILE* tmp = sfopen_append(server.logfile);
                if( ! tmp ) {
                	carp("Could not open server_logfile");
                        report("logging to stderr");
                        set_logfile(NULL);
                } else {
                	fclose(stderr);
                        stderr = tmp;
                }
        }
        return 0;
} /* -- end start_logging() -- */

/* Writes the pidfile.  Since this is a type of log,
it is defined here.  */
int
write_pidfile( void ) {
        if( ! server.pidFP )
        	return 0;
        fprintf(server.pidFP, "%lu\n", getpid());
        if( fclose(server.pidFP) ) {
        	choke("Could not write pid to pidfile");
                server.pidfile = NULL;
                return 0;
        }
        return 0;

} /* -- end write_pidfile() -- */

/* Opens the pidfile.
Returns 0.
It is not an error if the pidfile cannot be written,
however, it is an error if the pidfile referes to an active
process. */
int
open_pidfile( void ) {
	/* Continue only the pidfile is to be written */
	if( !server.pidfile )
        	return 0;

        int retval;
        server.pidFP = sfopen_write(server.pidfile);
        if( (! server.pidFP) && (errno == EEXIST) ) {
        	if( check_pid() ) {
                	report("%s appears to be running.  "
                			"If this is not so, delete %s and try again",
                			PACKAGE, server.pidfile);
                	server.pidfile = NULL;
                        return -1;
                }
		/* Here is a potential problem */
                proc_to_prime_privs();
                retval = unlink(server.pidfile);
                suspend_proc_privs();
                if( retval ) {
                	choke("Could not remove stale pidfile");
                        return 0;
                }
                server.pidFP = sfopen_write(server.pidfile);
        }

        if( ! server.pidFP ) {
        	report("Skipping the pidfile");
                server.pidfile = NULL;
                return 0;
        }

        uint pf_mode = S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH;
        proc_to_prime_privs();
        retval = fchmod(fileno(server.pidFP), pf_mode);
	suspend_proc_privs();

        if( retval ) {
        	choke("could not chmod pidfile");
        }

        return 0;
} /* -- end open_pidfile() -- */

/* Checks the validity of a pidfile. Returns TRUE if a process
with that pid exists, FALSE otherwise.  On error, the
process is assumed to be valid. */
uint
check_pid( void ) {
        FILE* FP = sfopen_read(server.pidfile, 0);
        if( ! FP ) {
                choke("Could not read pidfile");
                return TRUE;
        }
        /* Twenty digits ought to be enough */
        char buf[21];
        memset(buf, 0, 21);
        fread(buf, 20, 1, FP);
        if( ferror(FP) ) {
                choke("Read of pidfile failed");
                fclose(FP);
                return TRUE;
        }
        fclose(FP);
        uint pid = strtoul(buf, NULL, 10);

        /* Check for a process */
        char* procfile = strprintf("/proc/%d", pid);
        if( ! procfile ) {
                choke("Could not make room for process dirname");
                return TRUE;
        }
        DIR* procdir = opendir(procfile);
        free(procfile);
        if( (! procdir) && (errno == ENOENT) ) {
        	closedir(procdir);
		return FALSE;
        }

        closedir(procdir);
        return TRUE;
} /* -- end check_pid -- */

/* Initializes the handled signals */
int
init_signals( void ) {
        sigfillset(&sigs);

        /* Ignoring these signals can result in unpredictable
        behaviour */
        sigdelset(&sigs, SIGSEGV);
        sigdelset(&sigs, SIGFPE);
        sigdelset(&sigs, SIGILL);

        /* These cannot be caught, blocked, or ignored on Linux,
        so they are here merely for the sake of completeness  */
        sigdelset(&sigs, SIGSTOP);
        sigdelset(&sigs, SIGKILL);

        if( sigprocmask(SIG_BLOCK, &sigs, NULL) ) {
        	choke("Could not block exterraniuos signals");
                return -1;
        }

        return 0;
} /* -- end init_signals() -- */

/* Sets the signal handling routines */
int
install_sig_handlers( void ) {
        struct sigaction sact;
        sact.sa_mask = sigs;
        sact.sa_flags = 0;

        struct {
        	char* name;
                int   sig;
                void* handler;
        } list[] = {
        	{"SIGTERM", SIGTERM, &sa_cidentd_exit},
                {"SIGINT", SIGINT, &sa_cidentd_exit},
                {"SIGUSR2", SIGUSR2, &sa_toggle_debug},
                {0,0,0}
        };

        int k = 0;
        sigset_t noblock;
        sigemptyset(&noblock);

        while( list[k].name ) {
                sigaddset(&noblock, list[k].sig);
                sact.sa_handler = list[k].handler;
                if( sigaction(list[k].sig, &sact, NULL) ) {
                        choke("Could not install %s handler", list[k].name);
                        return -1;
                }
                ++k;
        }

        if( sigprocmask(SIG_UNBLOCK, &noblock, NULL) ) {
                choke("Could not unblock signals");
                return -1;
        }

        return 0;
} /* -- end install_sig_handlers() -- */

/* The TERM and INT sig handler */
void
sa_cidentd_exit(int sig) {
	report("SIG%s Caught", ((sig == SIGINT) ? "INT" : "TERM") );
        exit(EXIT_SUCCESS);
} /* -- end sa_cidentd_exit() -- */

/* The USR2 signal handler */
void
sa_toggle_debug(int sig) {
	state.debug ^= 1;
        report("Debugging turned %s", (state.debug ? "on" : "off") );
} /* -- end sa_toggle_debug() -- */


/* -- end init.c -- */
