/***************************************************************************
 *   Copyright (C) 2003 by beltorak                                        *
 *   beltorak@ananzi.co.za                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef reutils_H
#define reutils_H

#include <pcre.h>

/* Holds everything needed for a regex */
struct quoted_regex {
	char*		qr_str;
	int		qr_opts;
	int		qr_extra_opts;
	pcre*		qr_code;
	pcre_extra*	qr_extra;
        /* This is the number of substrings in a regex */
        int		qr_stringcount;
};

/* This is all we care about in /proc/net/tcp */
struct procnettcp_info {
	uint		state;
	ulong		lhost;
        ulong		rhost;
        uint		lport;
        uint		rport;
        uid_t		owner;
};

/* ===== function reference ===== */

/* Removes a trailing newline if present */
void
chomp(char* string);

/* Simplest case for getting a compiled regex -- no options.
Returns ptr to a mallocated quoted_regex struct, or
NULL on error. IAW,
qr("string") == qre("string", 0, 0)
*/
struct quoted_regex*
qr( const char* re ) __attribute__ ((__always_inline__));

/* Returns a ptr to a malocated quoted_regex struct,
or NULL on error. */
struct quoted_regex*
qre( const char* re, int opts, int extra_opts );

/* frees a created quoted_regex struct */
void
free_quoted_regex( struct quoted_regex* pei );

/* Creates a fresh new quoted_regex */
struct quoted_regex*
new_qr( void );

/* Initializes the regular expression info; returns 0 on succes, -1 on error */
int
init_regexps( void );

/* These functions take a malloc()'d string ptr as an argument,
and upon success replace the string with the malloc()'d result.
Returns 0 on success, -1 on error */

/* Strips comments (but not any space before the comment).
If the line is a comment line, *line == NULL.  */
int
strip_comment( char** line );

/* Checks to see if a string is entirely whitespace.
** Returns TRUE if true, false if non-whitespace.
A NULL or NIL string is true. */
int
is_wspace_only(char* str);

/* Strips leading and trailing whitespace.  If the line is
either blank or entirely whitespace, *line == NULL. */
int
strip_whitespace( char** line );

/* Removes encapsulating quotes from a string */
void
dequote(char** string);

/* Splits a string to its "key: value" equivalent, dropping
leading or trailing whitespace.  Also removes encapsulating
quotes from value.  0 on success, -1 on error.
NOTE: key and value must be null to prevent memory leaks. */
int
split_key_value(char** key, char** value, char* string);

/* Extracts a port pair into the uint pointers provided. */
int
extract_port_pair(uint* local, uint* remote, char* str);

/* Examines a "/proc/net/tcp" line, returns the following
info in the supplied ptrs of procnettcp_info:
    localhost, remotehost, localport, remoteport, socket owner
All info is in host order.
Returns 0 on success, -1 on error. */
int
get_sockinfo( struct procnettcp_info* pntcp_info, char* str);

/* Returns a pointer to a pcre error string */
char*
pcre_strerror_np(int errnum);


#endif /* reutils_H */
